package com.cleanroommc.groovyscript.api.documentation.annotations;

import com.cleanroommc.groovyscript.sandbox.LoadStage;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Marks the given class as a Groovyscript registry class.<br>
 * Many elements are autogenerated by default.
 * <ul>
 *     <li>
 *     {@link #title()} is a localization key that is autogenerated with a base of
 *     <code>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.title</code>
 *     </li>
 *     <li>
 *     {@link #description()} is a localization key that is autogenerated with a base of
 *     <code>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.description</code>
 *     </li>
 *     <li>{@link #linkGenerator()} is the name, typically a modid, of a {@link com.cleanroommc.groovyscript.documentation.linkgenerator.ILinkGenerator ILinkGenerator}
 *     contained in {@link com.cleanroommc.groovyscript.documentation.linkgenerator.LinkGeneratorHooks LinkGeneratorHooks}.
 *     This will generate a link from the wiki to the source code of the specific file the class {@link RegistryDescription} annotates is in.</li>
 *     <li>
 *     {@link #override()} is the way to set the {@link MethodDescription} and {@link RecipeBuilderDescription} annotations for methods that are
 *     declared in the parent class but not in the annotated class, and need a custom annotation (often for examples).
 *     </li>
 *     <li>{@link #location()} is the name of a {@link com.cleanroommc.groovyscript.sandbox.LoadStage} and controls where all methods
 *     inside the {@link RegistryDescription} are generated.</li>
 *     <li>{@link #reloadability()} determines the status of the compat, defaulting to fully reloadable ({@link Reloadability#FULLY}).</li>
 *     <li>{@link #isFullyDocumented()} informs whether the {@link RegistryDescription} documents all relevant functionality.</li>
 *     <li>{@link #admonition()} is an array of admonition blocks to be generated at the top of the wiki page for additional information,
 *     commonly of the type {@link Admonition.Type#DANGER} to indicate important information.</li>
 *     <li>{@link #category()} determines what words are used to refer to the registry and what it contains.</li>
 *     <li>{@link #priority()} is an integer that influences the sorting of the {@link RegistryDescription} relative to other {@link RegistryDescription}s.</li>
 * </ul>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
public @interface RegistryDescription {

    /**
     * The localization key for the name of the compat, will default to generating
     * <code>
     * groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.title
     * </code>
     *
     * @return localization key for the compat name
     */
    String title() default "";

    /**
     * The localization key for the description of the compat, will default to generating
     * <code>
     * groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.description
     * </code>
     *
     * @return localization key for the compat description
     */
    String description() default "";

    /**
     * Controls what the parser is to transform the java class file into a link so the source code may be accessed from the documentation.
     * Addons should create their own instance of the {@link com.cleanroommc.groovyscript.documentation.linkgenerator.ILinkGenerator ILinkGenerator} class,
     * preferably with the name of the generator being set to the modid.
     * If it is not set or a corresponding LinkGenerator does not exist, the {@link com.cleanroommc.groovyscript.documentation.linkgenerator.ILinkGenerator ILinkGenerator}
     * converts to a link towards CleanroomMC's Groovyscript repo set to view the repo at a specific version tag.
     *
     * @return a string corresponding to a {@link com.cleanroommc.groovyscript.documentation.linkgenerator.ILinkGenerator ILinkGenerator}, which controls how the source code link is generated.
     * @see com.cleanroommc.groovyscript.documentation.linkgenerator.ILinkGenerator ILinkGenerator
     */
    String linkGenerator() default "";

    /**
     * @return the stage all the compat with the registry uses. Defaults to {@link LoadStage#POST_INIT}
     * @see com.cleanroommc.groovyscript.sandbox.LoadStage
     */
    LoadStage location() default LoadStage.POST_INIT;

    /**
     * An override to {@link MethodDescription} or {@link RecipeBuilderDescription} declarations,
     * using the wrapper class {@link MethodOverride}.
     * <p>
     * Methods can be referred to by name if there is only one method with the given name
     * in the class, otherwise must be referred to by the method name + method descriptor.
     * <p>
     * While this can be used for any method, it is preferred to only use it for
     * methods that require it - namely, methods that are declared in a parent class
     * and not overridden in the focused class. Primarily used for examples of those methods.
     *
     * @return the MethodOverride annotation containing any number of {@link MethodDescription} or {@link RecipeBuilderDescription} annotations
     * @see MethodOverride
     * @see MethodDescription
     * @see RecipeBuilderDescription
     */
    MethodOverride override() default @MethodOverride;

    /**
     * Is this compat reloadable in-game, and does reloading have any potential problems? Defaults to fully reloadable.
     *
     * @return compat reloadability status
     * @see Reloadability
     */
    Reloadability[] reloadability() default Reloadability.FULLY;

    /**
     * @return if the registry is fully documented. {@code false} should be accompanied by a TO{@literal }DO or similar
     */
    boolean isFullyDocumented() default true;

    /**
     * A list of admonitions to be displayed after the description, where the admonitions are formatted based on
     * <a href="https://squidfunk.github.io/mkdocs-material/reference/admonitions">Admonitions for Material for MkDocs</a>
     *
     * @return specialized array of documentation information to create admonitions
     * @see Admonition
     */
    Admonition[] admonition() default {};

    /**
     * Controls the localization keys used refer to adding, removing, or querying the registry.
     * Primarily used to control if the registry contains specifically "recipes" or more generic "entries".
     * <p>
     * Using {@link Category#CUSTOM} it will instead try to read specific lang keys, checking these places for overrides:
     * <code>
     * <br>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.operation.query
     * <br>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.operation.removing
     * <br>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.operation.values
     * <br>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.operation.adding
     * </code>
     *
     * @return the name of the objects within the registry. Defaults to {@code Category.RECIPES}
     * @see Category
     */
    Category category() default Category.RECIPES;

    /**
     * Priority of the registry, relative to other registries of the same mod.
     * Priorities sort entries such that lowest is first, then by the natural order of {@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() VirtualizedRegistry#getName()}
     *
     * @return the registry priority
     */
    int priority() default 1000;

    /**
     * How reloadable the compat is: <br>
     * - {@link Reloadability#FULLY}: can be reloaded in-game, with each reload reproducing the same game-state as would be produced by restarting minecraft.<br>
     * - {@link Reloadability#FLAWED}: can be reloaded in-game, but reloads may not recreate the same game-state as would be produced by restarting minecraft.<br>
     * - {@link Reloadability#DISABLED}: cannot be reloaded in-game, and requires a total minecraft restart to apply changes.<br>
     */
    enum Reloadability {

        /**
         * Can be reloaded in-game, with each reload reproducing the same game-state as would be produced by restarting minecraft
         */
        FULLY,
        /**
         * Can be reloaded in-game, but reloads may not recreate the same game-state as would be produced by restarting minecraft
         */
        FLAWED,
        /**
         * Cannot be reloaded in-game, and requires a total minecraft restart to apply changes
         */
        DISABLED;

        public boolean isReloadable() {
            return this != DISABLED;
        }

        public boolean hasFlaws() {
            return this == FLAWED;
        }
    }

    /**
     * Controls the name of what the registry contains.
     * Currently, either specifically "recipes", generically "entries", or entirely custom.
     */
    enum Category {

        RECIPES("recipes"),
        ENTRIES("entries"),
        CUSTOM;

        public final String query;
        public final String adding;
        public final String removing;
        public final String values;

        Category() {
            this.query = null;
            this.adding = null;
            this.removing = null;
            this.values = null;
        }

        Category(String category) {
            this.query = "groovyscript.wiki.query_" + category;
            this.adding = "groovyscript.wiki.adding_" + category;
            this.removing = "groovyscript.wiki.removing_" + category;
            this.values = "groovyscript.wiki.editing_values";
        }
    }
}
