package com.cleanroommc.groovyscript.compat.mods.bloodmagic;

import WayofTime.bloodmagic.altar.AltarTier;
import WayofTime.bloodmagic.api.impl.BloodMagicAPI;
import WayofTime.bloodmagic.api.impl.recipe.RecipeBloodAltar;
import com.cleanroommc.groovyscript.api.GroovyLog;
import com.cleanroommc.groovyscript.api.IIngredient;
import com.cleanroommc.groovyscript.api.documentation.annotations.*;
import com.cleanroommc.groovyscript.compat.mods.ModSupport;
import com.cleanroommc.groovyscript.core.mixin.bloodmagic.BloodMagicRecipeRegistrarAccessor;
import com.cleanroommc.groovyscript.helper.ingredient.IngredientHelper;
import com.cleanroommc.groovyscript.helper.recipe.AbstractRecipeBuilder;
import com.cleanroommc.groovyscript.registry.StandardListRegistry;
import net.minecraft.item.ItemStack;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

@RegistryDescription(
        admonition = {
                @Admonition("groovyscript.wiki.bloodmagic.blood_altar.note0"),
                @Admonition(
                        value = "groovyscript.wiki.bloodmagic.blood_altar.note1",
                        type = Admonition.Type.DANGER,
                        format = Admonition.Format.STANDARD,
                        hasTitle = true)
        }
)
public class BloodAltar extends StandardListRegistry<RecipeBloodAltar> {

    @RecipeBuilderDescription(example = {
            @Example(".input(item('minecraft:clay')).output(item('minecraft:gold_ingot')).minimumTier(0).drainRate(5).syphon(10).consumeRate(5)"),
            @Example(".input(item('minecraft:gold_ingot')).output(item('minecraft:diamond')).tier(3).drainRate(100).syphon(50000).consumeRate(500)")
    })
    public RecipeBuilder recipeBuilder() {
        return new RecipeBuilder();
    }

    @Override
    public Collection<RecipeBloodAltar> getRecipes() {
        return ((BloodMagicRecipeRegistrarAccessor) BloodMagicAPI.INSTANCE.getRecipeRegistrar()).getAltarRecipes();
    }

    @MethodDescription(type = MethodDescription.Type.ADDITION)
    public RecipeBloodAltar add(IIngredient input, ItemStack output, int minimumTier, int syphon, int consumeRate, int drainRate) {
        return recipeBuilder()
                .minimumTier(minimumTier)
                .consumeRate(consumeRate)
                .drainRate(drainRate)
                .input(input)
                .output(output)
                .register();
    }

    @MethodDescription(example = @Example("item('minecraft:ender_pearl')"))
    public boolean removeByInput(IIngredient input) {
        if (getRecipes().removeIf(recipe -> {
            boolean removeRecipe = recipe.getInput().test(IngredientHelper.toItemStack(input));
            if (removeRecipe) {
                addBackup(recipe);
            }
            return removeRecipe;
        })) {
            return true;
        }

        GroovyLog.msg("Error removing Blood Magic Blood Altar recipe")
                .add("could not find recipe with input {}", input)
                .error()
                .post();
        return false;
    }

    @MethodDescription(example = @Example("item('bloodmagic:slate:4')"))
    public boolean removeByOutput(ItemStack output) {
        if (getRecipes().removeIf(recipe -> {
            boolean matches = ItemStack.areItemStacksEqual(recipe.getOutput(), output);
            if (matches) {
                addBackup(recipe);
            }
            return matches;
        })) {
            return true;
        }

        GroovyLog.msg("Error removing Blood Magic Blood Altar recipe")
                .add("could not find recipe with output {}", output)
                .error()
                .post();
        return false;
    }

    @Property(property = "input", comp = @Comp(eq = 1))
    @Property(property = "output", comp = @Comp(eq = 1))
    public static class RecipeBuilder extends AbstractRecipeBuilder<RecipeBloodAltar> {

        @Property(comp = @Comp(gte = 0, unique = "groovyscript.wiki.bloodmagic.max_tier.required"))
        private int minimumTier;
        @Property(comp = @Comp(gte = 0))
        private int syphon;
        @Property(comp = @Comp(gte = 0))
        private int consumeRate;
        @Property(comp = @Comp(gte = 0))
        private int drainRate;

        @RecipeBuilderMethodDescription
        public RecipeBuilder minimumTier(int minimumTier) {
            this.minimumTier = minimumTier;
            return this;
        }

        @RecipeBuilderMethodDescription(field = "minimumTier")
        public RecipeBuilder tier(int tier) {
            return minimumTier(tier);
        }

        @RecipeBuilderMethodDescription
        public RecipeBuilder syphon(int syphon) {
            this.syphon = syphon;
            return this;
        }

        @RecipeBuilderMethodDescription
        public RecipeBuilder consumeRate(int consumeRate) {
            this.consumeRate = consumeRate;
            return this;
        }

        @RecipeBuilderMethodDescription
        public RecipeBuilder drainRate(int drainRate) {
            this.drainRate = drainRate;
            return this;
        }

        @Override
        protected int getMaxItemInput() {
            return 1;
        }

        @Override
        public String getErrorMsg() {
            return "Error adding Blood Magic Blood Altar recipe";
        }

        @Override
        public void validate(GroovyLog.Msg msg) {
            validateItems(msg, 1, 1, 1, 1);
            msg.add(minimumTier < 0, "minimumTier must be a nonnegative integer, yet it was {}", minimumTier);
            msg.add(minimumTier > AltarTier.MAXTIERS, "minimumTier must be less than the max tier (which is {}), yet it was {}", AltarTier.MAXTIERS, minimumTier);
            msg.add(syphon < 0, "syphon must be a nonnegative integer, yet it was {}", syphon);
            msg.add(consumeRate < 0, "consumeRate must be a nonnegative integer, yet it was {}", consumeRate);
            msg.add(drainRate < 0, "drainRate must be a nonnegative integer, yet it was {}", drainRate);
        }

        @Override
        @RecipeBuilderRegistrationMethod
        public @Nullable RecipeBloodAltar register() {
            if (!validate()) return null;
            RecipeBloodAltar recipe = new RecipeBloodAltar(input.get(0).toMcIngredient(), output.get(0), minimumTier, syphon, consumeRate, drainRate);
            ModSupport.BLOOD_MAGIC.get().bloodAltar.add(recipe);
            return recipe;
        }
    }
}
