package com.cleanroommc.groovyscript.compat.mods.mekanism;

import com.cleanroommc.groovyscript.api.GroovyLog;
import com.cleanroommc.groovyscript.api.IIngredient;
import com.cleanroommc.groovyscript.api.documentation.annotations.*;
import com.cleanroommc.groovyscript.compat.mods.ModSupport;
import com.cleanroommc.groovyscript.compat.mods.mekanism.recipe.GasRecipeBuilder;
import com.cleanroommc.groovyscript.compat.mods.mekanism.recipe.VirtualizedMekanismRegistry;
import com.cleanroommc.groovyscript.helper.Alias;
import com.cleanroommc.groovyscript.helper.ingredient.IngredientHelper;
import mekanism.api.gas.GasStack;
import mekanism.common.recipe.RecipeHandler;
import mekanism.common.recipe.inputs.AdvancedMachineInput;
import mekanism.common.recipe.machines.PurificationRecipe;
import mekanism.common.recipe.outputs.ItemStackOutput;
import net.minecraft.item.ItemStack;
import org.jetbrains.annotations.Nullable;

@RegistryDescription
public class PurificationChamber extends VirtualizedMekanismRegistry<PurificationRecipe> {

    public PurificationChamber() {
        super(RecipeHandler.Recipe.PURIFICATION_CHAMBER, Alias.generateOfClassAnd(PurificationChamber.class, "Purifier"));
    }

    @RecipeBuilderDescription(example = @Example(".input(item('minecraft:diamond')).gasInput(gas('deuterium')).output(item('minecraft:nether_star'))"))
    public RecipeBuilder recipeBuilder() {
        return new RecipeBuilder();
    }

    @MethodDescription(type = MethodDescription.Type.ADDITION, example = @Example(value = "item('minecraft:diamond'), gas('oxygen'), item('minecraft:nether_star')", commented = true))
    public PurificationRecipe add(IIngredient ingredient, GasStack gasInput, ItemStack output) {
        return recipeBuilder().gasInput(gasInput).output(output).input(ingredient).register();
    }

    @MethodDescription(example = @Example("item('mekanism:oreblock:0'), gas('oxygen')"))
    public boolean removeByInput(IIngredient ingredient, GasStack gasInput) {
        GroovyLog.Msg msg = GroovyLog.msg("Error removing Mekanism Purification Chamber recipe").error();
        msg.add(IngredientHelper.isEmpty(ingredient), () -> "input must not be empty");
        msg.add(Mekanism.isEmpty(gasInput), () -> "gas input must not be empty");
        if (msg.postIfNotEmpty()) return false;

        boolean found = false;
        for (ItemStack itemStack : ingredient.getMatchingStacks()) {
            PurificationRecipe recipe = recipeRegistry.get().remove(new AdvancedMachineInput(itemStack, gasInput.getGas()));
            if (recipe != null) {
                addBackup(recipe);
                found = true;
            }
        }
        if (!found) {
            removeError("could not find recipe for {} and {}", ingredient, gasInput);
        }
        return found;
    }

    @Property(property = "input", comp = @Comp(eq = 1))
    @Property(property = "output", comp = @Comp(eq = 1))
    @Property(property = "gasInput", comp = @Comp(eq = 1))
    public static class RecipeBuilder extends GasRecipeBuilder<PurificationRecipe> {

        @Override
        public String getErrorMsg() {
            return "Error adding Mekanism Purification Chamber recipe";
        }

        @Override
        public void validate(GroovyLog.Msg msg) {
            validateItems(msg, 1, 1, 1, 1);
            validateFluids(msg);
            validateGases(msg, 1, 1, 0, 0);
        }

        @Override
        @RecipeBuilderRegistrationMethod
        public @Nullable PurificationRecipe register() {
            if (!validate()) return null;
            PurificationRecipe recipe = null;
            for (ItemStack itemStack : input.get(0).getMatchingStacks()) {
                PurificationRecipe r = new PurificationRecipe(new AdvancedMachineInput(itemStack, gasInput.get(0).getGas()), new ItemStackOutput(output.get(0)));
                if (recipe == null) recipe = r;
                ModSupport.MEKANISM.get().purificationChamber.add(r);
            }
            return recipe;
        }
    }
}
