import json
import os
import tkinter as tk
from tkinter import filedialog, messagebox

class MinecraftModelBuilder:
    def __init__(self, root):
        self.root = root
        self.root.title("Minecraft Model Builder")
        self.root.geometry("600x540")
        self.root.resizable(False, False)

        self.layers = []

        # ====== Parent model input ======
        parent_frame = tk.Frame(root)
        parent_frame.pack(pady=10, fill="x", padx=20)
        tk.Label(parent_frame, text="Parent Model:").pack(anchor="w")
        self.parent_entry = tk.Entry(parent_frame, width=60)
        self.parent_entry.pack(side="left", padx=(0, 10))
        self.parent_entry.insert(0, "minecraft:item/")

        # ====== Folder selection ======
        folder_frame = tk.Frame(root)
        folder_frame.pack(pady=10, fill="x", padx=20)
        tk.Label(folder_frame, text="Save Folder:").pack(anchor="w")
        self.folder_path = tk.StringVar()
        folder_entry = tk.Entry(folder_frame, textvariable=self.folder_path, width=50)
        folder_entry.pack(side="left", padx=(0, 10))
        tk.Button(folder_frame, text="Browse...", command=self.select_folder).pack(side="left")

        # ====== File name input ======
        file_frame = tk.Frame(root)
        file_frame.pack(pady=10, fill="x", padx=20)
        tk.Label(file_frame, text="Output File Name:").pack(anchor="w")
        self.filename_entry = tk.Entry(file_frame, width=60)
        self.filename_entry.pack(side="left", padx=(0, 10))
        self.filename_entry.insert(0, "new_model")

        # ====== Layers section ======
        layer_title = tk.Label(root, text="Texture Layers:")
        layer_title.pack(anchor="w", padx=20)

        self.layers_frame = tk.Frame(root)
        self.layers_frame.pack(pady=10, fill="both", expand=True, padx=20)

        # Initial layer
        self.add_layer()

        # Add layer button
        add_layer_button = tk.Button(root, text="+ Add Layer", command=self.add_layer)
        add_layer_button.pack(pady=10)

        # Generate JSON button
        generate_button = tk.Button(
            root, 
            text="Generate JSON", 
            command=self.generate_json, 
            bg="#4CAF50", 
            fg="white"
        )
        generate_button.pack(pady=10)

        # Status label
        self.status_label = tk.Label(root, text="", fg="green")
        self.status_label.pack()

    # ====== Select Folder ======
    def select_folder(self):
        folder = filedialog.askdirectory()
        if folder:
            self.folder_path.set(folder)

    # ====== Add Layer ======
    def add_layer(self, default_value=""):
        layer_index = len(self.layers)
        frame = tk.Frame(self.layers_frame)
        frame.pack(anchor="w", pady=3)

        label = tk.Label(frame, text=f"layer{layer_index}:")
        label.pack(side="left")

        entry = tk.Entry(frame, width=50)
        entry.pack(side="left", padx=5)
        entry.insert(0, default_value)

        remove_btn = tk.Button(frame, text="x", fg="red", command=lambda f=frame: self.remove_layer(f))
        remove_btn.pack(side="left")

        self.layers.append((frame, entry))

    # ====== Remove Layer ======
    def remove_layer(self, frame):
        frame.destroy()
        self.layers = [(f, e) for (f, e) in self.layers if f != frame]
        self.relabel_layers()

    # ====== Re-Label Layers ======
    def relabel_layers(self):
        for i, (frame, entry) in enumerate(self.layers):
            label = frame.winfo_children()[0]
            label.config(text=f"layer{i}:")

    # ====== Generate JSON ======
    def generate_json(self):
        parent = self.parent_entry.get().strip()
        folder = self.folder_path.get().strip()
        filename = self.filename_entry.get().strip()

        if not parent:
            self.set_status("❌ Please enter a parent model.", error=True)
            return
        if not folder:
            self.set_status("❌ Please select a folder to save the file.", error=True)
            return
        if not filename:
            self.set_status("❌ Please enter a file name.", error=True)
            return

        # Construct JSON data
        textures = {}
        for i, (frame, entry) in enumerate(self.layers):
            value = entry.get().strip()
            if value:
                textures[f"layer{i}"] = value

        model_data = {
            "parent": parent,
            "textures": textures
        }

        # Save the file directly
        save_path = os.path.join(folder, f"{filename}.json")
        try:
            with open(save_path, "w") as f:
                json.dump(model_data, f, indent=2)
            self.set_status(f"✅ Saved successfully to: {save_path}")
        except Exception as e:
            self.set_status(f"❌ Error saving file: {e}", error=True)

    # ====== Update Status Label ======
    def set_status(self, text, error=False):
        self.status_label.config(text=text, fg="red" if error else "green")

# ====== Run App ======
if __name__ == "__main__":
    root = tk.Tk()
    app = MinecraftModelBuilder(root)
    root.mainloop()
