# -*- coding: utf-8 -*-
import os
import tkinter as tk
from tkinter import messagebox

def replace_lines_in_file(file_path, search_text, replacement_line):
    """
    Searches for lines containing 'search_text' and replaces the entire line.
    """
    with open(file_path, 'r', encoding='utf-8') as f:
        lines = f.readlines()

    modified = False
    new_lines = []
    for line in lines:
        if search_text in line:
            new_lines.append(replacement_line + '\n')
            modified = True
        else:
            new_lines.append(line)

    if modified:
        with open(file_path, 'w', encoding='utf-8') as f:
            f.writelines(new_lines)
        return f"✅ Replaced lines in {os.path.relpath(file_path)}"
    else:
        return f"⚠️ No match found in {os.path.relpath(file_path)}"


def replace_in_all_subfolders(search_text, replacement_line, file_extension=".txt"):
    """
    Replaces lines containing 'search_text' in all files with the given extension
    within the current folder and all subfolders (recursive).
    """
    current_dir = os.path.dirname(os.path.abspath(__file__))
    results = []

    for root, _, files in os.walk(current_dir):
        for filename in files:
            if filename.endswith(file_extension):
                file_path = os.path.join(root, filename)
                result = replace_lines_in_file(file_path, search_text, replacement_line)
                results.append(result)

    return results


def on_submit():
    try:
        search_text = search_text_entry.get().strip()
        replacement_line = replacement_entry.get().strip()
        file_extension = extension_entry.get().strip() or ".txt"

        if not file_extension.startswith('.'):
            file_extension = '.' + file_extension

        if not search_text or not replacement_line:
            messagebox.showwarning("Missing Info", "Please enter both search text and replacement line.")
            return

        results = replace_in_all_subfolders(search_text, replacement_line, file_extension)
        message = "\n".join(results) if results else "No files processed."
        messagebox.showinfo("Done", message)
    except Exception as e:
        messagebox.showerror("Error", f"An error occurred:\n{e}")


# === Tkinter GUI ===
root = tk.Tk()
root.title("Recursive Line Replacer")
root.geometry("420x300")
root.resizable(False, False)

tk.Label(root, text="Text to search for:").pack(pady=5)
search_text_entry = tk.Entry(root, width=50)
search_text_entry.pack(pady=5)

tk.Label(root, text="Replacement line:").pack(pady=5)
replacement_entry = tk.Entry(root, width=50)
replacement_entry.pack(pady=5)

tk.Label(root, text="File extension (e.g., .txt, .csv):").pack(pady=5)
extension_entry = tk.Entry(root, width=20)
extension_entry.insert(0, ".txt")
extension_entry.pack(pady=5)

tk.Button(root, text="Run Replacement", command=on_submit, bg="#4CAF50", fg="white").pack(pady=15)

root.mainloop()
