import os
import tkinter as tk
from tkinter import ttk, simpledialog, messagebox

def find_files_with_extension(base_dir, extension):
    """Recursively find all files with the given extension in the directory."""
    matched_files = []
    for root, _, files in os.walk(base_dir):
        for file in files:
            # Normalize both filename and extension for comparison
            if file.lower().endswith(f".{extension.lower()}"):
                matched_files.append(os.path.join(root, file))
    return matched_files

def add_line_to_file(file_path, new_line):
    """Add a new line at the next available free (empty) line in the file."""
    try:
        with open(file_path, "r+", encoding="utf-8") as f:
            lines = f.readlines()
            for i, line in enumerate(lines):
                if line.strip() == "":
                    lines[i] = new_line + "\n"
                    break
            else:
                if not lines or lines[-1].endswith("\n"):
                    lines.append(new_line + "\n")
                else:
                    lines.append("\n" + new_line + "\n")

            f.seek(0)
            f.writelines(lines)
            f.truncate()
    except UnicodeDecodeError:
        # Retry with a fallback encoding if file isn't UTF-8
        with open(file_path, "r+", encoding="latin-1") as f:
            lines = f.readlines()
            for i, line in enumerate(lines):
                if line.strip() == "":
                    lines[i] = new_line + "\n"
                    break
            else:
                if not lines or lines[-1].endswith("\n"):
                    lines.append(new_line + "\n")
                else:
                    lines.append("\n" + new_line + "\n")

            f.seek(0)
            f.writelines(lines)
            f.truncate()

def main():
    root = tk.Tk()
    root.withdraw()  # Hide main window initially

    base_dir = os.getcwd()

    # Ask user which extension to search for
    extension = simpledialog.askstring(
        "File Extension",
        "Enter the file extension to search for (e.g., txt, log, properties, config):"
    )
    if not extension:
        messagebox.showinfo("Cancelled", "No extension entered. Exiting.")
        return

    # Recursively find matching files
    files_found = find_files_with_extension(base_dir, extension)

    if not files_found:
        messagebox.showinfo(
            "No Files Found",
            f"No .{extension} files found in '{base_dir}' or its subfolders."
        )
        return

    # Show dropdown GUI for file selection
    root.deiconify()
    root.title("Select File to Edit")
    root.geometry("700x180")

    tk.Label(root, text=f"Select a .{extension} file:", font=("Arial", 12)).pack(pady=10)

    file_var = tk.StringVar(value=files_found[0])
    file_dropdown = ttk.Combobox(root, textvariable=file_var, values=files_found, width=100)
    file_dropdown.pack(pady=5)

    def on_select():
        file_path = file_var.get()
        new_line = simpledialog.askstring("New Line", "Enter the text to add:")
        if not new_line:
            return

        try:
            add_line_to_file(file_path, new_line)
            messagebox.showinfo("Success", f"Line added to:\n{file_path}")
        except Exception as e:
            messagebox.showerror("Error", f"Could not write to file:\n{e}")
        root.destroy()

    tk.Button(root, text="Add Line", command=on_select).pack(pady=10)
    root.mainloop()

if __name__ == "__main__":
    main()
