/*
 * Decompiled with CFR 0.152.
 */
package com.ubivismedia.aidungeon.api;

import com.ubivismedia.aidungeon.AIDungeon;
import com.ubivismedia.aidungeon.config.ConfigManager;
import com.ubivismedia.aidungeon.lib.gson.Gson;
import com.ubivismedia.aidungeon.lib.gson.GsonBuilder;
import com.ubivismedia.aidungeon.lib.gson.JsonObject;
import com.ubivismedia.aidungeon.lib.slf4j.Logger;
import com.ubivismedia.aidungeon.lib.slf4j.LoggerFactory;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Queue;
import java.util.Scanner;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

public class APIReporter {
    private static final Logger log = LoggerFactory.getLogger(APIReporter.class);
    private final AIDungeon plugin;
    private final ConfigManager configManager;
    private final Gson gson;
    private final File reportingDataFolder;
    private final File reportedEntitiesFile;
    private final File reportCacheFolder;
    private final String serverUUID;
    private final ScheduledExecutorService scheduler;
    private final Queue<ReportItem> reportQueue = new ConcurrentLinkedQueue<ReportItem>();
    private final Map<String, Long> lastReportedTimes = new HashMap<String, Long>();
    private static final String API_ENDPOINT_ENTITIES = "?endpoint=entities";
    private static final String API_ENDPOINT_STATS = "?endpoint=stats";
    private static final String REPORTED_ENTITIES_CONFIG = "reported_entities.yml";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final int MAX_RETRY_COUNT = 3;

    public APIReporter(AIDungeon plugin) {
        this.plugin = plugin;
        this.configManager = plugin.getConfigManager();
        this.gson = new GsonBuilder().setPrettyPrinting().create();
        this.scheduler = Executors.newScheduledThreadPool(1);
        this.reportingDataFolder = new File(plugin.getDataFolder(), "reporting");
        if (!this.reportingDataFolder.exists()) {
            this.reportingDataFolder.mkdirs();
        }
        this.reportCacheFolder = new File(this.reportingDataFolder, "cache");
        if (!this.reportCacheFolder.exists()) {
            this.reportCacheFolder.mkdirs();
        }
        this.reportedEntitiesFile = new File(this.reportingDataFolder, REPORTED_ENTITIES_CONFIG);
        String storedUUID = this.loadServerUUID();
        if (storedUUID == null || storedUUID.isEmpty()) {
            this.serverUUID = UUID.randomUUID().toString();
            this.saveServerUUID(this.serverUUID);
        } else {
            this.serverUUID = storedUUID;
        }
        if (this.isReportingEnabled()) {
            this.registerWithAPI().thenAccept(registered -> {
                if (registered.booleanValue()) {
                    plugin.getLogger().info("Successfully registered with API server");
                } else {
                    plugin.getLogger().warning("Could not register with API server, will retry on next report");
                }
                this.scheduleReporting();
                this.loadCachedReports();
            });
        }
    }

    private String loadServerUUID() {
        File uuidFile = new File(this.reportingDataFolder, "server_uuid.txt");
        if (!uuidFile.exists()) {
            return null;
        }
        try {
            Scanner scanner = new Scanner(uuidFile);
            String uuid = scanner.useDelimiter("\\Z").next();
            scanner.close();
            return uuid.trim();
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to load server UUID", e);
            return null;
        }
    }

    private void saveServerUUID(String uuid) {
        File uuidFile = new File(this.reportingDataFolder, "server_uuid.txt");
        try {
            FileWriter writer = new FileWriter(uuidFile);
            writer.write(uuid);
            writer.close();
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to save server UUID", e);
        }
    }

    public boolean isReportingEnabled() {
        return this.configManager.isApiReportingEnabled();
    }

    private void scheduleReporting() {
        int reportFrequency = this.configManager.getApiReportingFrequency();
        this.scheduler.scheduleWithFixedDelay(this::processReportQueue, 60L, reportFrequency * 60, TimeUnit.SECONDS);
        this.plugin.getLogger().info("API reporting scheduled every " + reportFrequency + " minutes");
    }

    private void loadCachedReports() {
        File[] cacheFiles = this.reportCacheFolder.listFiles((dir, name) -> name.endsWith(".json"));
        if (cacheFiles == null || cacheFiles.length == 0) {
            return;
        }
        this.plugin.getLogger().info("Found " + cacheFiles.length + " cached reports, adding to queue");
        for (File file : cacheFiles) {
            try {
                Scanner scanner = new Scanner(file);
                String json = scanner.useDelimiter("\\Z").next();
                scanner.close();
                ReportItem item = this.gson.fromJson(json, ReportItem.class);
                if (item != null) {
                    this.reportQueue.add(item);
                }
                file.delete();
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.WARNING, "Failed to load cached report: " + file.getName(), e);
            }
        }
    }

    public CompletableFuture<Boolean> reportEntity(String entityType, String entityName, JsonObject entityData, String dungeonTheme, String biome) {
        if (!this.isReportingEnabled()) {
            return CompletableFuture.completedFuture(false);
        }
        String entityKey = (entityType = entityType.toUpperCase()) + ":" + (entityName = entityName.replace(" ", "_").toUpperCase());
        if (this.hasBeenReportedRecently(entityKey)) {
            this.logDebug("Entity " + entityKey + " was reported recently, skipping report");
            return CompletableFuture.completedFuture(false);
        }
        String entityId = UUID.nameUUIDFromBytes(entityKey.getBytes()).toString();
        ReportItem reportItem = new ReportItem();
        reportItem.id = entityId;
        reportItem.entityType = entityType;
        reportItem.entityName = entityName;
        reportItem.entityData = entityData.toString();
        reportItem.dungeonTheme = dungeonTheme;
        reportItem.biome = biome;
        reportItem.pluginVersion = this.plugin.getDescription().getVersion();
        reportItem.reportType = "ENTITY";
        reportItem.serverUuid = this.serverUUID;
        this.reportQueue.add(reportItem);
        this.markAsReported(entityKey);
        this.logDebug("Entity " + entityKey + " added to report queue");
        return CompletableFuture.completedFuture(true);
    }

    public CompletableFuture<Boolean> reportData(String reportType, String reportId, JsonObject dataObject) {
        if (!this.isReportingEnabled()) {
            return CompletableFuture.completedFuture(false);
        }
        reportType = reportType.toUpperCase();
        ReportItem reportItem = new ReportItem();
        reportItem.id = reportId;
        reportItem.reportType = reportType;
        reportItem.rawData = dataObject.toString();
        reportItem.pluginVersion = this.plugin.getDescription().getVersion();
        reportItem.serverUuid = this.serverUUID;
        this.reportQueue.add(reportItem);
        this.logDebug("Data of type " + reportType + " added to report queue");
        return CompletableFuture.completedFuture(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean hasBeenReportedRecently(String entityKey) {
        Map<String, Long> map = this.lastReportedTimes;
        synchronized (map) {
            if (!this.lastReportedTimes.containsKey(entityKey)) {
                return false;
            }
            int cooldownHours = this.configManager.getApiReportingResendCooldown();
            long cooldownMillis = (long)cooldownHours * 60L * 60L * 1000L;
            long lastReportTime = this.lastReportedTimes.get(entityKey);
            return System.currentTimeMillis() - lastReportTime < cooldownMillis;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void markAsReported(String entityKey) {
        Map<String, Long> map = this.lastReportedTimes;
        synchronized (map) {
            this.lastReportedTimes.put(entityKey, System.currentTimeMillis());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processReportQueue() {
        if (this.reportQueue.isEmpty()) {
            this.logDebug("No reports to process");
            return;
        }
        this.logDebug("Processing report queue with " + this.reportQueue.size() + " items");
        int batchSize = this.configManager.getApiReportingBatchSize();
        ArrayList<ReportItem> batch = new ArrayList<ReportItem>(batchSize);
        Queue<ReportItem> queue = this.reportQueue;
        synchronized (queue) {
            int count = 0;
            while (!this.reportQueue.isEmpty() && count < batchSize) {
                ReportItem item = this.reportQueue.poll();
                if (item == null) continue;
                batch.add(item);
                ++count;
            }
        }
        if (batch.isEmpty()) {
            return;
        }
        this.logDebug("Processing batch of " + batch.size() + " reports");
        for (ReportItem item : batch) {
            try {
                boolean success = false;
                success = "ENTITY".equals(item.reportType) ? this.sendEntityReport(item) : this.sendGenericReport(item);
                if (success) continue;
                this.cacheFailedReport(item);
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.WARNING, "Error processing report", e);
                this.cacheFailedReport(item);
            }
        }
    }

    private boolean sendEntityReport(ReportItem reportItem) {
        String apiUrl = this.configManager.getApiReportingBaseUrl() + API_ENDPOINT_ENTITIES;
        JsonObject requestBody = new JsonObject();
        requestBody.addProperty("id", reportItem.id);
        requestBody.addProperty("entityType", reportItem.entityType);
        requestBody.addProperty("entityName", reportItem.entityName);
        requestBody.addProperty("entityData", reportItem.entityData);
        requestBody.addProperty("dungeonTheme", reportItem.dungeonTheme);
        requestBody.addProperty("biome", reportItem.biome);
        requestBody.addProperty("pluginVersion", reportItem.pluginVersion);
        requestBody.addProperty("serverUuid", reportItem.serverUuid);
        try {
            int responseCode = this.sendHttpRequest(apiUrl, requestBody.toString());
            if (responseCode == 200 || responseCode == 409) {
                this.logDebug("Entity report sent successfully, response: " + responseCode);
                return true;
            }
            this.plugin.getLogger().warning("Failed to send entity report, response code: " + responseCode);
            return false;
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.WARNING, "Error sending entity report", e);
            return false;
        }
    }

    private boolean sendGenericReport(ReportItem reportItem) {
        Object endpoint = "";
        switch (reportItem.reportType) {
            case "STATS": {
                endpoint = API_ENDPOINT_STATS;
                break;
            }
            default: {
                endpoint = "?endpoint=" + reportItem.reportType.toLowerCase();
            }
        }
        String apiUrl = this.configManager.getApiReportingBaseUrl() + (String)endpoint;
        String requestBody = reportItem.rawData;
        try {
            int responseCode = this.sendHttpRequest(apiUrl, requestBody);
            if (responseCode == 200) {
                this.logDebug("Report sent successfully, response: " + responseCode);
                return true;
            }
            this.plugin.getLogger().warning("Failed to send report, response code: " + responseCode);
            return false;
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.WARNING, "Error sending report", e);
            return false;
        }
    }

    private int sendHttpRequest(String apiUrl, String jsonPayload) throws IOException {
        URL url = new URL(apiUrl);
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setRequestMethod("POST");
        connection.setRequestProperty("Content-Type", "application/json");
        connection.setRequestProperty("User-Agent", "AIDungeon/" + this.plugin.getDescription().getVersion());
        connection.setDoOutput(true);
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(10000);
        try (OutputStream os = connection.getOutputStream();){
            byte[] input = jsonPayload.getBytes(StandardCharsets.UTF_8);
            os.write(input, 0, input.length);
        }
        return connection.getResponseCode();
    }

    private void cacheFailedReport(ReportItem reportItem) {
        String filename = reportItem.reportType + "_" + reportItem.id + "_" + System.currentTimeMillis() + ".json";
        File cacheFile = new File(this.reportCacheFolder, filename);
        try {
            String json = this.gson.toJson(reportItem);
            FileWriter writer = new FileWriter(cacheFile);
            writer.write(json);
            writer.close();
            this.logDebug("Cached failed report: " + filename);
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to cache report", e);
        }
    }

    private void logDebug(String message) {
        if (this.configManager.getConfig().getBoolean("debug.logApiCalls", false)) {
            this.plugin.getLogger().info("[API] " + message);
        }
    }

    private CompletableFuture<Boolean> registerWithAPI() {
        CompletableFuture<Boolean> future = new CompletableFuture<Boolean>();
        if (!this.isReportingEnabled()) {
            future.complete(false);
            return future;
        }
        JsonObject registrationData = new JsonObject();
        registrationData.addProperty("serverUuid", this.serverUUID);
        registrationData.addProperty("pluginVersion", this.plugin.getDescription().getVersion());
        registrationData.addProperty("serverSoftware", this.plugin.getServer().getName());
        registrationData.addProperty("serverVersion", this.plugin.getServer().getVersion());
        registrationData.addProperty("minecraftVersion", this.plugin.getServer().getBukkitVersion());
        String serverName = this.plugin.getServer().getMotd();
        if (serverName != null && !serverName.isEmpty()) {
            registrationData.addProperty("serverName", serverName);
        }
        int playerCount = this.plugin.getServer().getOnlinePlayers().size();
        registrationData.addProperty("playerCount", playerCount);
        try {
            String apiUrl = this.configManager.getApiReportingBaseUrl() + "?endpoint=register";
            String jsonPayload = registrationData.toString();
            int responseCode = this.sendHttpRequest(apiUrl, jsonPayload);
            if (responseCode == 200) {
                this.logDebug("Plugin successfully registered with API server");
                future.complete(true);
            } else {
                this.plugin.getLogger().warning("Failed to register plugin with API server, response code: " + responseCode);
                future.complete(false);
            }
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.WARNING, "Error registering plugin with API server", e);
            future.complete(false);
        }
        return future;
    }

    public void shutdown() {
        this.scheduler.shutdown();
        try {
            this.processReportQueue();
            this.scheduler.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            this.plugin.getLogger().log(Level.WARNING, "Error shutting down API reporter", e);
            Thread.currentThread().interrupt();
        }
    }

    private static class ReportItem {
        public String id;
        public String reportType;
        public String pluginVersion;
        public String serverUuid;
        public String entityType;
        public String entityName;
        public String entityData;
        public String dungeonTheme;
        public String biome;
        public String rawData;
        public int retryCount = 0;

        private ReportItem() {
        }
    }
}

