/*
 * Decompiled with CFR 0.152.
 */
package com.ubivismedia.aidungeon.dungeon;

import com.ubivismedia.aidungeon.AIDungeon;
import com.ubivismedia.aidungeon.api.LightweightGeminiClient;
import com.ubivismedia.aidungeon.config.ConfigManager;
import com.ubivismedia.aidungeon.database.DatabaseManager;
import com.ubivismedia.aidungeon.dungeon.DungeonGenerator;
import com.ubivismedia.aidungeon.integration.CustomMobResolver;
import com.ubivismedia.aidungeon.model.Dungeon;
import com.ubivismedia.aidungeon.model.Room;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.logging.Level;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.World;
import org.bukkit.block.Biome;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;

public class DungeonManager {
    private final AIDungeon plugin;
    private final DatabaseManager databaseManager;
    private final ConfigManager configManager;
    private final LightweightGeminiClient geminiApiClient;
    private final Map<Integer, Dungeon> activeDungeons = new HashMap<Integer, Dungeon>();
    private final DungeonGenerator dungeonGenerator;
    private final CustomMobResolver customMobResolver;
    private boolean initialLoadComplete = false;

    public DungeonManager(AIDungeon plugin) {
        this.plugin = plugin;
        this.databaseManager = plugin.getDatabaseManager();
        this.configManager = plugin.getConfigManager();
        String apiKey = this.configManager.getGeminiApiKey();
        String model = this.configManager.getGeminiModel();
        int timeout = this.configManager.getGeminiTimeout();
        this.geminiApiClient = new LightweightGeminiClient(plugin, apiKey, model, timeout);
        this.customMobResolver = new CustomMobResolver(plugin);
        this.dungeonGenerator = new DungeonGenerator(plugin, this, this.geminiApiClient, this.customMobResolver);
        Bukkit.getScheduler().runTaskLater((Plugin)plugin, this::loadActiveDungeons, 100L);
    }

    private void loadActiveDungeons() {
        this.plugin.getLogger().info("Loading active dungeons from database...");
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            try {
                if (!this.databaseManager.isConnectionHealthy()) {
                    this.plugin.getLogger().warning("Database not ready for loading dungeons. Will retry in 10 seconds.");
                    Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, this::loadActiveDungeons, 200L);
                    return;
                }
                ArrayList<Dungeon> dungeonBasicInfo = new ArrayList<Dungeon>();
                try (Connection conn = this.databaseManager.getConnection();
                     PreparedStatement stmt = conn.prepareStatement("SELECT id, world_name, biome_type, theme, x_coord, y_coord, z_coord, created_at, status, boss_defeated FROM dungeons WHERE status = 'ACTIVE' OR status = 'GENERATING'");){
                    ResultSet rs = stmt.executeQuery();
                    while (rs.next()) {
                        Dungeon dungeon = new Dungeon();
                        dungeon.setId(rs.getInt("id"));
                        dungeon.setWorldName(rs.getString("world_name"));
                        dungeon.setBiomeType(rs.getString("biome_type"));
                        dungeon.setTheme(rs.getString("theme"));
                        dungeon.setXCoord(rs.getInt("x_coord"));
                        dungeon.setYCoord(rs.getInt("y_coord"));
                        dungeon.setZCoord(rs.getInt("z_coord"));
                        dungeon.setCreatedAt(rs.getTimestamp("created_at").toLocalDateTime());
                        dungeon.setStatus(Dungeon.Status.valueOf(rs.getString("status")));
                        dungeon.setBossDefeated(rs.getBoolean("boss_defeated"));
                        dungeonBasicInfo.add(dungeon);
                    }
                }
                this.plugin.getLogger().info("Found " + dungeonBasicInfo.size() + " active dungeons in database. Starting lazy loading...");
                for (Dungeon dungeon : dungeonBasicInfo) {
                    this.activeDungeons.put(dungeon.getId(), dungeon);
                }
                this.initialLoadComplete = true;
                this.loadDungeonRoomsAsync(dungeonBasicInfo);
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to load active dungeons: ", e);
                this.initialLoadComplete = true;
                Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, this::loadActiveDungeons, 200L);
            }
        });
    }

    private void loadDungeonRoomsAsync(List<Dungeon> dungeons) {
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            int loadedCount = 0;
            for (Dungeon dungeon : dungeons) {
                try {
                    if (dungeon.getRooms() != null && !dungeon.getRooms().isEmpty()) continue;
                    this.loadRoomsForDungeon(dungeon);
                    if (++loadedCount % 5 == 0) {
                        this.plugin.getLogger().info("Loaded rooms for " + loadedCount + " of " + dungeons.size() + " dungeons.");
                    }
                    try {
                        Thread.sleep(200L);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
                catch (SQLException e) {
                    this.plugin.getLogger().log(Level.WARNING, "Error loading rooms for dungeon #" + dungeon.getId(), e);
                }
            }
            this.plugin.getLogger().info("Completed loading rooms for " + loadedCount + " dungeons.");
        });
    }

    public Dungeon getDungeon(int dungeonId) {
        Dungeon dungeon = this.activeDungeons.get(dungeonId);
        if (dungeon != null && (dungeon.getRooms() == null || dungeon.getRooms().isEmpty())) {
            try {
                this.loadRoomsForDungeon(dungeon);
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.WARNING, "Error lazy-loading rooms for dungeon #" + dungeonId, e);
            }
        }
        return dungeon;
    }

    private void loadRoomsForDungeon(Dungeon dungeon) throws SQLException {
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("SELECT * FROM rooms WHERE dungeon_id = ?");){
            stmt.setInt(1, dungeon.getId());
            ResultSet rs = stmt.executeQuery();
            ArrayList<Room> rooms = new ArrayList<Room>();
            while (rs.next()) {
                Room room = new Room();
                room.setId(rs.getInt("id"));
                room.setDungeonId(rs.getInt("dungeon_id"));
                room.setRoomType(rs.getString("room_type"));
                room.setXCoord(rs.getInt("x_coord"));
                room.setYCoord(rs.getInt("y_coord"));
                room.setZCoord(rs.getInt("z_coord"));
                room.setWidth(rs.getInt("width"));
                room.setHeight(rs.getInt("height"));
                room.setLength(rs.getInt("length"));
                room.setExplorationStatus(Room.ExplorationStatus.valueOf(rs.getString("exploration_status")));
                rooms.add(room);
            }
            dungeon.setRooms(rooms);
        }
    }

    public boolean isInitialLoadComplete() {
        return this.initialLoadComplete;
    }

    public void scanAroundPlayer(Player player) {
        if (!this.initialLoadComplete) {
            return;
        }
        if (player == null || !player.isOnline()) {
            return;
        }
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            Location playerLoc = player.getLocation();
            World world = player.getWorld();
            if (world.getEnvironment() != World.Environment.NORMAL && !this.configManager.getConfig().getBoolean("generation.allowInOtherDimensions", false)) {
                return;
            }
            int scanRadius = this.configManager.getConfig().getInt("generation.scanRadius", 100);
            for (int r = 0; r <= scanRadius; r += 16) {
                for (int x = -r; x <= r; x += 16) {
                    for (int z = -r; z <= r; z += 16) {
                        if (r > 0 && Math.abs(x) < r && Math.abs(z) < r) continue;
                        int chunkX = playerLoc.getBlockX() + x;
                        int chunkZ = playerLoc.getBlockZ() + z;
                        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
                            if (!world.isChunkLoaded(chunkX >> 4, chunkZ >> 4)) {
                                world.getChunkAt(chunkX >> 4, chunkZ >> 4);
                                this.processChunk(world, chunkX, chunkZ);
                            } else {
                                this.processChunk(world, chunkX, chunkZ);
                            }
                        });
                    }
                }
            }
        });
    }

    private void processChunk(World world, int chunkX, int chunkZ) {
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            if (this.isChunkProcessed(world.getName(), chunkX, chunkZ)) {
                return;
            }
            if (this.isSuitableLocation(world, chunkX, chunkZ) && !this.isDungeonNearby(world.getName(), chunkX, chunkZ)) {
                if (!this.arePlayersNearby(world, chunkX, chunkZ)) {
                    this.scheduleDungeonGeneration(world, chunkX, chunkZ);
                } else {
                    this.markForLaterGeneration(world, chunkX, chunkZ);
                }
            }
            this.markChunkAsProcessed(world.getName(), chunkX, chunkZ);
        });
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private boolean isChunkProcessed(String worldName, int chunkX, int chunkZ) {
        try (Connection conn = this.databaseManager.getConnection();){
            boolean bl;
            block14: {
                PreparedStatement stmt = conn.prepareStatement("SELECT 1 FROM processed_chunks WHERE world_name = ? AND chunk_x = ? AND chunk_z = ?");
                try {
                    stmt.setString(1, worldName);
                    stmt.setInt(2, chunkX >> 4);
                    stmt.setInt(3, chunkZ >> 4);
                    ResultSet rs = stmt.executeQuery();
                    bl = rs.next();
                    if (stmt == null) break block14;
                }
                catch (Throwable throwable) {
                    if (stmt != null) {
                        try {
                            stmt.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                stmt.close();
            }
            return bl;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error checking if chunk is processed: ", e);
            return false;
        }
    }

    private void markChunkAsProcessed(String worldName, int chunkX, int chunkZ) {
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("INSERT OR IGNORE INTO processed_chunks (world_name, chunk_x, chunk_z) VALUES (?, ?, ?)");){
            stmt.setString(1, worldName);
            stmt.setInt(2, chunkX >> 4);
            stmt.setInt(3, chunkZ >> 4);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error marking chunk as processed: ", e);
        }
    }

    private boolean isSuitableLocation(World world, int x, int z) {
        int y = world.getHighestBlockYAt(x, z);
        if (y < 40) {
            return false;
        }
        Material surfaceMaterial = world.getBlockAt(x, y, z).getType();
        if (surfaceMaterial == Material.WATER || surfaceMaterial == Material.ICE || surfaceMaterial == Material.SEAGRASS) {
            return false;
        }
        boolean hasSolidTerrain = false;
        int checkRadius = 20;
        int minSolidBlocks = 100;
        int solidBlockCount = 0;
        for (int dx = -checkRadius; dx <= checkRadius; dx += 5) {
            for (int dz = -checkRadius; dz <= checkRadius; dz += 5) {
                for (int dy = -10; dy <= 5; dy += 5) {
                    Material material;
                    int checkY = y + dy;
                    if (checkY <= 0 || checkY >= 256 || !(material = world.getBlockAt(x + dx, checkY, z + dz).getType()).isSolid() || material.isAir() || ++solidBlockCount < minSolidBlocks) continue;
                    hasSolidTerrain = true;
                    break;
                }
                if (hasSolidTerrain) break;
            }
            if (hasSolidTerrain) break;
        }
        if (!hasSolidTerrain) {
            return false;
        }
        Biome biome = world.getBiome(x, y, z);
        String biomeKey = biome.toString();
        HashMap biomeWeights = new HashMap();
        this.configManager.getConfig().getConfigurationSection("generation.biomeWeights").getKeys(false).forEach(key -> biomeWeights.put(key, this.configManager.getConfig().getDouble("generation.biomeWeights." + key, 0.0)));
        if (!biomeWeights.containsKey(biomeKey) || (Double)biomeWeights.get(biomeKey) <= 0.0) {
            return false;
        }
        double spawnChance = this.configManager.getDungeonSpawnChance() * biomeWeights.getOrDefault(biomeKey, 1.0);
        return !(Math.random() > spawnChance);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isDungeonNearby(String worldName, int x, int z) {
        int minDistance = this.configManager.getMinDistanceBetweenDungeons();
        for (Dungeon dungeon : this.activeDungeons.values()) {
            double distance;
            if (!dungeon.getWorldName().equals(worldName) || !((distance = Math.sqrt(Math.pow(dungeon.getXCoord() - x, 2.0) + Math.pow(dungeon.getZCoord() - z, 2.0))) < (double)minDistance)) continue;
            return true;
        }
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("SELECT COUNT(*) FROM dungeons WHERE world_name = ? AND x_coord BETWEEN ? AND ? AND z_coord BETWEEN ? AND ?");){
            stmt.setString(1, worldName);
            stmt.setInt(2, x - minDistance);
            stmt.setInt(3, x + minDistance);
            stmt.setInt(4, z - minDistance);
            stmt.setInt(5, z + minDistance);
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return false;
            if (rs.getInt(1) <= 0) return false;
            boolean bl = true;
            return bl;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error checking for nearby dungeons: ", e);
        }
        return false;
    }

    private boolean arePlayersNearby(World world, int x, int z) {
        int sightRange = this.configManager.getPlayerSightRange();
        for (Player player : world.getPlayers()) {
            Location playerLoc = player.getLocation();
            double distance = Math.sqrt(Math.pow(playerLoc.getX() - (double)x, 2.0) + Math.pow(playerLoc.getZ() - (double)z, 2.0));
            if (!(distance < (double)sightRange)) continue;
            return true;
        }
        return false;
    }

    private void markForLaterGeneration(World world, int x, int z) {
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("INSERT INTO pending_locations (world_name, biome_type, x_coord, z_coord) VALUES (?, ?, ?, ?)");){
            stmt.setString(1, world.getName());
            stmt.setString(2, world.getBiome(x, world.getHighestBlockYAt(x, z), z).toString());
            stmt.setInt(3, x);
            stmt.setInt(4, z);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to mark location for later generation: ", e);
        }
    }

    private void scheduleDungeonGeneration(World world, int x, int z) {
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
            int y = world.getHighestBlockYAt(x, z);
            Biome biome = world.getBiome(x, y, z);
            String biomeType = biome.toString();
            ((CompletableFuture)this.dungeonGenerator.generateDungeon(world, x, y, z, biomeType).thenAccept(dungeon -> {
                if (dungeon != null) {
                    this.activeDungeons.put(dungeon.getId(), (Dungeon)dungeon);
                    this.plugin.getLogger().info("Generated dungeon #" + dungeon.getId() + " at " + x + ", " + y + ", " + z);
                }
            })).exceptionally(ex -> {
                this.plugin.getLogger().log(Level.SEVERE, "Error generating dungeon: ", (Throwable)ex);
                return null;
            });
        });
    }

    public Map<Integer, Dungeon> getActiveDungeons() {
        return Collections.unmodifiableMap(this.activeDungeons);
    }

    public List<Dungeon> getDungeonsInWorld(String worldName) {
        return this.activeDungeons.values().stream().filter(d -> d.getWorldName().equals(worldName)).toList();
    }

    public void trackPlayerExploration(Player player, Dungeon dungeon, Room room) {
        if (player == null || dungeon == null || room == null) {
            return;
        }
        boolean debug = this.plugin.getConfigManager().getConfig().getBoolean("debug.logExploration", false);
        if (debug) {
            this.plugin.getLogger().info("DEBUG: Tracking exploration for player " + player.getName() + " in dungeon " + dungeon.getId() + ", room " + room.getId());
        }
        if (room.getExplorationStatus() != Room.ExplorationStatus.EXPLORED) {
            if (debug) {
                this.plugin.getLogger().info("DEBUG: Setting room " + room.getId() + " as EXPLORED (previously " + String.valueOf((Object)room.getExplorationStatus()) + ")");
            }
            room.setExplorationStatus(Room.ExplorationStatus.EXPLORED);
            Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
                try (Connection conn = this.plugin.getDatabaseManager().getConnection();
                     PreparedStatement stmt = conn.prepareStatement("UPDATE rooms SET exploration_status = ? WHERE id = ?");){
                    stmt.setString(1, Room.ExplorationStatus.EXPLORED.toString());
                    stmt.setInt(2, room.getId());
                    int updated = stmt.executeUpdate();
                    if (debug) {
                        this.plugin.getLogger().info("DEBUG: Database update for room exploration status: " + updated + " rows affected");
                    }
                }
                catch (SQLException e) {
                    this.plugin.getLogger().log(Level.SEVERE, "Failed to update room exploration status: ", e);
                }
            });
        }
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            try (Connection conn = this.databaseManager.getConnection();
                 PreparedStatement stmt = conn.prepareStatement("INSERT INTO player_progress (player_uuid, dungeon_id, exploration_percent, last_visit) VALUES (?, ?, ?, CURRENT_TIMESTAMP) ON " + (this.configManager.getDbType().equalsIgnoreCase("mysql") ? "DUPLICATE KEY UPDATE" : "CONFLICT(player_uuid, dungeon_id) DO UPDATE SET") + " exploration_percent = ?, last_visit = CURRENT_TIMESTAMP");){
                double explorationPercent = dungeon.getExplorationPercentage();
                stmt.setString(1, player.getUniqueId().toString());
                stmt.setInt(2, dungeon.getId());
                stmt.setDouble(3, explorationPercent);
                stmt.setDouble(4, explorationPercent);
                stmt.executeUpdate();
                if (dungeon.isFullyExplored() && dungeon.getStatus() == Dungeon.Status.ACTIVE) {
                    this.checkDungeonCompletion(dungeon);
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to update player progress: ", e);
            }
        });
    }

    private void checkDungeonCompletion(Dungeon dungeon) {
        if (!dungeon.isFullyExplored() || dungeon.getStatus() != Dungeon.Status.ACTIVE) {
            return;
        }
        dungeon.setStatus(Dungeon.Status.EXPLORED);
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("UPDATE dungeons SET status = ? WHERE id = ?");){
            stmt.setString(1, Dungeon.Status.EXPLORED.toString());
            stmt.setInt(2, dungeon.getId());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to update dungeon status: ", e);
            return;
        }
        if (this.configManager.getConfig().getBoolean("exploration.collapseAfterExploration", true)) {
            int collapseDelay = this.configManager.getConfig().getInt("exploration.collapseDelay", 24);
            this.plugin.getLogger().info("Scheduling collapse of dungeon #" + dungeon.getId() + " in " + collapseDelay + " hours");
            Bukkit.getScheduler().runTaskLaterAsynchronously((Plugin)this.plugin, () -> this.collapseDungeon(dungeon.getId()), (long)(collapseDelay * 60 * 60) * 20L);
        }
    }

    public void collapseDungeon(int dungeonId) {
        Object stmt;
        Connection conn;
        Dungeon dungeon = this.activeDungeons.get(dungeonId);
        if (dungeon == null) {
            try {
                conn = this.databaseManager.getConnection();
                try {
                    stmt = conn.prepareStatement("SELECT * FROM dungeons WHERE id = ?");
                    try {
                        stmt.setInt(1, dungeonId);
                        ResultSet rs = stmt.executeQuery();
                        if (rs.next()) {
                            dungeon = new Dungeon();
                            dungeon.setId(rs.getInt("id"));
                            dungeon.setWorldName(rs.getString("world_name"));
                            dungeon.setStatus(Dungeon.Status.valueOf(rs.getString("status")));
                            this.loadRoomsForDungeon(dungeon);
                        }
                    }
                    finally {
                        if (stmt != null) {
                            stmt.close();
                        }
                    }
                }
                finally {
                    if (conn != null) {
                        conn.close();
                    }
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to load dungeon for collapse: ", e);
                return;
            }
        }
        if (dungeon == null) {
            this.plugin.getLogger().warning("Failed to find dungeon #" + dungeonId + " for collapse.");
            return;
        }
        dungeon.setStatus(Dungeon.Status.COLLAPSING);
        try {
            conn = this.databaseManager.getConnection();
            try {
                stmt = conn.prepareStatement("UPDATE dungeons SET status = ? WHERE id = ?");
                try {
                    stmt.setString(1, Dungeon.Status.COLLAPSING.toString());
                    stmt.setInt(2, dungeonId);
                    stmt.executeUpdate();
                }
                finally {
                    if (stmt != null) {
                        stmt.close();
                    }
                }
            }
            finally {
                if (conn != null) {
                    conn.close();
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to update dungeon status for collapse: ", e);
        }
        World world = Bukkit.getWorld((String)dungeon.getWorldName());
        if (world != null) {
            block24: for (Player player : world.getPlayers()) {
                for (Room room : dungeon.getRooms()) {
                    if (!room.containsLocation(player.getLocation())) continue;
                    player.sendMessage("\u00a7c[AIDungeon] \u00a7eThe dungeon is collapsing! You have 60 seconds to escape!");
                    continue block24;
                }
            }
            Dungeon finalDungeon = dungeon;
            World finalWorld = world;
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> this.performDungeonCollapse(finalDungeon, finalWorld), 1200L);
        } else {
            this.markDungeonAsCollapsed(dungeonId);
        }
    }

    private void performDungeonCollapse(Dungeon dungeon, World world) {
        block0: for (Player player : world.getPlayers()) {
            for (Room room : dungeon.getRooms()) {
                if (!room.containsLocation(player.getLocation())) continue;
                Location surface = new Location(world, player.getLocation().getX(), (double)(world.getHighestBlockYAt(player.getLocation().getBlockX(), player.getLocation().getBlockZ()) + 1), player.getLocation().getZ());
                player.teleport(surface);
                player.sendMessage("\u00a7c[AIDungeon] \u00a7eYou have been teleported to safety.");
                continue block0;
            }
        }
        ((CompletableFuture)this.dungeonGenerator.collapseDungeon(dungeon, world).thenRun(() -> this.markDungeonAsCollapsed(dungeon.getId()))).exceptionally(ex -> {
            this.plugin.getLogger().log(Level.SEVERE, "Error during dungeon collapse: ", (Throwable)ex);
            this.markDungeonAsCollapsed(dungeon.getId());
            return null;
        });
    }

    private void markDungeonAsCollapsed(int dungeonId) {
        Dungeon dungeon = this.activeDungeons.get(dungeonId);
        if (dungeon != null) {
            dungeon.setStatus(Dungeon.Status.COLLAPSED);
        }
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("UPDATE dungeons SET status = ? WHERE id = ?");){
            stmt.setString(1, Dungeon.Status.COLLAPSED.toString());
            stmt.setInt(2, dungeonId);
            stmt.executeUpdate();
            this.activeDungeons.remove(dungeonId);
            this.plugin.getLogger().info("Dungeon #" + dungeonId + " has been marked as collapsed.");
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to mark dungeon as collapsed: ", e);
        }
    }

    public Dungeon getDungeonAtLocation(Location location) {
        for (Dungeon dungeon : this.activeDungeons.values()) {
            if (!dungeon.getWorldName().equals(location.getWorld().getName())) continue;
            for (Room room : dungeon.getRooms()) {
                if (!room.containsLocation(location)) continue;
                return dungeon;
            }
        }
        return null;
    }

    public Room getRoomAtLocation(Location location) {
        for (Dungeon dungeon : this.activeDungeons.values()) {
            if (!dungeon.getWorldName().equals(location.getWorld().getName())) continue;
            for (Room room : dungeon.getRooms()) {
                if (!room.containsLocation(location)) continue;
                return room;
            }
        }
        return null;
    }

    public void processPendingLocations() {
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("SELECT * FROM pending_locations LIMIT 5");
             PreparedStatement deleteStmt = conn.prepareStatement("DELETE FROM pending_locations WHERE id = ?");){
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                int id = rs.getInt("id");
                String worldName = rs.getString("world_name");
                int x = rs.getInt("x_coord");
                int z = rs.getInt("z_coord");
                World world = Bukkit.getWorld((String)worldName);
                if (world != null) {
                    if (this.arePlayersNearby(world, x, z)) continue;
                    int y = world.getHighestBlockYAt(x, z);
                    String biomeType = rs.getString("biome_type");
                    this.scheduleDungeonGeneration(world, x, z);
                }
                deleteStmt.setInt(1, id);
                deleteStmt.executeUpdate();
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to process pending locations: ", e);
        }
    }

    public void registerBossDefeat(int dungeonId, Player player) {
        Dungeon dungeon = this.getDungeon(dungeonId);
        if (dungeon == null) {
            return;
        }
        dungeon.setBossDefeated(true);
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement("UPDATE dungeons SET boss_defeated = 1 WHERE id = ?");){
            stmt.setInt(1, dungeonId);
            stmt.executeUpdate();
            World world = Bukkit.getWorld((String)dungeon.getWorldName());
            if (world != null) {
                block12: for (Player p : world.getPlayers()) {
                    for (Room room : dungeon.getRooms()) {
                        if (!room.containsLocation(p.getLocation())) continue;
                        p.sendMessage("\u00a76[AIDungeon] \u00a7e" + player.getName() + " has defeated the dungeon boss!");
                        continue block12;
                    }
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to register boss defeat: ", e);
        }
    }
}

