package com.provismet.cobblemon.gimmick.api.data.registry;

import com.cobblemon.mod.common.api.battles.model.PokemonBattle;
import com.cobblemon.mod.common.battles.dispatch.UntilDispatch;
import com.cobblemon.mod.common.entity.pokemon.PokemonEntity;
import com.cobblemon.mod.common.net.messages.client.animation.PlayPosableAnimationPacket;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import com.provismet.cobblemon.gimmick.api.data.particle.ParticleAnimation;
import com.provismet.cobblemon.gimmick.registry.GTGDynamicRegistryKeys;
import kotlin.Unit;
import net.minecraft.class_2960;
import net.minecraft.class_3414;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_6880;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Registerable object containing the data for special effects surrounding a Pokémon.
 *
 * @param particles The optional particle animation to play.
 * @param pokemonAnimation The optional animation to trigger on the Pokémon.
 * @param formChangeDelay Optional and only triggers in battle, how many seconds to wait until the form change occurs.
 * @param sound Optional sound to play when the effect is triggered.
 */
public record EffectsData (Optional<ParticleAnimation> particles, Optional<String> pokemonAnimation, Optional<Float> formChangeDelay, Optional<class_3414> sound) {
    public static final Codec<EffectsData> CODEC = RecordCodecBuilder.create(instance -> instance.group(
        ParticleAnimation.CODEC.optionalFieldOf("particles").forGetter(EffectsData::particles),
        Codec.STRING.optionalFieldOf("animation").forGetter(EffectsData::pokemonAnimation),
        Codec.FLOAT.optionalFieldOf("formDelaySeconds").forGetter(EffectsData::formChangeDelay),
        class_3414.field_41698.optionalFieldOf("sound").forGetter(EffectsData::sound)
    ).apply(instance, EffectsData::new));

    public static class_5321<EffectsData> key (class_2960 id) {
        return class_5321.method_29179(GTGDynamicRegistryKeys.EFFECTS, id);
    }

    public static Optional<class_6880.class_6883<EffectsData>> get (class_5455 registryManager, class_2960 id) {
        return registryManager.method_46759(GTGDynamicRegistryKeys.EFFECTS)
            .flatMap(registry -> registry.method_46746(EffectsData.key(id)));
    }

    public static void run (PokemonEntity pokemon, class_2960 id) {
        EffectsData.get(pokemon.method_56673(), id).ifPresent(reference -> reference.comp_349().run(pokemon));
    }

    public static void run (PokemonEntity pokemon, @Nullable PokemonEntity other, PokemonBattle battle, class_2960 id) {
        EffectsData.get(pokemon.method_56673(), id).ifPresent(reference -> reference.comp_349().run(pokemon, other, battle));
    }

    public void run (PokemonEntity pokemon) {
        this.particles.ifPresent(particleAnimation -> particleAnimation.runParticles(pokemon, null));
        this.pokemonAnimation.ifPresent(animation -> this.playAnimation(pokemon, animation));
        this.sound.ifPresent(pokemon::method_56078);
    }

    public void run (PokemonEntity pokemon, @Nullable PokemonEntity other, PokemonBattle battle) {
        this.particles.ifPresent(particleAnimation -> particleAnimation.runParticles(pokemon, other));
        this.sound.ifPresent(pokemon::method_56078);
        this.pokemonAnimation.ifPresent(animation -> battle.dispatchToFront(battle1 -> {
            this.playAnimation(pokemon, animation);
            return new UntilDispatch(() -> true);
        }));
        this.formChangeDelay.ifPresent(delay -> battle.dispatchWaitingToFront(delay, () -> Unit.INSTANCE));
    }

    private void playAnimation (PokemonEntity pokemon, String animationName) {
        PlayPosableAnimationPacket packet = new PlayPosableAnimationPacket(pokemon.method_5628(), Set.of(animationName), List.of());
        packet.sendToPlayersAround(
            pokemon.method_23317(),
            pokemon.method_23318(),
            pokemon.method_23321(),
            128,
            pokemon.method_37908().method_27983(),
            player -> false
        );
    }
}
