package fi.dy.masa.servux.util.nbt;

import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;
import java.util.UUID;

import net.minecraft.nbt.*;
import net.minecraft.util.Uuids;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec2f;
import net.minecraft.util.math.Vec3d;
import net.minecraft.util.math.Vec3i;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.mojang.serialization.DataResult;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.MapCodec;

import fi.dy.masa.servux.Servux;
import fi.dy.masa.servux.util.data.Constants;

public class NbtUtils
{
    /**
     * Get the Entity's UUID from NBT.
     *
     * @param nbt ()
     * @return ()
     */
    public static @Nullable UUID getUUIDCodec(@Nonnull NbtCompound nbt)
    {
        return getUUIDCodec(nbt, "UUID");
    }

    /**
     * Get the Entity's UUID from NBT.
     *
     * @param nbt ()
     * @param key ()
     * @return ()
     */
    public static @Nullable UUID getUUIDCodec(@Nonnull NbtCompound nbt, String key)
    {
        if (nbt.contains(key))
        {
            return nbt.get(key, Uuids.INT_STREAM_CODEC).orElse(null);
        }

        return null;
    }

    /**
     * Get the Entity's UUID from NBT.
     *
     * @param nbtIn ()
     * @param key   ()
     * @param uuid  ()
     * @return ()
     */
    public static NbtCompound putUUIDCodec(@Nonnull NbtCompound nbtIn, @Nonnull UUID uuid, String key)
    {
        nbtIn.put(key, Uuids.INT_STREAM_CODEC, uuid);
        return nbtIn;
    }

    public static @Nonnull NbtCompound putVec2fCodec(@Nonnull NbtCompound tag, @Nonnull Vec2f pos, String key)
    {
        tag.put(key, Vec2f.CODEC, pos);
        return tag;
    }

    public static @Nonnull NbtCompound putVec3iCodec(@Nonnull NbtCompound tag, @Nonnull Vec3i pos, String key)
    {
        tag.put(key, Vec3i.CODEC, pos);
        return tag;
    }

    public static @Nonnull NbtCompound putVec3dCodec(@Nonnull NbtCompound tag, @Nonnull Vec3d pos, String key)
    {
        tag.put(key, Vec3d.CODEC, pos);
        return tag;
    }

    public static @Nonnull NbtCompound putPosCodec(@Nonnull NbtCompound tag, @Nonnull BlockPos pos, String key)
    {
        tag.put(key, BlockPos.CODEC, pos);
        return tag;
    }

    public static Vec2f getVec2fCodec(@Nonnull NbtCompound tag, String key)
    {
        return tag.get(key, Vec2f.CODEC).orElse(Vec2f.ZERO);
    }

    public static Vec3i getVec3iCodec(@Nonnull NbtCompound tag, String key)
    {
        return tag.get(key, Vec3i.CODEC).orElse(Vec3i.ZERO);
    }

    public static Vec3d getVec3dCodec(@Nonnull NbtCompound tag, String key)
    {
        return tag.get(key, Vec3d.CODEC).orElse(Vec3d.ZERO);
    }

    public static BlockPos getPosCodec(@Nonnull NbtCompound tag, String key)
    {
        return tag.get(key, BlockPos.CODEC).orElse(BlockPos.ORIGIN);
    }

    @Nonnull
    public static NbtCompound writeVec3iToArray(@Nonnull Vec3i pos, @Nonnull NbtCompound tag, String tagName)
    {
        return writeBlockPosToArrayTag(pos, tag, tagName);
    }

    @Nonnull
    public static NbtCompound writeVec3iToArrayTag(@Nonnull Vec3i pos, @Nonnull NbtCompound tag, String tagName)
    {
        return writeBlockPosToArrayTag(pos, tag, tagName);
    }

    @Nonnull
    public static NbtCompound writeBlockPosToArrayTag(@Nonnull Vec3i pos, @Nonnull NbtCompound tag, String tagName)
    {
        int[] arr = new int[]{pos.getX(), pos.getY(), pos.getZ()};
        tag.putIntArray(tagName, arr);
        return tag;
    }

    @Nullable
    public static BlockPos readBlockPosFromIntArray(@Nonnull NbtCompound nbt, String key)
    {
        return readBlockPosFromArrayTag(nbt, key);
    }

    @Nullable
    public static BlockPos readBlockPosFromArrayTag(@Nonnull NbtCompound tag, String tagName)
    {
        if (tag.contains(tagName))
        {
            int[] pos = tag.getIntArray(tagName).orElse(new int[0]);

            if (pos.length == 3)
            {
                return new BlockPos(pos[0], pos[1], pos[2]);
            }
        }

        return null;
    }

    @Nullable
    public static Vec3i readVec3iFromIntArray(@Nonnull NbtCompound nbt, String key)
    {
        return readVec3iFromIntArrayTag(nbt, key);
    }

    @Nullable
    public static Vec3i readVec3iFromIntArrayTag(@Nonnull NbtCompound tag, String tagName)
    {
        if (tag.contains(tagName))
        {
            int[] pos = tag.getIntArray(tagName).orElse(new int[0]);

            if (pos.length == 3)
            {
                return new Vec3i(pos[0], pos[1], pos[2]);
            }
        }

        return null;
    }

    public static NbtCompound createBlockPosTag(Vec3i pos)
    {
        return writeBlockPosToTag(pos, new NbtCompound());
    }

    public static NbtCompound writeBlockPosToTag(Vec3i pos, NbtCompound tag)
    {
        tag.putInt("x", pos.getX());
        tag.putInt("y", pos.getY());
        tag.putInt("z", pos.getZ());
        return tag;
    }

    @Nullable
    public static BlockPos readBlockPos(@Nullable NbtCompound tag)
    {
        if (tag != null &&
            tag.contains("x") &&
            tag.contains("y") &&
            tag.contains("z"))
        {
            return new BlockPos(tag.getInt("x", 0), tag.getInt("y", 0), tag.getInt("z", 0));
        }

        return null;
    }

    public static NbtCompound writeVec3dToTag(Vec3d vec, NbtCompound tag)
    {
        tag.putDouble("dx", vec.x);
        tag.putDouble("dy", vec.y);
        tag.putDouble("dz", vec.z);
        return tag;
    }

    public static NbtCompound writeEntityPositionToTag(Vec3d pos, NbtCompound tag)
    {
        NbtList posList = new NbtList();

        posList.add(NbtDouble.of(pos.x));
        posList.add(NbtDouble.of(pos.y));
        posList.add(NbtDouble.of(pos.z));
        tag.put("Pos", posList);

        return tag;
    }

    @Nullable
    public static Vec3d readVec3d(@Nullable NbtCompound tag)
    {
        if (tag != null &&
                tag.contains("dx") &&
                tag.contains("dy") &&
                tag.contains("dz"))
        {
            return new Vec3d(tag.getDouble("dx", 0d), tag.getDouble("dy", 0d), tag.getDouble("dz", 0d));
        }

        return null;
    }

    @Nullable
    public static Vec3d readEntityPositionFromTag(@Nullable NbtCompound tag)
    {
        if (tag != null && tag.contains("Pos"))
        {
            NbtList tagList = tag.getListOrEmpty("Pos");

            if (tagList.getType() == Constants.NBT.TAG_DOUBLE && tagList.size() == 3)
            {
                return new Vec3d(tagList.getDouble(0, 0d), tagList.getDouble(1, 0d), tagList.getDouble(2, 0d));
            }
        }

        return null;
    }

    @Nullable
    public static Vec3i readVec3iFromTag(@Nullable NbtCompound tag)
    {
        if (tag != null &&
            tag.contains("x") &&
            tag.contains("y") &&
            tag.contains("z"))
        {
            return new Vec3i(tag.getInt("x", 0), tag.getInt("y", 0), tag.getInt("z", 0));
        }

        return null;
    }

    @Nullable
    public static NbtCompound readNbtFromFileAsPath(@Nonnull Path file)
    {
        return readNbtFromFileAsPath(file, NbtSizeTracker.ofUnlimitedBytes());
    }

    @Nullable
    public static NbtCompound readNbtFromFileAsPath(@Nonnull Path file, NbtSizeTracker tracker)
    {
        if (!Files.exists(file) || !Files.isReadable(file))
        {
            return null;
        }

        try
        {
            return NbtIo.readCompressed(Files.newInputStream(file), tracker);
        }
        catch (Exception e)
        {
            Servux.LOGGER.warn("readNbtFromFileAsPath: Failed to read NBT data from file '{}'", file.toString());
        }

        return null;
    }

    /**
     * Write the compound tag, gzipped, to the output stream.
     */
    public static void writeCompressed(@Nonnull NbtCompound tag, @Nonnull OutputStream outputStream)
    {
        try
        {
            NbtIo.writeCompressed(tag, outputStream);
        }
        catch (Exception err)
        {
            Servux.LOGGER.warn("writeCompressed: Failed to write NBT data to output stream");
        }
    }

    public static void writeCompressed(@Nonnull NbtCompound tag, @Nonnull Path file)
    {
        try
        {
            NbtIo.writeCompressed(tag, file);
        }
        catch (Exception err)
        {
            Servux.LOGGER.warn("writeCompressed: Failed to write NBT data to file");
        }
    }

    /**
     * Reads in a Flat Map from NBT -- this way we don't need Mojang's code complexity
     * @param <T> ()
     * @param nbt ()
     * @param mapCodec ()
     * @return ()
     */
    public static <T> Optional<T> readFlatMap(@Nonnull NbtCompound nbt, MapCodec<T> mapCodec)
    {
        DynamicOps<NbtElement> ops = NbtOps.INSTANCE;

        return switch (ops.getMap(nbt).flatMap(map -> mapCodec.decode(ops, map)))
        {
            case DataResult.Success<T> result -> Optional.of(result.value());
            case DataResult.Error<T> error -> error.partialValue();
            default -> Optional.empty();
        };
    }

    /**
     * Writes a Flat Map to NBT -- this way we don't need Mojang's code complexity
     * @param <T> ()
     * @param mapCodec ()
     * @param value ()
     * @return ()
     */
    public static <T> NbtCompound writeFlatMap(MapCodec<T> mapCodec, T value)
    {
        DynamicOps<NbtElement> ops = NbtOps.INSTANCE;
        NbtCompound nbt = new NbtCompound();

        switch (mapCodec.encoder().encodeStart(ops, value))
        {
            case DataResult.Success<NbtElement> result -> nbt.copyFrom((NbtCompound) result.value());
            case DataResult.Error<NbtElement> error -> error.partialValue().ifPresent(partial -> nbt.copyFrom((NbtCompound) partial));
        }

        return nbt;
    }
}
