package uk.co.cablepost.ftech_robots.buildInstructions;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.jetbrains.annotations.Nullable;
import uk.co.cablepost.ftech_robots.models.BlockPosAndState;

import java.util.*;
import java.util.function.Predicate;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2328;
import net.minecraft.class_2338;
import net.minecraft.class_2346;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2680;
import net.minecraft.class_3218;

public class BuildInstructions {
    private final class_1799 _itemStack;
    private @Nullable UUID _instructionsUuid = null;
    private @Nullable List<BlockPosAndState> _instructions = null;
    private @Nullable Integer _startingSize = null;

    public BuildInstructions(class_1799 itemStack, class_1937 world){
        _itemStack = itemStack;
        readInstructionsNbt(world);
    }

    // https://fabric.moddedmc.wiki/misc-topics/codecs
    public static final Codec<BlockPosAndState> BLOCK_POS_AND_STATE_CODEC = RecordCodecBuilder.create(instance -> instance.group(
        class_2338.field_25064.fieldOf("pos").forGetter(BlockPosAndState::getBlockPos),
        class_2680.field_24734.fieldOf("state").forGetter(BlockPosAndState::getBlockState)
        //Identifier.CODEC.fieldOf("block").forGetter(BlockPosAndState::getBlockIdentifier)
    ).apply(instance, BlockPosAndState::new));

    private void readInstructionsNbt(class_1937 world){
        class_2487 nbtCompound = _itemStack.method_7948();

        if(!nbtCompound.method_10545("instructions_uuid")){
            _instructions = null;
            return;
        }

        _instructionsUuid = nbtCompound.method_25926("instructions_uuid");
        _startingSize = nbtCompound.method_10550("instructions_starting_size");

        @Nullable class_2520 loadedInstructions = null;

        if(world instanceof class_3218 serverWorld) {
            loadedInstructions = BuildInstructionsFileUtils.load(serverWorld, nbtCompound.method_25926("instructions_uuid").toString());
        }

        if(loadedInstructions == null){
            _instructions = null;
            return;
        }

        //NbtElement instructionsNbt = nbtCompound.get("instructions");
        Codec<List<BlockPosAndState>> listCodec = BLOCK_POS_AND_STATE_CODEC.listOf();
        Optional<List<BlockPosAndState>> instructions = listCodec.parse(class_2509.field_11560, loadedInstructions).result();
        if(instructions.isEmpty()){
            throw new RuntimeException("Failed to read instructions from NBT");
        }

        _instructions = new ArrayList<>(instructions.get());
    }

    public void writeInstructionsNbt(class_1937 world, boolean saveToFile){
        if(_instructions == null){
            wipeInstructionsAndInstructionsNbt(world, saveToFile);
            return;
        }

        assert _startingSize != null;
        assert _instructionsUuid != null;

        class_2487 nbtCompound = _itemStack.method_7948();

        Codec<List<BlockPosAndState>> listCodec = BLOCK_POS_AND_STATE_CODEC.listOf();
        Optional<class_2520> instructionsNbt = listCodec.encodeStart(class_2509.field_11560, _instructions).result();
        if(instructionsNbt.isEmpty()){
            throw new RuntimeException("Failed to write instructions to NBT");
        }

        if(saveToFile && world instanceof class_3218 serverWorld) {
            BuildInstructionsFileUtils.save(serverWorld, _instructionsUuid.toString(), instructionsNbt.get());
        }
        //nbtCompound.put("instructions", instructionsNbt.get());

        nbtCompound.method_10569("instructions_starting_size", _startingSize);
        nbtCompound.method_10569("instructions_hash", instructionsNbt.get().hashCode());
        nbtCompound.method_25927("instructions_uuid", _instructionsUuid);
    }

    public void removeCompletedInstructions(class_1937 world){
        if(_instructions == null){
            return;
        }

        _instructions.removeIf((x) -> {
            if(!world.method_24794(x.blockPos)){
                return true;
            }

            class_2680 worldBlockState = world.method_8320(x.blockPos);

            if(worldBlockState.method_26204() instanceof class_2328){
                return true;
            }

            if(!worldBlockState.method_26215() && worldBlockState.method_26204().method_36555() == -1){
                return true;
            }

            return  (x.blockState.method_26215() && worldBlockState.method_26215()) ||
                (x.blockState.method_26215() && worldBlockState.method_45474()) ||
                Objects.equals(x.blockState.method_26204().method_9539(), worldBlockState.method_26204().method_9539()) ||
                (Objects.equals(worldBlockState.method_26204().method_9539(), "block.minecraft.dirt") && Objects.equals(x.blockState.method_26204().method_9539(), "block.minecraft.grass_block")) ||
                (Objects.equals(worldBlockState.method_26204().method_9539(), "block.minecraft.grass_block") && Objects.equals(x.blockState.method_26204().method_9539(), "block.minecraft.dirt"))
            ;
        });
    }

    private Optional<BlockPosAndState> findFirstInInstructions(Predicate<BlockPosAndState> predicate){
        if(_instructions == null){
            return Optional.empty();
        }

        for(BlockPosAndState blockPosAndState : _instructions){
            if(predicate.test(blockPosAndState)){
                return Optional.of(blockPosAndState);
            }
        }

        return Optional.empty();
    }

    public @Nullable BlockPosAndState getNextBlockToChange(class_1937 world, @Nullable class_1792 ofItem){
        if(_instructions == null){
            return null;
        }

        // Block to break
        if(ofItem == null || ofItem == class_1802.field_8162){
            Optional<BlockPosAndState> blockThatNeedsBreaking = findFirstInInstructions((x) ->
                !world.method_8320(x.blockPos).method_26215() &&
                !world.method_8320(x.blockPos).method_45474() &&
                x.blockState.method_26215()
            );

            if (blockThatNeedsBreaking.isPresent()) {
                return new BlockPosAndState(blockThatNeedsBreaking.get().blockPos, class_2246.field_10124.method_9564());
            }
        }

        // Block to place adjacent to other block
        if(ofItem == null || ofItem != class_1802.field_8162){
            Optional<BlockPosAndState> blockThatNeedsPlacingNextToOtherBlock = findFirstInInstructions((x) ->
                (ofItem == null || x.blockState.method_26204().method_8389().equals(ofItem)) &&
                (
                    (x.blockState.method_26204() instanceof class_2346 && !world.method_8320(x.blockPos.method_10074()).method_45474()) ||
                    !(x.blockState.method_26204() instanceof class_2346)
                ) &&
                (x.blockState.method_26204().method_9558(x.blockState.method_26204().method_9564(), world, x.blockPos)) &&
                !Objects.equals(x.blockState.method_26204().method_9539(), world.method_8320(x.blockPos).method_26204().method_9539()) &&
                (
                    world.method_8320(x.blockPos.method_10084()).method_51367() ||
                    world.method_8320(x.blockPos.method_10074()).method_51367() ||
                    world.method_8320(x.blockPos.method_10095()).method_51367() ||
                    world.method_8320(x.blockPos.method_10078()).method_51367() ||
                    world.method_8320(x.blockPos.method_10072()).method_51367() ||
                    world.method_8320(x.blockPos.method_10067()).method_51367()
                )
            );

            if (blockThatNeedsPlacingNextToOtherBlock.isPresent()) {
                return blockThatNeedsPlacingNextToOtherBlock.map(blockPosAndState -> new BlockPosAndState(blockPosAndState.blockPos, blockPosAndState.blockState)).orElse(null);
            }
        }

        // Block to place
        if(ofItem == null || ofItem != class_1802.field_8162){
            Optional<BlockPosAndState> blockThatNeedsPlacing = findFirstInInstructions((x) ->
                (ofItem == null || x.blockState.method_26204().method_8389().equals(ofItem)) &&
                (
                    (x.blockState.method_26204() instanceof class_2346 && !world.method_8320(x.blockPos.method_10074()).method_45474()) ||
                    !(x.blockState.method_26204() instanceof class_2346)
                ) &&
                (x.blockState.method_26204().method_9558(x.blockState.method_26204().method_9564(), world, x.blockPos)) &&
                !Objects.equals(x.blockState.method_26204().method_9539(), world.method_8320(x.blockPos).method_26204().method_9539())
            );

            return blockThatNeedsPlacing.map(blockPosAndState -> new BlockPosAndState(blockPosAndState.blockPos, blockPosAndState.blockState)).orElse(null);
        }

        return null;
    }

    public @Nullable List<BlockPosAndState> getInstructions(){
        if(_instructions == null){
            return null;
        }

        return _instructions;
    }

    public @Nullable Integer getBlocksToChangeSize(){
        if(_instructions == null){
            return null;
        }

        return _instructions.size();
    }

    public @Nullable Integer getStartingSize(){
        return _startingSize;
    }

    public void removeBlockToChange(class_2338 blockPos, class_2680 blockState){
        if(_instructions == null){
            return;
        }

        _instructions.removeIf((x) ->
            x.blockPos.method_19455(blockPos) == 0 &&
            x.blockState.hashCode() == blockState.hashCode()
        );

        if(_instructions.isEmpty()){
            _instructions = null;
        }
    }

    public void setInstructionsAndWrite(ArrayList<BlockPosAndState> instructions, class_1937 world, class_2338 startingPos){
        if(_instructions != null){
            wipeInstructionsAndInstructionsNbt(world, true);
        }

        assert instructions != null;

        _instructionsUuid = UUID.randomUUID();
        _instructions = instructions;
        addBreakInstructionsToStart(world);
        sortInstructions(startingPos);
        removeCompletedInstructions(world);
        _startingSize = _instructions.size();

        writeInstructionsNbt(world, true);
    }

    private void addBreakInstructionsToStart(class_1937 world){
        if(_instructions == null){
            return;
        }

        for(BlockPosAndState instruction : new ArrayList<>(_instructions)){
            class_2680 worldBlockState = world.method_8320(instruction.blockPos);
            if(
                !worldBlockState.method_26215() &&
                !worldBlockState.method_45474() &&
                !Objects.equals(instruction.blockState.method_26204().method_9539(), worldBlockState.method_26204().method_9539()) &&
                !(
                    (
                        Objects.equals(worldBlockState.method_26204().method_9539(), "block.minecraft.dirt") &&
                        Objects.equals(instruction.blockState.method_26204().method_9539(), "block.minecraft.grass_block")
                    ) ||
                    (
                        Objects.equals(instruction.blockState.method_26204().method_9539(), "block.minecraft.dirt") &&
                        Objects.equals(worldBlockState.method_26204().method_9539(), "block.minecraft.grass_block")
                    )
                )
            ){
                _instructions.add(0, new BlockPosAndState(
                    instruction.blockPos,
                    class_2246.field_10124.method_9564()
                ));
            }
        }
    }

    private void sortInstructions(class_2338 startingPos){
        assert _instructions != null;
        _instructions.sort((a, b) -> {
            // Both break / place and both same Y level
            if(a.blockState.method_26215() == b.blockState.method_26215() && a.blockPos.method_10264() == b.blockPos.method_10264()){
                if(a.blockState.method_26215() && b.blockState.method_26215()){
                    int aDis = startingPos.method_19455(a.blockPos);
                    int bDis = startingPos.method_19455(b.blockPos);
                    if(aDis < bDis){
                        return -1;
                    }
                    if(bDis < aDis){
                        return 1;
                    }
                }

                return 0;
            }

            // One is to place, other is to remove
            if(a.blockState.method_26215() && !b.blockState.method_26215()){
                return -1;
            }

            if(!a.blockState.method_26215() && b.blockState.method_26215()){
                return 1;
            }

            // Both blocks to remove, start top down
            if(a.blockState.method_26215() && b.blockState.method_26215()){
                if(a.blockPos.method_10264() > b.blockPos.method_10264()){
                    return -1;
                }
                return 1;
            }

            // Both blocks to place, start bottom up
            if(a.blockPos.method_10264() > b.blockPos.method_10264()){
                return 1;
            }

            return -1;
        });
    }

    public void wipeInstructionsAndInstructionsNbt(class_1937 world, boolean saveToFile){
        _instructions = null;
        class_2487 nbtCompound = _itemStack.method_7948();

        //nbtCompound.remove("instructions");
        if(saveToFile && nbtCompound.method_10545("instructions_uuid") && world instanceof class_3218 serverWorld) {
            BuildInstructionsFileUtils.delete(serverWorld, nbtCompound.method_25926("instructions_uuid").toString());
        }

        nbtCompound.method_10551("instructions_starting_size");
        nbtCompound.method_10551("instructions_hash");
    }
}
