/*
 * Decompiled with CFR 0.152.
 */
package it.renvins.serverpulse.common;

import it.renvins.serverpulse.api.service.IDatabaseService;
import it.renvins.serverpulse.common.config.DatabaseConfiguration;
import it.renvins.serverpulse.common.logger.PulseLogger;
import it.renvins.serverpulse.common.platform.Platform;
import it.renvins.serverpulse.common.scheduler.Task;
import it.renvins.serverpulse.common.scheduler.TaskScheduler;
import java.net.ConnectException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;

public class DatabaseService
implements IDatabaseService {
    private final PulseLogger logger;
    private final Platform platform;
    private final DatabaseConfiguration configuration;
    private final TaskScheduler scheduler;
    private HttpClient httpClient;
    private volatile Task retryTask;
    private final int MAX_RETRIES = 5;
    private final long RETRY_DELAY_TICKS = 600L;
    private volatile boolean isConnected = false;
    private volatile int retryCount = 0;
    private String pingUrl;
    private String writeUrl;

    public DatabaseService(PulseLogger logger, Platform platform, DatabaseConfiguration configuration, TaskScheduler scheduler) {
        this.logger = logger;
        this.platform = platform;
        this.configuration = configuration;
        this.scheduler = scheduler;
        this.httpClient = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(10L)).build();
    }

    @Override
    public void load() {
        if (!this.checkConnectionData()) {
            this.logger.error("InfluxDB connection data is missing or invalid. Shutting down...");
            this.platform.disable();
            return;
        }
        this.logger.info("Connecting to InfluxDB...");
        Object baseUrl = this.configuration.getHost();
        if (!((String)baseUrl).endsWith("/")) {
            baseUrl = (String)baseUrl + "/";
        }
        this.pingUrl = (String)baseUrl + "ping";
        this.writeUrl = (String)baseUrl + "api/v2/write?org=" + this.configuration.getOrg() + "&bucket=" + this.configuration.getBucket() + "&precision=ns";
        this.scheduler.runAsync(this::connect);
    }

    @Override
    public void unload() {
        this.logger.info("Unloading InfluxDB connection...");
        this.stopRetryTask();
        this.disconnect();
        if (this.httpClient != null) {
            this.httpClient.close();
        }
    }

    @Override
    public CompletableFuture<Boolean> writeLineProtocol(String lineProtocol) {
        if (!this.isConnected) {
            return CompletableFuture.completedFuture(false);
        }
        try {
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(this.writeUrl)).header("Authorization", "Token " + this.configuration.getToken()).header("Content-Type", "text/plain; charset=utf-8").POST(HttpRequest.BodyPublishers.ofString(lineProtocol, StandardCharsets.UTF_8)).timeout(Duration.ofSeconds(10L)).build();
            return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).thenApply(response -> {
                if (response.statusCode() == 204) {
                    return true;
                }
                this.logger.error("Failed to write to InfluxDB: " + response.statusCode() + " - " + (String)response.body());
                return false;
            })).exceptionally(throwable -> {
                this.logger.error("Error writing to InfluxDB: " + throwable.getMessage());
                if (throwable.getCause() instanceof ConnectException) {
                    this.isConnected = false;
                    this.startRetryTaskIfNeeded();
                }
                return false;
            });
        }
        catch (Exception e) {
            this.logger.error("Failed to create write request: " + e.getMessage());
            return CompletableFuture.completedFuture(false);
        }
    }

    @Override
    public boolean ping() {
        try {
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(this.pingUrl)).GET().timeout(Duration.ofSeconds(5L)).build();
            HttpResponse<Void> response = this.httpClient.send(request, HttpResponse.BodyHandlers.discarding());
            return response.statusCode() == 204;
        }
        catch (Exception e) {
            this.logger.warning("InfluxDB ping failed: " + e.getMessage());
            return false;
        }
    }

    @Override
    public void disconnect() {
        this.isConnected = false;
    }

    @Override
    public boolean isConnected() {
        return this.isConnected;
    }

    @Override
    public synchronized void startRetryTaskIfNeeded() {
        if (this.retryTask != null && !this.retryTask.isCancelled()) {
            return;
        }
        if (!this.platform.isEnabled()) {
            this.logger.warning("Plugin disabling, not starting retry task...");
            return;
        }
        this.retryCount = 0;
        this.logger.warning("Connection failed. Starting connection retry task (Max 5 attempts)...");
        this.retryTask = this.scheduler.runTaskTimerAsync(() -> {
            if (this.isConnected) {
                this.logger.info("Connection successful, stopping retry task...");
                this.stopRetryTask();
                return;
            }
            if (!this.platform.isEnabled()) {
                this.logger.warning("Plugin disabled during retry task execution...");
                this.stopRetryTask();
                return;
            }
            if (this.retryCount >= 5) {
                this.logger.error("Max connection retries (5) reached. Disabling ServerPulse metrics...");
                this.stopRetryTask();
                this.disconnect();
                this.scheduler.runSync(this.platform::disable);
                return;
            }
            ++this.retryCount;
            this.logger.info("Retrying InfluxDB connection... Attempt " + this.retryCount + "/5");
            this.connect();
        }, 600L, 600L);
    }

    private void connect() {
        this.disconnect();
        try {
            this.logger.info("Attempting to connect to InfluxDB via HTTP API...");
            boolean isPingSuccessful = this.ping();
            if (isPingSuccessful) {
                this.isConnected = true;
                this.retryCount = 0;
                this.stopRetryTask();
                this.logger.info("Successfully connected to InfluxDB and ping successful...");
            } else {
                this.logger.warning("Ping failed. Will retry...");
                this.isConnected = false;
                this.startRetryTaskIfNeeded();
            }
        }
        catch (Exception e) {
            this.logger.error("Failed to connect or ping InfluxDB: " + e.getMessage());
            this.isConnected = false;
            this.startRetryTaskIfNeeded();
        }
    }

    private synchronized void stopRetryTask() {
        if (this.retryTask != null) {
            if (!this.retryTask.isCancelled()) {
                try {
                    this.retryTask.cancel();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            this.retryTask = null;
        }
    }

    private boolean checkConnectionData() {
        String url = this.configuration.getHost();
        String token = this.configuration.getToken();
        String org = this.configuration.getOrg();
        String bucket = this.configuration.getBucket();
        boolean valid = true;
        if (url == null || url.isEmpty()) {
            this.logger.error("Missing or empty 'metrics.influxdb.url' in config...");
            valid = false;
        }
        if (bucket == null || bucket.isEmpty()) {
            this.logger.error("Missing or empty 'metrics.influxdb.bucket' in config...");
            valid = false;
        }
        if (org == null || org.isEmpty()) {
            this.logger.error("Missing or empty 'metrics.influxdb.org' in config...");
            valid = false;
        }
        if (token == null || token.isEmpty() || token.equals("my-token")) {
            this.logger.error("Missing, empty, or default 'metrics.influxdb.token' in config...");
            valid = false;
        }
        return valid;
    }
}

