package net.wizardsoflua.lua.nbt;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Objects.requireNonNull;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import org.jetbrains.annotations.Nullable;
import net.minecraft.class_2479;
import net.minecraft.class_2481;
import net.minecraft.class_2487;
import net.minecraft.class_2489;
import net.minecraft.class_2494;
import net.minecraft.class_2495;
import net.minecraft.class_2497;
import net.minecraft.class_2499;
import net.minecraft.class_2501;
import net.minecraft.class_2503;
import net.minecraft.class_2514;
import net.minecraft.class_2516;
import net.minecraft.class_2519;
import net.minecraft.class_2520;
import net.sandius.rembulan.ByteString;
import net.sandius.rembulan.ConversionException;
import net.sandius.rembulan.Conversions;
import net.sandius.rembulan.Table;
import net.sandius.rembulan.impl.DefaultTable;
import net.wizardsoflua.extension.spell.api.resource.LuaTypes;
import net.wizardsoflua.lua.table.TableIterable;

public class NbtConverter {
  private static final String DEFAULT_PATH = "nbt";
  private final LuaTypes types;
  private @Nullable Map<Class<? extends class_2520>, NbtMerger<? extends class_2520>> mergers;

  public NbtConverter(LuaTypes types) {
    this.types = requireNonNull(types, "types");
  }

  private LuaTypes getTypes() {
    return types;
  }

  private Map<Class<? extends class_2520>, NbtMerger<? extends class_2520>> getMergers() {
    if (mergers == null) {
      mergers = new HashMap<>();
      registerMerger(class_2481.class, new NbtByteMerger(this));
      registerMerger(class_2487.class, new NbtCompoundMerger(this));
      registerMerger(class_2489.class, new NbtDoubleMerger(this));
      registerMerger(class_2494.class, new NbtFloatMerger(this));
      registerMerger(class_2497.class, new NbtIntMerger(this));
      registerMerger(class_2499.class, new NbtListMerger(this));
      registerMerger(class_2503.class, new NbtLongMerger(this));
      registerMerger(class_2516.class, new NbtShortMerger(this));
      registerMerger(class_2519.class, new NbtStringMerger(this));
      registerMerger(class_2479.class, new NbtByteArrayMerger(this));
      registerMerger(class_2495.class, new NbtIntArrayMerger(this));
    }
    return mergers;
  }

  private <NBT extends class_2520> void registerMerger(Class<NBT> cls, NbtMerger<NBT> merger) {
    if (getMergers().containsKey(cls)) {
      throw new IllegalArgumentException("Duplicate merger for " + cls);
    }
    getMergers().put(cls, merger);
  }

  private <NBT extends class_2520> NbtMerger<NBT> getMerger(Class<NBT> nbtType) {
    @SuppressWarnings("unchecked")
    NbtMerger<NBT> result = (NbtMerger<NBT>) getMergers().get(nbtType);
    checkArgument(result != null, "Unsupported NBT type", nbtType.getSimpleName());
    return result;
  }

  private String keyToString(Object luaKey, int index, String path) {
    ByteString result = Conversions.stringValueOf(luaKey);
    if (result == null) {
      String actualType = getTypes().getLuaTypeNameOfLuaObject(luaKey);
      throw new ConversionException("Can't convert key " + index + " in " + path
          + "! string/number expected, but got " + actualType);
    }
    return result.toString();
  }

  ConversionException conversionException(String path, Object actual, String expected) {
    String actualType = getTypes().getLuaTypeNameOfLuaObject(actual);
    return new ConversionException(
        "Can't convert " + path + "! " + expected + " expected, but got " + actualType);
  }

  public class_2487 merge(class_2487 nbt, Table data) {
    return merge(nbt, data, DEFAULT_PATH);
  }

  class_2487 merge(class_2487 nbt, Table data, String path) {
    class_2487 result = nbt.method_10553();
    insert(result, data, path);
    return result;
  }

  public void insert(class_2487 nbt, Table data) {
    insert(nbt, data, DEFAULT_PATH);
  }

  void insert(class_2487 nbt, Table data, String path) {
    int i = 0;
    for (Entry<Object, Object> entry : new TableIterable(data)) {
      String key = keyToString(entry.getKey(), ++i, path);
      Object newLuaValue = entry.getValue();

      String entryPath = path + "." + key;
      class_2520 oldNbtValue = nbt.method_10580(key);
      class_2520 newNbtValue;
      if (oldNbtValue != null) {
        newNbtValue = merge(oldNbtValue, newLuaValue, key, entryPath);
      } else {
        newNbtValue = toNbt(newLuaValue, entryPath);
      }
      nbt.method_10566(key, newNbtValue);
    }
  }

  <NBT extends class_2520> NBT merge(NBT nbt, Object data, String key, String path) {
    checkNotNull(key, "key == null!");
    checkNotNull(nbt, "nbt == null!");
    checkNotNull(data, "data == null!");
    @SuppressWarnings("unchecked")
    Class<NBT> nbtType = (Class<NBT>) nbt.getClass();
    NbtMerger<NBT> converter = getMerger(nbtType);
    return converter.merge(nbt, data, key, path);
  }

  public static Object toLua(class_2520 nbt) {
    checkNotNull(nbt, "nbt == null!");
    if (nbt instanceof class_2514) {
      return toLua((class_2514) nbt);
    }
    if (nbt instanceof class_2519) {
      return toLua((class_2519) nbt);
    }
    if (nbt instanceof class_2499) {
      return toLua((class_2499) nbt);
    }
    if (nbt instanceof class_2487) {
      return toLua((class_2487) nbt);
    }
    if (nbt instanceof class_2501) {
      return toLua((class_2501) nbt);
    }
    if (nbt instanceof class_2495) {
      return toLua((class_2495) nbt);
    }
    if (nbt instanceof class_2479) {
      return toLua((class_2479) nbt);
    }
    throw new IllegalArgumentException(
        "Unsupported NBT type for conversion: " + nbt.getClass().getName());
  }

  public static Number toLua(class_2514 nbt) {
    checkNotNull(nbt, "nbt == null!");
    if (nbt instanceof class_2489) {
      return nbt.method_10697();
    }
    if (nbt instanceof class_2494) {
      return nbt.method_10697();
    }
    return nbt.method_10699();
  }

  public static ByteString toLua(class_2519 nbt) {
    checkNotNull(nbt, "nbt == null!");
    return nbt.method_68658().map(ByteString::of).orElseThrow();
  }

  public static Table toLua(class_2499 nbt) {
    checkNotNull(nbt, "nbt == null!");
    Table result = new DefaultTable();
    int luaKey = 1;
    for (class_2520 nbtValue : nbt) {
      Object luaValue = toLua(nbtValue);
      result.rawset(luaKey++, luaValue);
    }
    return result;
  }

  public static Table toLua(class_2487 nbt) {
    checkNotNull(nbt, "nbt == null!");
    Table result = new DefaultTable();
    for (String key : nbt.method_10541()) {
      class_2520 nbtValue = nbt.method_10580(key);
      Object luaValue = toLua(nbtValue);
      result.rawset(key, luaValue);
    }
    return result;
  }

  public static Table toLua(class_2501 nbt) {
    checkNotNull(nbt, "nbt == null!");
    Table result = new DefaultTable();
    long[] arr = nbt.method_10615();
    for (int i = 0; i < arr.length; ++i) {
      long key = i + 1;
      Object value = arr[i];
      result.rawset(key, value);
    }
    return result;
  }

  public static Table toLua(class_2495 nbt) {
    checkNotNull(nbt, "nbt == null!");
    Table result = new DefaultTable();
    int[] arr = nbt.method_10588();
    for (int i = 0; i < arr.length; ++i) {
      long key = i + 1;
      Object value = arr[i];
      result.rawset(key, value);
    }
    return result;
  }

  public static Table toLua(class_2479 nbt) {
    checkNotNull(nbt, "nbt == null!");
    Table result = new DefaultTable();
    byte[] arr = nbt.method_10521();
    for (int i = 0; i < arr.length; ++i) {
      long key = i + 1;
      Object value = arr[i];
      result.rawset(key, value);
    }
    return result;
  }

  public class_2520 toNbt(Object data) {
    return toNbt(data, DEFAULT_PATH);
  }

  private class_2520 toNbt(Object data, String path) {
    checkNotNull(data, "data == null!");
    data = adjustType(data, path);
    if (data instanceof Boolean) {
      return toNbt((Boolean) data);
    }
    if (data instanceof Byte) {
      return toNbt((Byte) data);
    }
    if (data instanceof ByteString) {
      return toNbt((ByteString) data);
    }
    if (data instanceof Double) {
      return toNbt((Double) data);
    }
    if (data instanceof Float) {
      return toNbt((Float) data);
    }
    if (data instanceof Integer) {
      return toNbt((Integer) data);
    }
    if (data instanceof Long) {
      return toNbt((Long) data);
    }
    if (data instanceof Short) {
      return toNbt((Short) data);
    }
    if (data instanceof String) {
      return toNbt((String) data);
    }
    if (data instanceof Table) {
      return toNbt((Table) data, path);
    }
    throw new IllegalArgumentException(
        "Unsupported type for NBT conversion: " + data.getClass().getName());
  }

  /**
   * Workaround for <a href="https://bugs.mojang.com/browse/MC-112257">MC-112257</a>.
   */
  private Object adjustType(Object data, String path) {
    switch (path) {
      case DEFAULT_PATH + ".tag.RepairCost":
        data = toIntIfPossible(data);
        break;
    }
    return data;
  }

  /**
   * Convert {@code data} to an {@code int} if possible.
   *
   * @param data
   * @return an {@code int} or {@code data}
   */
  private static Object toIntIfPossible(Object data) {
    if (data instanceof Number) {
      Number number = (Number) data;
      int result = number.intValue();
      if (result == number.doubleValue()) {
        return result;
      }
    }
    return data;
  }

  public static class_2481 toNbt(boolean data) {
    return toNbt(data ? (byte) 1 : (byte) 0);
  }

  public static class_2481 toNbt(Boolean data) {
    return toNbt(data.booleanValue());
  }

  public static class_2481 toNbt(byte data) {
    return class_2481.method_23233(data);
  }

  public static class_2481 toNbt(Byte data) {
    return toNbt(data.byteValue());
  }

  public static class_2519 toNbt(ByteString data) {
    return toNbt(data.toString());
  }

  public static class_2489 toNbt(double data) {
    return class_2489.method_23241(data);
  }

  public static class_2489 toNbt(Double data) {
    return toNbt(data.doubleValue());
  }

  public static class_2494 toNbt(float data) {
    return class_2494.method_23244(data);
  }

  public static class_2494 toNbt(Float data) {
    return toNbt(data.floatValue());
  }

  public static class_2497 toNbt(int data) {
    return class_2497.method_23247(data);
  }

  public static class_2497 toNbt(Integer data) {
    return toNbt(data.intValue());
  }

  public static class_2503 toNbt(long data) {
    return class_2503.method_23251(data);
  }

  public static class_2503 toNbt(Long data) {
    return toNbt(data.longValue());
  }

  public static class_2516 toNbt(short data) {
    return class_2516.method_23254(data);
  }

  public static class_2516 toNbt(Short data) {
    return toNbt(data.shortValue());
  }

  public static class_2519 toNbt(String data) {
    return class_2519.method_23256(data);
  }

  public class_2520 toNbt(Table data) {
    return toNbt(data, DEFAULT_PATH);
  }

  private class_2520 toNbt(Table data, String path) {
    Table table = data;
    if (isArray(table)) {
      return toNbtList(table, path);
    } else {
      return toNbtCompound(table, path);
    }
  }

  /**
   * Try to guess if the given table is an array.
   *
   * @param table
   * @return true if we guess it's an array
   */
  public static boolean isArray(Table table) {
    long count = 0;
    for (Map.Entry<Object, Object> entry : new TableIterable(table)) {
      count++;
      Object key = entry.getKey();
      Long intValue = Conversions.integerValueOf(key);
      if (intValue == null) {
        return false;
      }
      if (intValue.longValue() != count) {
        return false;
      }
    }
    return count > 0;
  }

  public class_2499 toNbtList(Table data) {
    return toNbtList(data, DEFAULT_PATH);
  }

  class_2499 toNbtList(Table data, String path) {
    class_2499 result = new class_2499();
    int i = 0;
    for (Entry<Object, Object> entry : new TableIterable(data)) {
      Object value = entry.getValue();
      class_2520 nbtValue = toNbt(value, path + '[' + (++i) + ']');
      result.add(nbtValue);
    }
    return result;
  }

  public class_2487 toNbtCompound(Table data) {
    return toNbtCompound(data, DEFAULT_PATH);
  }

  class_2487 toNbtCompound(Table data, String path) {
    class_2487 result = new class_2487();
    int i = 0;
    for (Entry<Object, Object> entry : new TableIterable(data)) {
      String key = keyToString(entry.getKey(), ++i, path);
      Object value = entry.getValue();
      class_2520 nbtValue = toNbt(value, path + '.' + key);
      result.method_10566(key, nbtValue);
    }
    return result;
  }
}
