package net.wizardsoflua.extract;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Objects;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.minecraft.class_2168;
import net.minecraft.class_2561;
import net.minecraft.class_5250;
import net.minecraft.class_8828;
import net.wizardsoflua.WizardsOfLuaMod;

public class LuaApiExtractor {

  public int run(class_2168 source, Path outputDir) throws CommandSyntaxException {
    try {
      if (extractFiles(source, outputDir)) {
        return 1;
      } ;
    } catch (IOException | URISyntaxException e) {
      handleException(e, source);
    }
    return 0;
  }

  private boolean extractFiles(class_2168 source, Path outputDir)
      throws IOException, URISyntaxException {
    deleteFilesInDirectory(outputDir);

    Path sourcePath = Paths.get(Objects
        .requireNonNull(LuaApiExtractor.class.getProtectionDomain().getCodeSource().getLocation())
        .toURI());

    if (Files.isDirectory(sourcePath)) {
      // Running from IDE (e.g., Eclipse or IntelliJ)
      Path devLuaDir = findProjectRoot().resolve("src/api/lua");
      if (Files.exists(devLuaDir)) {
        copyDirectory(devLuaDir, outputDir);
        source.method_9226(() -> class_2561.method_30163("Lua API files copied from source dir: "
            + devLuaDir.toAbsolutePath() + " to: " + outputDir.toAbsolutePath()), false);
        return true;
      } else {
        class_2561 errorText = class_2561.method_30163("Lua API source directory not found: " + devLuaDir.toAbsolutePath()
            + " to: " + outputDir.toAbsolutePath());
        source.method_9213(errorText);
        return false;
      }
    } else {
      // Running from JAR - extract from nested zip file
      try (ZipFile jar = new ZipFile(sourcePath.toFile())) {
        ZipEntry zipEntry = jar.getEntry("api.zip");
        if (zipEntry == null) {
          class_2561 errorText = class_2561.method_30163("api.zip not found in JAR file: " + sourcePath.toAbsolutePath());
          source.method_9213(errorText);
          return false;
        }

        // Create a temporary file to extract the nested zip
        Path tempZipFile = Files.createTempFile("api", ".zip");
        try {
          // Extract the nested zip file to temporary location
          try (InputStream zipStream = jar.getInputStream(zipEntry);
              OutputStream tempOut = Files.newOutputStream(tempZipFile, StandardOpenOption.CREATE,
                  StandardOpenOption.TRUNCATE_EXISTING)) {
            zipStream.transferTo(tempOut);
          }

          // Now extract from the temporary zip file
          try (ZipFile zipFile = new ZipFile(tempZipFile.toFile())) {
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
              ZipEntry entry = entries.nextElement();
              if (entry.isDirectory())
                continue;

              Path destFile = outputDir.resolve(entry.getName());
              Files.createDirectories(destFile.getParent());

              try (InputStream in = zipFile.getInputStream(entry);
                  OutputStream out = Files.newOutputStream(destFile, StandardOpenOption.CREATE,
                      StandardOpenOption.TRUNCATE_EXISTING)) {
                in.transferTo(out);
              }
            }
          }

          source.method_9226(
              () -> class_2561.method_30163("Api files extracted from JAR to: " + outputDir.toAbsolutePath()),
              false);
          return true;

        } finally {
          // Clean up temporary file
          try {
            Files.deleteIfExists(tempZipFile);
          } catch (IOException e) {
            // Log but don't fail if we can't delete temp file
            WizardsOfLuaMod.LOGGER.warn("Failed to delete temporary file: " + tempZipFile, e);
          }
        }
      }
    }
  }

  private void deleteFilesInDirectory(Path dir) throws IOException {
    if (!Files.exists(dir))
      return;
    Files.walk(dir).sorted(Comparator.reverseOrder()).forEach(this::deleteFile);
  }

  private void deleteFile(Path p) {
    try {
      Files.delete(p);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private Path findProjectRoot() {
    Path current = Paths.get("").toAbsolutePath();
    while (current != null && !Files.exists(current.resolve("src/api/lua"))) {
      current = current.getParent();
    }
    return current != null ? current : Paths.get("");
  }

  private void copyDirectory(Path source, Path target) throws IOException {
    Files.walk(source).forEach(path -> {
      try {
        Path dest = target.resolve(source.relativize(path));
        if (Files.isDirectory(path)) {
          Files.createDirectories(dest);
        } else {
          Files.copy(path, dest, StandardCopyOption.REPLACE_EXISTING);
        }
      } catch (IOException e) {
        throw new RuntimeException("Failed to copy: " + path, e);
      }
    });
  }

  private void handleException(Throwable t, class_2168 source) {
    String message = String.format(
        "An unexpected error occured during the extraction of api files: %s", t.getMessage());
    WizardsOfLuaMod.LOGGER.error(message, t);
    String stackTrace = getStackTrace(t);
    class_2561 errorText = class_5250.method_43477(class_8828.method_54232(message)).method_27693(stackTrace);
    source.method_9213(errorText);
  }

  private String getStackTrace(Throwable throwable) {
    StringWriter writer = new StringWriter();
    throwable.printStackTrace(new PrintWriter(writer));
    String result = writer.toString();
    if (result.length() > 200) {
      result = result.substring(0, 200) + "...";
    }
    return result;
  }
}
