package net.wizardsoflua.lua.classes;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.List;
import net.sandius.rembulan.LuaRuntimeException;
import net.sandius.rembulan.runtime.ExecutionContext;
import net.sandius.rembulan.runtime.ResolvedControlThrowable;
import net.wizardsoflua.filesystem.WolServerFileSystem;
import net.wizardsoflua.lua.function.NamedFunction2;
import net.wizardsoflua.lua.function.NamedFunction3;
import net.wizardsoflua.spell.SpellScope;

public class LuaFilesystem< //
    J extends WolServerFileSystem, //
    LC extends AbstractLuaClass<?, ?> //
> extends AbstractLuaInstance<J, LC> {

  public static class Class
      extends AbstractLuaClass<WolServerFileSystem, LuaFilesystem<WolServerFileSystem, Class>> {
    public Class(SpellScope spellScope) {
      super("Filesystem", spellScope, null);
      addFunction(new ListFunction());
      addFunction(new IsDirFunction());
      addFunction(new IsFileFunction());
      addFunction(new MakeDirFunction());
      addFunction(new DeleteFunction());
      addFunction(new MoveFunction());
    }

    @Override
    protected final LuaFilesystem<WolServerFileSystem, Class> createNewLuaInstance(
        WolServerFileSystem javaInstance) {
      return new LuaFilesystem<>(this, javaInstance);
    }

    class ListFunction extends NamedFunction2 {
      @Override
      public String getName() {
        return "list";
      }

      @Override
      public void invoke(ExecutionContext context, Object selfArg, Object pathArg)
          throws ResolvedControlThrowable {
        LuaFilesystem<?, ?> self =
            getConverters().castTo(LuaFilesystem.class, selfArg, 1, "self", getName());
        String pathStr =
            getConverters().toJavaNullable(String.class, pathArg, 2, "path", getName());
        WolServerFileSystem fileSystem = self.getDelegate();
        Path path = pathStr == null ? fileSystem.getRootFolder() : fileSystem.getPath(pathStr);
        if (!Files.isDirectory(path)) {
          throw new LuaRuntimeException(String.format("%s is not a directory!", path));
        } else {
          try {
            List<String> result = Files.list(path) //
                .map(p -> p.getFileName().toString()) //
                .sorted() //
                .toList();
            context.getReturnBuffer().setTo(getConverters().toLua(result));
          } catch (IOException e) {
            throw new UncheckedIOException(e);
          }
        }
      }
    }

    class IsDirFunction extends NamedFunction2 {
      @Override
      public String getName() {
        return "isDir";
      }

      @Override
      public void invoke(ExecutionContext context, Object selfArg, Object pathArg)
          throws ResolvedControlThrowable {
        LuaFilesystem<?, ?> self =
            getConverters().castTo(LuaFilesystem.class, selfArg, 1, "self", getName());
        String pathStr = getConverters().toJava(String.class, pathArg, 2, "path", getName());
        WolServerFileSystem fileSystem = self.getDelegate();
        Path path = fileSystem.getPath(pathStr);
        boolean result = Files.exists(path) && Files.isDirectory(path);
        context.getReturnBuffer().setTo(getConverters().toLua(result));
      }
    }

    class IsFileFunction extends NamedFunction2 {
      @Override
      public String getName() {
        return "isFile";
      }

      @Override
      public void invoke(ExecutionContext context, Object selfArg, Object pathArg)
          throws ResolvedControlThrowable {
        LuaFilesystem<?, ?> self =
            getConverters().castTo(LuaFilesystem.class, selfArg, 1, "self", getName());
        String pathStr = getConverters().toJava(String.class, pathArg, 2, "path", getName());
        WolServerFileSystem fileSystem = self.getDelegate();
        Path path = fileSystem.getPath(pathStr);
        boolean result = Files.exists(path) && Files.isRegularFile(path);
        context.getReturnBuffer().setTo(getConverters().toLua(result));
      }
    }

    class MakeDirFunction extends NamedFunction2 {
      @Override
      public String getName() {
        return "makeDir";
      }

      @Override
      public void invoke(ExecutionContext context, Object selfArg, Object pathArg)
          throws ResolvedControlThrowable {
        LuaFilesystem<?, ?> self =
            getConverters().castTo(LuaFilesystem.class, selfArg, 1, "self", getName());
        String pathStr = getConverters().toJava(String.class, pathArg, 2, "path", getName());
        WolServerFileSystem fileSystem = self.getDelegate();
        Path path = fileSystem.getPath(pathStr);
        if (Files.exists(path) && Files.isRegularFile(path)) {
          context.getReturnBuffer().setTo(false);
        } else {
          try {
            Files.createDirectories(path);
            boolean result = Files.exists(path) && Files.isDirectory(path);
            context.getReturnBuffer().setTo(result);
          } catch (IOException ex) {
            throw new UncheckedIOException(ex);
          }
        }
      }
    }

    class DeleteFunction extends NamedFunction2 {
      @Override
      public String getName() {
        return "delete";
      }

      @Override
      public void invoke(ExecutionContext context, Object selfArg, Object pathArg)
          throws ResolvedControlThrowable {
        LuaFilesystem<?, ?> self =
            getConverters().castTo(LuaFilesystem.class, selfArg, 1, "self", getName());
        String pathStr = getConverters().toJava(String.class, pathArg, 2, "path", getName());
        WolServerFileSystem fileSystem = self.getDelegate();
        Path path = fileSystem.getPath(pathStr);
        if (fileSystem.isRootFolder(path)) {
          throw new LuaRuntimeException("Can't delete world root directory");
        }
        try {
          boolean result = Files.deleteIfExists(path) && !Files.exists(path);
          context.getReturnBuffer().setTo(result);
        } catch (IOException ex) {
          throw new UncheckedIOException(ex);
        }
      }
    }

    class MoveFunction extends NamedFunction3 {
      @Override
      public String getName() {
        return "move";
      }

      @Override
      public void invoke(ExecutionContext context, Object selfArg, Object fromPathArg,
          Object toPathArg) throws ResolvedControlThrowable {
        LuaFilesystem<?, ?> self =
            getConverters().castTo(LuaFilesystem.class, selfArg, 1, "self", getName());
        String source = getConverters().toJava(String.class, fromPathArg, 2, "fromPath", getName());
        String destination =
            getConverters().toJava(String.class, toPathArg, 3, "toPath", getName());
        WolServerFileSystem fileSystem = self.getDelegate();
        Path srcPath = fileSystem.getPath(source);
        if (fileSystem.isRootFolder(srcPath)) {
          throw new LuaRuntimeException("Can't move world root directory");
        }
        if (!Files.exists(srcPath)) {
          throw new LuaRuntimeException(String.format("%s does not exist!", srcPath));
        }
        Path destPath = fileSystem.getPath(destination);
        if (fileSystem.isRootFolder(destPath)) {
          throw new LuaRuntimeException("Can't overwrite world root directory");
        }
        boolean isFile = Files.isRegularFile(srcPath);
        boolean isDir = Files.isDirectory(srcPath);
        try {
          Path target = Files.move(srcPath, destPath, StandardCopyOption.ATOMIC_MOVE,
              StandardCopyOption.REPLACE_EXISTING);
          boolean result = Files.exists(target)
              && (isFile == Files.isRegularFile(target) || isDir == Files.isDirectory(target));
          context.getReturnBuffer().setTo(result);
        } catch (IOException ex) {
          throw new UncheckedIOException(ex);
        }
      }
    }
  }

  public LuaFilesystem(LC luaClass, J javaInstance) {
    super(luaClass, javaInstance, true);
  }
}
