package net.wizardsoflua.spell;

import static java.util.Objects.requireNonNull;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Predicate;
import org.jetbrains.annotations.Nullable;
import com.google.common.base.Supplier;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.minecraft.class_1297;
import net.minecraft.class_2165;
import net.minecraft.class_2168;
import net.minecraft.class_243;
import net.minecraft.class_2561;
import net.minecraft.class_2564;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.sandius.rembulan.LuaRuntimeException;
import net.sandius.rembulan.runtime.ExecutionContext;
import net.sandius.rembulan.runtime.IllegalOperationAttemptException;
import net.sandius.rembulan.runtime.LuaFunction;
import net.sandius.rembulan.runtime.UnresolvedControlThrowable;
import net.wizardsoflua.event.CustomEvent;
import net.wizardsoflua.event.EventHandlerType;
import net.wizardsoflua.event.WolEvent;
import net.wizardsoflua.lua.Converters;
import net.wizardsoflua.lua.module.event.EventInterceptor;
import net.wizardsoflua.lua.module.event.EventQueue;
import net.wizardsoflua.lua.scheduling.LuaScheduler;

public class Spell extends VirtualEntity {
  public static final String NAME = "Spell";
  private final long sid; // immutable spell id
  private final SpellScope spellScope;
  private final SpellProgram program;
  private boolean visible = false;
  private final Multimap<String, EventQueue> queues = HashMultimap.create();
  private final EventQueue.Context eventQueueContext = new EventQueue.Context() {
    @Override
    public void stop(EventQueue queue) {
      for (String name : queue.getNames()) {
        queues.remove(name, queue);
      }
    }

    @Override
    public long getCurrentTime() {
      return spellScope.getWorld().method_8510();
    }

    @Override
    public void pauseIfRequested(ExecutionContext context)
        throws IllegalOperationAttemptException, LuaRuntimeException, UnresolvedControlThrowable {
      program.getScheduler().pauseIfRequested(context);
    }
  };
  /**
   * The order of interceptors matters as later interceptors must not be called if the event was
   * canceled by a previous one. Also we need to avoid ConcurrentModificationException e.g. when
   * stopping during interception.
   */
  private final Multimap<String, EventInterceptor> interceptors =
      Multimaps.newListMultimap(new LinkedHashMap<String, Collection<EventInterceptor>>(),
          new Supplier<List<EventInterceptor>>() {
            @Override
            public List<EventInterceptor> get() {
              return new CopyOnWriteArrayList<EventInterceptor>();
            }
          });

  private final EventInterceptor.Context interceptorContext = new EventInterceptor.Context() {
    @Override
    public void stop(EventInterceptor interceptor) {
      for (String eventName : interceptor.getNames()) {
        interceptors.remove(eventName, interceptor);
      }
    }

    @Override
    public void handleException(String contextMessage, Throwable t) {
      program.handleException(contextMessage, t);
    }

    @Override
    public LuaScheduler getScheduler() {
      return program.getScheduler();
    }

    @Override
    public Converters getConverters() {
      return spellScope.getConverters();
    }
  };

  public Spell(long sid, SpellScope spellScope, class_243 pos, SpellProgram program) {
    super(spellScope, pos);
    this.sid = sid;
    this.spellScope = requireNonNull(spellScope, "spellScope");
    this.program = requireNonNull(program, "program");
    String name = Spell.NAME + "-" + sid;
    setName(name);
    program.getScheduler().addPauseContext(this::shouldPause);
    program.setSpell(this);
    spellScope.setSpell(this);
  }

  public boolean shouldPause() {
    if (queues.isEmpty()) {
      // no queues -> nothing to wait for, so keep running
      return false;
    }
    long now = spellScope.getWorld().method_8510();

    for (EventQueue queue : queues.values()) {
      long waitUntil = queue.getWaitUntil();
      if (now < waitUntil) {
        // we are still waiting for a message
        if (!queue.isEmpty()) {
          // but we have received one -> wake up
          return false;
        }
        // but we havn't yet received one -> pause
        return true;
      }
    }
    return false;
  }

  public long getSid() {
    return sid;
  }

  public SpellProgram getProgram() {
    return program;
  }

  public @Nullable class_1297 getOwner() {
    return spellScope.getOwner();
  }

  public SpellScope getSpellScope() {
    return spellScope;
  }

  public boolean isVisible() {
    return visible;
  }

  public void setVisible(boolean visible) {
    this.visible = visible;
  }

  @Override
  public void tick() {
    if (program.isFinished()) {
      return;
    }
    super.tick();
    program.resume();
    if (program.isFinished()) {
      setDead();
      return;
    }

    if (visible) {
      SpellAuraFX.spawnParticle(this);
    }
  }

  @Override
  public void setDead() {
    if (!program.isFinished()) {
      program.finish();
    }
    if (!program.isTerminated()) {
      super.setDead();
      program.terminate();
    }
  }

  public boolean onEvent(String eventName, WolEvent event, EventHandlerType eventHandlerType) {
    if (spellScope.getConverters().isSupported(event.getClass())) {
      Object luaEvent = spellScope.getConverters().toLua(event);
      if (eventHandlerType == EventHandlerType.BOTH
          || eventHandlerType == EventHandlerType.INTERCEPTORS) {
        // Handle interceptors
        for (EventInterceptor eventInterceptor : interceptors.get(eventName)) {
          Boolean proceed = eventInterceptor.onEvent(luaEvent);
          if (event.shouldAbort()) {
            return false;
          }
          if (proceed != null && !proceed && event.canBeCanceled()) {
            return false;
          }
        }
      }
      if (eventHandlerType == EventHandlerType.BOTH
          || eventHandlerType == EventHandlerType.COLLECTORS) {
        // Handle collectors
        event.onBeforeCollect();
        for (EventQueue eventQueue : queues.get(eventName)) {
          eventQueue.add(event);
        }
      }
    }
    return true;
  }

  public boolean fireEvent(CustomEvent customEvent) {
    return spellScope.getSpellRegistry().forwardEvent(customEvent.name(), customEvent,
        EventHandlerType.BOTH);
  }

  public EventQueue createEventQueue(List<String> eventNames) {
    EventQueue result = new EventQueue(eventNames, eventQueueContext);
    for (String name : eventNames) {
      queues.put(name, result);
    }
    return result;
  }

  public EventInterceptor createEventInterceptor(List<String> eventNames, LuaFunction eventHandler,
      @Nullable Long luaTicksLimit) {
    luaTicksLimit = luaTicksLimit == null ? this.program.getLuaTicksLimit() : luaTicksLimit;
    EventInterceptor result =
        new EventInterceptor(eventNames, eventHandler, luaTicksLimit, interceptorContext);
    for (String name : eventNames) {
      interceptors.put(name, result);
    }
    return result;
  }

  public int execute(String cmdStr, class_2165 commandOutput) {
    cmdStr = cmdStr.startsWith("/") ? cmdStr.substring(1) : cmdStr;
    class_2168 commandSource = createCommandSource(commandOutput);
    CommandDispatcher<class_2168> commandDispatcher =
        getWorld().method_8503().method_3734().method_9235();
    try {
      int result = commandDispatcher.execute(cmdStr, commandSource);
      return result;
    } catch (CommandSyntaxException e) {
      commandSource.method_9213(getErrorMessage(e));
      return 0;
    }
  }

  private static class_2561 getErrorMessage(CommandSyntaxException e) {
    class_2561 message = class_2564.method_10883(e.getRawMessage());
    String context = e.getContext();
    return context != null
        ? class_2561.method_43469("command.context.parse_error", message, e.getCursor(), context)
        : message;
  }


  @Override
  public SpellServerCommandSource createCommandSource(class_2165 output) {
    return super.createCommandSource(output).setSpell(this);
  }

  public Iterable<Spell> findSpells(List<Predicate<Spell>> predicates) {
    return spellScope.getSpellRegistry().filter(predicates);
  }

  public class_2680 getBlockState() {
    return spellScope.getWorld().method_8320(getBlockPos());
  }

  public class_2586 getBlockEntity() {
    class_2586 result = spellScope.getWorld().method_8321(getBlockPos());
    return result;
  }

  public void setBlockState(class_2680 blockState) {
    getWorld().method_8501(getBlockPos(), blockState);
  }
}
