package net.wizardsoflua.spell;

import static com.google.common.collect.Lists.transform;
import static java.lang.String.valueOf;
import static java.util.Collections.unmodifiableCollection;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Predicate;
import java.util.stream.Stream;
import net.wizardsoflua.event.EventHandlerType;
import net.wizardsoflua.event.WolEvent;

public class SpellRegistry {
  private final List<Spell> spells = new CopyOnWriteArrayList<>();

  public void add(Spell spell) {
    spells.add(spell);
  }

  public void tick() {
    for (Spell spell : spells) {
      spell.tick();
    }
  }

  public void onSpellTermiated(Spell spell) {
    spells.remove(spell);
  }

  public boolean forwardEvent(String eventName, WolEvent event, EventHandlerType eventHandlerType) {
    for (Spell spell : spells) {
      boolean proceed = spell.onEvent(eventName, event, eventHandlerType);
      if (!proceed) {
        return false;
      }
    }
    return true;
  }

  public Collection<Spell> getAll() {
    return unmodifiableCollection(spells);
  }

  public Collection<Spell> get(Predicate<Spell> predicate) {
    return unmodifiableCollection(spells.stream().filter(predicate).toList());
  }

  public Iterable<Spell> filter(List<Predicate<Spell>> predicates) {
    Stream<Spell> stream = spells.stream();
    for (Predicate<Spell> predicate : predicates) {
      stream = stream.filter(predicate);
    }
    return stream.toList();
  }

  public Iterable<String> getActiveSids() {
    return transform(spells, s -> valueOf(s.getSid()));
  }

  public Iterable<String> getActiveNames() {
    return transform(spells, Spell::getName);
  }

  public int breakSpells(Iterable<Spell> spells) {
    int result = 0;
    for (Spell spell : spells) {
      spell.setDead();
      result++;
    }
    return result;
  }
}
