package net.wizardsoflua.command.dynamic;

import static java.util.Objects.requireNonNull;
import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.ArgumentBuilder;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.mojang.brigadier.suggestion.SuggestionProvider;
import com.mojang.brigadier.tree.CommandNode;
import me.lucko.fabric.api.permissions.v0.Permissions;
import net.minecraft.class_2168;
import net.minecraft.class_2561;
import net.minecraft.class_3222;
import net.minecraft.class_7157;
import net.minecraft.server.MinecraftServer;
import net.wizardsoflua.spell.SpellCaster;

/**
 * Manages dynamic commands defined at runtime.
 */
public class DynamicCommandManager {
  private static final SimpleCommandExceptionType NO_PERMISSION = new SimpleCommandExceptionType(
      class_2561.method_43470("You do not have permission to use this command"));

  private final SpellCaster spellCaster;
  private final CommandNodeRemover commandNodeRemover;

  private final Map<String, CommandDef> dynamicCommandsDefs = new LinkedHashMap<>();
  private final Multimap<String, String> reachableCommands = LinkedListMultimap.create();
  private final Deque<CommandDef> definitionOrder = new ArrayDeque<CommandDef>();
  private final Set<CommandNode<class_2168>> createdNodes =
      Collections.newSetFromMap(new IdentityHashMap<>());

  private boolean needsReload = false;
  private boolean needsSync = false;

  public DynamicCommandManager(SpellCaster spellCaster) throws ReflectiveOperationException {
    this.spellCaster = requireNonNull(spellCaster, "spellCaster");
    this.commandNodeRemover = new CommandNodeRemover();
  }

  public Map<String, CommandDef> getAll() {
    return Collections.unmodifiableMap(dynamicCommandsDefs);
  }

  public CommandDef get(String id) {
    return dynamicCommandsDefs.get(id);
  }

  public boolean contains(String id) {
    return dynamicCommandsDefs.containsKey(id);
  }

  public void set(String commandId, String pattern, String code, class_2168 src,
      int level) {
    ParseResult parseResult = DynamicCommandParser.parse(pattern);
    CommandDef commandDef = new CommandDef(pattern, code, level, parseResult);
    commandDef.check(src.method_54310(), createdNodes);

    CommandDef oldCommandDef = dynamicCommandsDefs.put(commandId, commandDef);
    if (oldCommandDef != null) {
      Collection<String> tokenPaths = oldCommandDef.getTokenPaths();
      for (String tokenPath : tokenPaths) {
        reachableCommands.remove(tokenPath, commandId);
      }
    }
    for (String tokenPath : parseResult.getTokenPaths()) {
      reachableCommands.put(tokenPath, commandId);
    }
    markForReload();
  }

  public boolean remove(String commandId, class_2168 src) {
    CommandDef commandDef = dynamicCommandsDefs.remove(commandId);
    if (commandDef == null) {
      return false;
    }
    for (String tokenPath : commandDef.getTokenPaths()) {
      reachableCommands.remove(tokenPath, commandId);
    }
    markForReload();
    return true;
  }

  public void clearAll() {
    if (dynamicCommandsDefs.isEmpty()) {
      return;
    }
    dynamicCommandsDefs.clear();
    reachableCommands.clear();
    markForReload();
  }

  public void markForReload() {
    needsReload = true;
  }

  public void tick(MinecraftServer server) {
    if (needsSync) {
      needsSync = false;
      sendCommandTree(server);
    }
    if (needsReload) {
      needsReload = false;
      reload(server);
    }
  }

  private void sendCommandTree(MinecraftServer server) {
    var mgr = server.method_3734();
    for (class_3222 player : server.method_3760().method_14571()) {
      mgr.method_9241(player);
    }
  }

  private void reload(MinecraftServer server) {
    CommandDispatcher<class_2168> disp = server.method_3734().method_9235();
    class_7157 regAccess = class_7157.method_46722(server.method_30611(),
        server.method_27728().method_45560());
    reload(disp, regAccess);
  }

  public void reload(CommandDispatcher<class_2168> disp, class_7157 regAccess) {
    deregister(disp, regAccess);
    register(disp, regAccess);
  }

  private void deregister(CommandDispatcher<class_2168> dispatcher,
      class_7157 registryAccess) {
    if (definitionOrder.isEmpty()) {
      return;
    }
    while (!definitionOrder.isEmpty()) {
      CommandDef def = definitionOrder.pop();
      def.deregister(dispatcher, commandNodeRemover);
      needsSync = true;
    }
    createdNodes.clear();
  }

  private void register(CommandDispatcher<class_2168> dispatcher,
      class_7157 registryAccess) {
    for (var entry : dynamicCommandsDefs.entrySet()) {
      String commandId = entry.getKey();
      CommandDef commandDef = entry.getValue();

      commandDef.register(dispatcher, buildLiteral(commandId, commandDef, registryAccess));
      definitionOrder.push(commandDef);
      createdNodes.addAll(commandDef.getCreatedNodes());
      needsSync = true;
    }
  }

  @SuppressWarnings("unchecked")
  private LiteralArgumentBuilder<class_2168> buildLiteral(String commandId, CommandDef def,
      class_7157 registryAccess) {
    List<Token> tokens = def.getTokens();
    int lastPlaceholderIdx = def.findLastPlaceholderIndex();

    List<ArgumentBuilder<class_2168, ?>> builders = new ArrayList<>();
    List<Function<CommandContext<class_2168>, Object>> extractors = new ArrayList<>();
    List<Object> luaArgNames = new ArrayList<>();

    int argCount = 0;
    for (int i = 0; i < tokens.size(); i++) {
      Token tok = tokens.get(i);
      if (!tok.isPlaceholder()) {
        builders.add(method_9247(tok.tokenName()));
      } else {
        argCount++;
        String nodeName = tok.tokenName();
        Placeholder ph = tok.placeholder();
        boolean greedy = (ph == Placeholder.STRING && i == lastPlaceholderIdx);

        if (ph == Placeholder.STRING && tok.enumValues() != null) {
          // enum‐style string: use SuggestionProvider + runtime check
          List<String> opts = tok.enumValues();
          var argB = method_9244(nodeName,
              greedy ? StringArgumentType.greedyString() : StringArgumentType.string())
                  .suggests((SuggestionProvider<class_2168>) (ctx, sb) -> {
                    for (String v : opts)
                      sb.suggest(v);
                    return sb.buildFuture();
                  });
          builders.add(argB);
          extractors.add(ctx -> {
            String val = StringArgumentType.getString(ctx, nodeName);
            if (!opts.contains(val)) {
              throw new IllegalArgumentException("Unknown value \"" + val + "\"");
            }
            return val;
          });
        } else {
          // standard placeholder
          builders.add(ph.createBuilder(nodeName, greedy, registryAccess));
          extractors.add(ctx -> ph.extract(ctx, nodeName, registryAccess));
        }
        luaArgNames.add(tok.isNamedPlaceholder() ? tok.tokenName() : (argCount));
      }
    }

    // Add nodeId check after each token
    int size = builders.size();
    if (size > 0) {
      for (int i = 0; i < size; ++i) {
        String tokenPath = def.getTokenPath(i);
        builders.set(i, builders.get(i).requires(src -> {
          if (!contains(commandId)) {
            return false;
          }
          Collection<String> commandIds = reachableCommands.get(tokenPath);
          return src.method_9228() == null || commandIds.stream() //
              .map(cId -> Permissions.check(src, cId, def.getLevel())) //
              .anyMatch(b -> b);
        }));
      }
    }

    // build executor
    Command<class_2168> exec = ctx -> {
      class_2168 src = ctx.getSource();
      LinkedHashMap<Object, Object> args = new LinkedHashMap<Object, Object>();
      try {
        for (int j = 0; j < extractors.size(); j++) {
          Object v = extractors.get(j).apply(ctx);
          if (v != null) {
            Object key = luaArgNames.get(j);
            args.put(key, v);
          }
        }
      } catch (IllegalArgumentException e) {
        class_2561 message = class_2561.method_30163(e.getMessage());
        throw new CommandSyntaxException(new SimpleCommandExceptionType(message), message);
      }
      return spellCaster.castNewSpell(src, def.getCode(), args);
    };

    // chain builders
    ArgumentBuilder<class_2168, ?> chain;
    if (builders.isEmpty()) {
      throw new IllegalStateException("Unexpected state. Builders are empty.");
    } else {
      int i = builders.size() - 1;
      chain = builders.get(i).executes(ctx -> {
        class_2168 src = ctx.getSource();
        if (src.method_9228() != null && !Permissions.check(src, commandId, def.getLevel())) {
          throw NO_PERMISSION.create();
        }
        return exec.run(ctx);
      });
      for (int j = builders.size() - 2; j >= 0; j--) {
        chain = builders.get(j).then(chain);
      }
    }
    return (LiteralArgumentBuilder<class_2168>) chain;
  }
}
