package net.wizardsoflua.command.dynamic;

import static com.google.common.base.Preconditions.checkArgument;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Parses and validates a dynamic‐command pattern string into a list of Tokens.
 */
public class DynamicCommandParser {
  // matches code like "s[foo|bar|baz]"
  private static final Pattern ENUM_PATTERN = Pattern.compile("^([a-zA-Z]+)\\[(.+)]$");

  /**
   * Splits a pattern (e.g. "%s foo:%i bar") into Tokens, ensures every placeholder is one of the
   * supported types, and enforces “all-or-none” naming.
   */
  public static ParseResult parse(String pattern) {
    checkArgument(pattern != null && !pattern.isBlank(), "pattern is missing");

    String[] parts = pattern.trim().split("\\s+");
    List<Token> tokens = new ArrayList<>(parts.length);
    if (parts.length == 0) {
      throw new IllegalArgumentException("No tokens in pattern found!");
    }

    int totalPlaceholders = 0;
    int namedPlaceholders = 0;
    String tokenPath = "";

    for (int i = 0; i < parts.length; ++i) {
      String raw = parts[i];
      if (raw.startsWith("%")) {
        // unnamed placeholder, possibly enum
        totalPlaceholders++;
        String placeholderCode = raw.substring(1);
        String nodeName = String.valueOf(totalPlaceholders);

        List<String> enumValues = null;
        Matcher m = ENUM_PATTERN.matcher(placeholderCode);
        if (m.matches()) {
          placeholderCode = m.group(1);
          enumValues = List.of(m.group(2).split("\\|"));
        }

        Placeholder ph = Placeholder.byCode(placeholderCode);
        tokenPath += tokenPath.isEmpty() ? nodeName : "." + nodeName;
        tokens.add(new Token(raw, tokenPath, nodeName, true, null, ph, enumValues));

      } else if (raw.contains(":%")) {
        // named placeholder, possibly enum
        totalPlaceholders++;
        namedPlaceholders++;
        int idx = raw.indexOf(":%");
        String placeholderName = raw.substring(0, idx);
        if (placeholderName.isEmpty()) {
          throw new IllegalArgumentException(
              "Placeholder missing name before ':%' in '" + raw + "'");
        }
        String nodeName = placeholderName;
        String placeholderCode = raw.substring(idx + 2);

        List<String> enumValues = null;
        Matcher m = ENUM_PATTERN.matcher(placeholderCode);
        if (m.matches()) {
          placeholderCode = m.group(1);
          enumValues = List.of(m.group(2).split("\\|"));
        }

        Placeholder ph = Placeholder.byCode(placeholderCode);
        tokenPath += tokenPath.isEmpty() ? nodeName : "." + nodeName;
        tokens.add(new Token(raw, tokenPath, nodeName, true, placeholderName, ph, enumValues));

      } else {
        // plain literal
        String nodeName = raw;
        tokenPath += tokenPath.isEmpty() ? nodeName : "." + nodeName;
        tokens.add(new Token(raw, tokenPath, nodeName, false, null, null, null));
      }
    }

    // enforce that either all placeholders are named or none are
    if (namedPlaceholders > 0 && namedPlaceholders < totalPlaceholders) {
      throw new IllegalArgumentException(
          "Invalid command pattern. Either name all placeholders or none of them.");
    }
    return new ParseResult(tokens);
  }
}
