package net.wizardsoflua.lua.module.event;

import static com.google.common.base.Preconditions.checkNotNull;
import java.util.ArrayDeque;
import java.util.Deque;
import org.jetbrains.annotations.Nullable;
import com.google.common.collect.ImmutableSet;
import net.sandius.rembulan.LuaRuntimeException;
import net.sandius.rembulan.runtime.ExecutionContext;
import net.sandius.rembulan.runtime.IllegalOperationAttemptException;
import net.sandius.rembulan.runtime.UnresolvedControlThrowable;
import net.wizardsoflua.event.WolEvent;

public class EventQueue {
  public interface Context {
    void stop(EventQueue eventQueue);

    long getCurrentTime();

    void pauseIfRequested(ExecutionContext context)
        throws IllegalOperationAttemptException, LuaRuntimeException, UnresolvedControlThrowable;
  }

  private final ImmutableSet<String> names;
  private final Context context;
  private Deque<WolEvent> elements = new ArrayDeque<>();
  private long waitUntil = -1;

  public EventQueue(Iterable<String> names, Context context) {
    this.names = ImmutableSet.copyOf(names);
    this.context = checkNotNull(context, "context==null!");
  }

  public ImmutableSet<String> getNames() {
    return names;
  }

  public void pauseIfRequested(ExecutionContext context)
      throws IllegalOperationAttemptException, LuaRuntimeException, UnresolvedControlThrowable {
    this.context.pauseIfRequested(context);
  }

  public void stop() {
    context.stop(this);
  }

  public long getWaitUntil() {
    return waitUntil;
  }

  public void setTimeout(@Nullable Long timeout) {
    if (timeout != null) {
      waitUntil = context.getCurrentTime() + timeout;
    } else {
      waitUntil = Long.MAX_VALUE;
    }
  }

  public boolean isEmpty() {
    return elements.isEmpty();
  }

  public void clear() {
    elements.clear();
  }

  public @Nullable WolEvent latest() {
    WolEvent element = null;
    while (!elements.isEmpty()) {
      element = elements.removeFirst();
    }
    return element;
  }

  public WolEvent pop() {
    WolEvent result = elements.pop();
    stopWaitingForEvents();
    return result;
  }

  public void add(WolEvent event) {
    elements.addLast(event);
  }

  private void stopWaitingForEvents() {
    waitUntil = -1;
  }
}
