package net.wizardsoflua.command;

import static com.mojang.brigadier.arguments.IntegerArgumentType.integer;
import static com.mojang.brigadier.arguments.StringArgumentType.string;
import static java.util.Objects.requireNonNull;
import static net.minecraft.class_2186.method_9305;
import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;
import static net.wizardsoflua.WizardsOfLuaMod.WOL_PERMISSION_LEVEL;
import java.util.Collection;
import java.util.function.Predicate;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import me.lucko.fabric.api.permissions.v0.Permissions;
import net.minecraft.class_124;
import net.minecraft.class_2168;
import net.minecraft.class_2170.class_5364;
import net.minecraft.class_2561;
import net.minecraft.class_5250;
import net.minecraft.class_7157;
import net.wizardsoflua.WizardsOfLuaMod;
import net.wizardsoflua.spell.Spell;
import net.wizardsoflua.spell.SpellRegistry;

public class WolSpellListCommand implements CommandRegisterer {
  private static final String WOL_PERMISSION = WizardsOfLuaMod.MOD_ID + ".wol";
  private static final String WOL_SPELL_PERMISSION = WOL_PERMISSION + ".spell";
  private static final String WOL_SPELL_LIST_PERMISSION = WOL_SPELL_PERMISSION + ".list";

  private static final String SID_ARG = "sid";
  private static final String NAME_ARG = "name";
  private static final String OWNER_ARG = "owner";

  private static final int MAX_LINE_LENGTH = 100;
  private static final String ELLIPSIS = "...";

  private final SpellRegistry spellRegistry;

  public WolSpellListCommand(SpellRegistry spellRegistry) {
    this.spellRegistry = requireNonNull(spellRegistry, "spellRegistry");
  }

  @Override
  public void register(CommandDispatcher<class_2168> d,
      class_7157 registryAccess, class_5364 environment) {
    d.register(method_9247("wol")
        .requires(src -> src.method_9228() == null
            || Permissions.check(src, WOL_SPELL_PERMISSION, WOL_PERMISSION_LEVEL))
        .then(method_9247("spell") //
            .requires(src -> src.method_9228() == null
                || Permissions.check(src, WOL_SPELL_PERMISSION, WOL_PERMISSION_LEVEL))
            .then(method_9247("list") //
                .requires(src -> src.method_9228() == null
                    || Permissions.check(src, WOL_SPELL_LIST_PERMISSION, WOL_PERMISSION_LEVEL))
                .executes(this::listByCaller) //
                .then(method_9247("all") //
                    .requires(source -> source.method_9228() == null || Permissions.check(source,
                        WOL_SPELL_LIST_PERMISSION + ".all", WOL_PERMISSION_LEVEL)) //
                    .executes(this::listAll))
                .then(method_9247("bySid") //
                    .requires(source -> source.method_9228() == null || Permissions.check(source,
                        WOL_SPELL_LIST_PERMISSION + ".bySid", WOL_PERMISSION_LEVEL)) //
                    .then(method_9244(SID_ARG, integer(0)).executes(this::listBySid)))
                .then(method_9247("byName") //
                    .requires(source -> source.method_9228() == null || Permissions.check(source,
                        WOL_SPELL_LIST_PERMISSION + ".byName", WOL_PERMISSION_LEVEL)) //
                    .then(method_9244(NAME_ARG, string()).executes(this::listByName)))
                .then(method_9247("byOwner") //
                    .requires(source -> source.method_9228() == null || Permissions.check(source,
                        WOL_SPELL_LIST_PERMISSION + ".byOwner", WOL_PERMISSION_LEVEL)) //
                    .then(method_9244(OWNER_ARG, method_9305()).executes(this::listByOwner))) //
            )));
  }

  private int listByCaller(CommandContext<class_2168> ctx) {
    var source = ctx.getSource();
    var entity = source.method_9228();
    String msg = "Your active spells";
    return listSpells(source, msg, s -> entity.equals(s.getOwner()));
  }

  private int listAll(CommandContext<class_2168> ctx) {
    String msg = "Active spells";
    return listSpells(ctx.getSource(), msg, spellRegistry.getAll());
  }

  private int listBySid(CommandContext<class_2168> ctx) {
    int sid = com.mojang.brigadier.arguments.IntegerArgumentType.getInteger(ctx, SID_ARG);
    String msg = "Active spells with sid " + sid;
    return listSpells(ctx.getSource(), msg, s -> s.getSid() == sid);
  }

  private int listByName(CommandContext<class_2168> ctx) {
    String name = com.mojang.brigadier.arguments.StringArgumentType.getString(ctx, NAME_ARG);
    String msg = "Active spells with name '" + name + "'";
    return listSpells(ctx.getSource(), msg, s -> name.equals(s.getName()));
  }

  private int listByOwner(CommandContext<class_2168> ctx) throws CommandSyntaxException {
    var owner = net.minecraft.class_2186.method_9315(ctx, OWNER_ARG);
    String msg = "Active spells of " + owner.method_5477().getString();
    return listSpells(ctx.getSource(), msg, s -> owner.equals(s.getOwner()));
  }

  private int listSpells(class_2168 src, String msg, Predicate<Spell> filter) {
    return listSpells(src, msg, spellRegistry.get(filter));
  }

  private int listSpells(class_2168 src, String message, Collection<Spell> spells) {
    int count = spells.size();
    if (count == 0) {
      src.method_9226(() -> class_2561.method_30163("No spells found"), false);
    } else {
      class_5250 formatted = format(message, spells);
      src.method_9226(() -> formatted, false);
    }
    return count;
  }

  private class_5250 format(String message, Iterable<Spell> spells) {
    class_5250 result = class_2561.method_43470(message + ":\n").method_27661().method_27692(class_124.field_1065);
    for (Spell spell : spells) {
      class_5250 line =
          class_2561.method_43470("").method_10852(class_2561.method_43470("#" + spell.getSid()).method_27692(class_124.field_1054))
              .method_10852(class_2561.method_43470(" ")).method_10852(spell.getDisplayName()).method_10852(class_2561.method_43470(": "));
      String code = spell.getProgram().getCode();
      int max = MAX_LINE_LENGTH - line.getString().length();
      if (code.length() > max) {
        if (max < ELLIPSIS.length()) {
          code = ELLIPSIS;
        } else {
          code = code.substring(0, max - ELLIPSIS.length()) + ELLIPSIS;
        }
      }
      line.method_10852(class_2561.method_43470(code).method_27692(class_124.field_1075));
      result.method_10852(line).method_10852(class_2561.method_43470("\n"));
    }
    return result;
  }
}
