package net.wizardsoflua.command.dynamic;

import static java.util.Objects.requireNonNull;
import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.ArgumentBuilder;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.mojang.brigadier.suggestion.SuggestionProvider;
import me.lucko.fabric.api.permissions.v0.Permissions;
import net.minecraft.class_2168;
import net.minecraft.class_2170.class_5364;
import net.minecraft.class_2561;
import net.minecraft.class_3222;
import net.minecraft.class_7157;
import net.minecraft.server.MinecraftServer;
import net.wizardsoflua.command.CommandRegisterer;
import net.wizardsoflua.spell.SpellCaster;

/**
 * Manages dynamic commands defined at runtime.
 */
public class DynamicCommandManager implements CommandRegisterer {
  private static final SimpleCommandExceptionType NO_PERMISSION = new SimpleCommandExceptionType(
      class_2561.method_43470("You do not have permission to use this command"));

  private final SpellCaster spellCaster;

  /** ID → (pattern + code + level) */
  private final Map<String, CommandDef> dynamicCommandsDefs = new LinkedHashMap<>();
  /** nodeId → commandId */
  private final Multimap<String, String> reachableCommands = LinkedListMultimap.create();

  private boolean needsReload = false;
  private boolean needsSync = false;

  public DynamicCommandManager(SpellCaster spellCaster) {
    this.spellCaster = requireNonNull(spellCaster, "spellCaster");
  }

  public record CommandDef(String pattern, String code, int level, ParseResult parseResult) {
  }

  public Map<String, CommandDef> getAll() {
    return Collections.unmodifiableMap(dynamicCommandsDefs);
  }

  public void set(String commandId, String pattern, String code, class_2168 src,
      int level) {
    ParseResult parseResult = DynamicCommandParser.parse(pattern);
    CommandDef commandDef = new CommandDef(pattern, code, level, parseResult);
    CommandDef oldCommandDef = dynamicCommandsDefs.put(commandId, commandDef);
    if (oldCommandDef != null) {
      Map<String, Collection<String>> map = reachableCommands.asMap();
      Collection<String> nodeIds = oldCommandDef.parseResult().getNodeIds();
      for (String nodeId : nodeIds) {
        Collection<String> col = map.get(nodeId);
        col.remove(commandId);
      }
    }
    for (String nodeId : parseResult.getNodeIds()) {
      reachableCommands.put(nodeId, commandId);
    }
    markForReload();
  }

  public boolean remove(String commandId, class_2168 src) {
    CommandDef commandDef = dynamicCommandsDefs.remove(commandId);
    if (commandDef == null) {
      return false;
    }
    for (String nodeId : commandDef.parseResult().getNodeIds()) {
      reachableCommands.remove(nodeId, commandId);
    }
    markForReload();
    return true;
  }

  public void clearAll(class_2168 src) {
    if (dynamicCommandsDefs.isEmpty()) {
      return;
    }
    dynamicCommandsDefs.clear();
    reachableCommands.clear();
    markForReload();
  }

  @Override
  public void register(CommandDispatcher<class_2168> dispatcher,
      class_7157 registryAccess, class_5364 environment) {
    for (var entry : dynamicCommandsDefs.entrySet()) {
      dispatcher.register(buildLiteral(entry.getKey(), entry.getValue(), registryAccess));
      needsSync = true;
    }
  }

  public void tick(MinecraftServer server) {
    if (needsSync) {
      needsSync = false;
      sendCommandTree(server);
    }
    if (needsReload) {
      needsReload = false;
      reload(server);
    }
  }

  private void reload(MinecraftServer server) {
    server.method_3734().method_44252(server.method_3739(), "/reload");
  }

  private void sendCommandTree(MinecraftServer server) {
    var mgr = server.method_3734();
    for (class_3222 player : server.method_3760().method_14571()) {
      mgr.method_9241(player);
    }
  }

  private void markForReload() {
    needsReload = true;
  }

  @SuppressWarnings("unchecked")
  private LiteralArgumentBuilder<class_2168> buildLiteral(String commandId, CommandDef def,
      class_7157 registryAccess) {
    List<Token> tokens = def.parseResult.tokens();
    int lastPlaceholderIdx = def.parseResult.findLastPlaceholderIndex();

    List<ArgumentBuilder<class_2168, ?>> builders = new ArrayList<>();
    List<Function<CommandContext<class_2168>, Object>> extractors = new ArrayList<>();
    List<Object> luaArgNames = new ArrayList<>();
    int autoCount = 0;

    for (int i = 0; i < tokens.size(); i++) {
      Token tok = tokens.get(i);
      if (!tok.isPlaceholder()) {
        builders.add(method_9247(tok.raw()));
      } else {
        String argName = tok.isNamed() ? tok.name()
            : String.valueOf(++autoCount) + ":" + tok.placeholder().getName();
        Placeholder ph = tok.placeholder();
        boolean greedy = (ph == Placeholder.STRING && i == lastPlaceholderIdx);

        if (ph == Placeholder.STRING && tok.enumValues() != null) {
          // enum‐style string: use SuggestionProvider + runtime check
          List<String> opts = tok.enumValues();
          var argB = method_9244(argName,
              greedy ? StringArgumentType.greedyString() : StringArgumentType.string())
                  .suggests((SuggestionProvider<class_2168>) (ctx, sb) -> {
                    for (String v : opts)
                      sb.suggest(v);
                    return sb.buildFuture();
                  });
          builders.add(argB);
          extractors.add(ctx -> {
            String val = StringArgumentType.getString(ctx, argName);
            if (!opts.contains(val)) {
              throw new IllegalArgumentException("Unknown value \"" + val + "\"");
            }
            return val;
          });
        } else {
          // standard placeholder
          builders.add(ph.createBuilder(argName, greedy, registryAccess));
          extractors.add(ctx -> ph.extract(ctx, argName, registryAccess));
        }
        luaArgNames.add(tok.isNamed() ? tok.name() : autoCount);
      }
    }

    // Add nodeId check after each token
    int size = builders.size();
    if (size > 0) {
      for (int i = 0; i < size; ++i) {
        String nodeId = def.parseResult.getNodeId(i);
        builders.set(i, builders.get(i).requires(src -> {
          Collection<String> commandIds = reachableCommands.get(nodeId);
          return src.method_9228() == null || commandIds.stream() //
              .map(cId -> Permissions.check(src, cId, def.level())) //
              .anyMatch(b -> b);
        }));
      }
    }

    // build executor
    Command<class_2168> exec = ctx -> {
      class_2168 src = ctx.getSource();
      LinkedHashMap<Object, Object> args = new LinkedHashMap<Object, Object>();
      try {

        for (int j = 0; j < extractors.size(); j++) {
          Object v = extractors.get(j).apply(ctx);
          if (v != null) {
            args.put(luaArgNames.get(j), v);
          }
        }
      } catch (IllegalArgumentException e) {
        class_2561 message = class_2561.method_30163(e.getMessage());
        throw new CommandSyntaxException(new SimpleCommandExceptionType(message), message);
      }
      return spellCaster.castNewSpell(src, def.code(), args);
    };

    // chain builders
    ArgumentBuilder<class_2168, ?> chain;
    if (builders.isEmpty()) {
      chain = method_9247("")
          .requires(
              src -> src.method_9228() == null || Permissions.check(src, commandId, def.level()))
          .executes(exec);
    } else {
      int i = builders.size() - 1;
      chain = builders.get(i).executes(ctx -> {
        String cmdId = commandId;
        class_2168 src = ctx.getSource();
        if (src.method_9228() != null && !Permissions.check(src, commandId, def.level())) {
          throw NO_PERMISSION.create();
        }
        return exec.run(ctx);
      });
      for (int j = builders.size() - 2; j >= 0; j--) {
        chain = builders.get(j).then(chain);
      }
    }
    return (LiteralArgumentBuilder<class_2168>) chain;
  }
}
