package net.wizardsoflua.lua.module.event;

import static java.util.Objects.requireNonNull;
import java.util.List;
import org.jetbrains.annotations.Nullable;
import com.google.common.collect.ImmutableSet;
import net.sandius.rembulan.exec.CallException;
import net.sandius.rembulan.runtime.LuaFunction;
import net.wizardsoflua.lua.Converters;
import net.wizardsoflua.lua.scheduling.LuaScheduler;

public class EventInterceptor {
  private final ImmutableSet<String> names;
  private final LuaFunction eventHandler;
  private final Context context;
  private long luaTicksLimit;
  private boolean pending = false;

  public interface Context {
    void stop(EventInterceptor subscription);

    void handleException(String contextMessage, Throwable t);

    LuaScheduler getScheduler();

    Converters getConverters();
  }

  public EventInterceptor(List<String> names, LuaFunction eventHandler, long luaTicksLimit,
      Context context) {
    this.luaTicksLimit = luaTicksLimit;
    this.names = ImmutableSet.copyOf(requireNonNull(names, "names"));
    this.eventHandler = requireNonNull(eventHandler, "eventHandler");
    this.context = requireNonNull(context, "context");;
  }

  public ImmutableSet<String> getNames() {
    return names;
  }

  public LuaFunction getEventHandler() {
    return eventHandler;
  }

  public @Nullable Boolean onEvent(Object luaEvent) {
    try {
      Object[] result = callDuringEventIntercepting(eventHandler, luaEvent);
      if (result == null || result.length < 1) {
        return null;
      }
      return context.getConverters().toJavaNullable(Boolean.class, result[0], "result[0]");
    } catch (CallException | InterruptedException ex) {
      context.handleException("Error in event interceptor", ex);
      return true; // proceed with other interceptors and handlers
    }
  }

  private Object[] callDuringEventIntercepting(LuaFunction function, Object... args)
      throws CallException, InterruptedException {
    boolean oldPending = pending;
    pending = true;
    try {
      return context.getScheduler().callUnpausable(luaTicksLimit, function, args);
    } finally {
      pending = oldPending;
    }
  }

  public boolean isDuringEventIntercepting() {
    return pending;
  }

  public void stop() {
    context.stop(this);
  }
}
