require "wol.globals"
---@meta
---@class Vec3
---@operator add(Vec3): Vec3
---@operator sub(Vec3): Vec3
---@operator mul(number): Vec3
---@operator mul(Vec3): number
---@operator unm: Vec3
---@operator concat(Vec3): string
---@field x number
---@field y number
---@field z number
--Vec3 = {}
Vec3 = declare("Vec3")

-- This declares a convenient 'constructor' delegating to Vec3:new()
-- Use it like this: v = Vec3(x, y, z)
local mt = {
  __call = function(_, x, y, z)
    return Vec3:new(x, y, z)
  end
}
setmetatable(Vec3, mt)

--- Initializes a new vector.
---@param x number
---@param y number
---@param z number
---@return Vec3
function Vec3:new(x, y, z)
  x = tonumber(x) or error("x is not a number")
  y = tonumber(y) or error("y is not a number")
  z = tonumber(z) or error("z is not a number")
  local o = { x = x, y = y, z = z }
  setmetatable(o, self)
  self.__index = self
  return o
end

--- Returns a string representation of this vector.
---@return string
function Vec3:tostring()
  return "{" .. self.x .. ", " .. self.y .. ", " .. self.z .. "}"
end

--- Metamethod for `tostring(vec)` or string concatenation.
---@param self Vec3
---@return string
Vec3.__tostring = Vec3.tostring

--- Returns a new vector that is the sum of this vector and another.
---@param other Vec3
---@return Vec3
function Vec3:add(other)
  return Vec3:new(self.x + other.x, self.y + other.y, self.z + other.z)
end

--- Metamethod for `vec1 + vec2`.
---@param v1 Vec3
---@param v2 Vec3
---@return Vec3
Vec3.__add = function(v1, v2)
  return v1:add(v2)
end

--- Returns a new vector that is the difference between this vector and another.
---@param other Vec3
---@return Vec3
function Vec3:subtract(other)
  return Vec3:new(self.x - other.x, self.y - other.y, self.z - other.z)
end

--- Metamethod for `vec1 - vec2`.
---@param v1 Vec3
---@param v2 Vec3
---@return Vec3
Vec3.__sub = function(v1, v2)
  return v1:subtract(v2)
end

--- Returns the squared magnitude of this vector.
---@return number
function Vec3:sqrMagnitude()
  return self.x^2 + self.y^2 + self.z^2
end

--- Returns the magnitude (length) of this vector.
---@return number
function Vec3:magnitude()
  return math.sqrt(self:sqrMagnitude())
end

--- Returns the dot product of this vector and another.
---@param other Vec3
---@return number
function Vec3:dotProduct(other)
  return self.x * other.x + self.y * other.y + self.z * other.z
end

--- Scales this vector by a scalar value.
---@param factor number
---@return Vec3
function Vec3:scale(factor)
  return Vec3:new(self.x * factor, self.y * factor, self.z * factor)
end

--- Metamethod for `vec * scalar` or `vec1 * vec2` (dot product).
---@param a Vec3|number
---@param b Vec3|number
---@return Vec3|number
Vec3.__mul = function(a, b)
  if type(a) == "number" then
    return b:scale(a)
  elseif type(b) == "number" then
    return a:scale(b)
  else
    return a:dotProduct(b)
  end
end

--- Inverts this vector (negates all components).
---@return Vec3
function Vec3:invert()
  return Vec3:new(-self.x, -self.y, -self.z)
end

--- Metamethod for `-vec`.
---@param v Vec3
---@return Vec3
Vec3.__unm = function(v)
  return v:invert()
end

--- Metamethod for `vec1 .. vec2` (string concatenation).
---@param a any
---@param b any
---@return string
function Vec3.__concat(a, b)
  return tostring(a) .. tostring(b)
end

--- Compares two vectors for equality.
---@param a Vec3
---@param b Vec3
---@return boolean
function Vec3.__eq(a, b)
  return a.x == b.x and a.y == b.y and a.z == b.z
end

--- Normalizes this vector to unit length.
---@return Vec3
function Vec3:normalize()
  local len = self:magnitude()
  if len == 0 then error("Can't normalize a zero-length vector") end
  return self:scale(1 / len)
end

--- Floors all components of the vector.
---@return Vec3
function Vec3:floor()
  return Vec3:new(math.floor(self.x), math.floor(self.y), math.floor(self.z))
end

--- Calculates the chunk coordinates (assuming 16-unit chunks).
---@return number, number
function Vec3:chunk()
  return self.x // 16, self.z // 16
end
