package net.wizardsoflua.command.dynamic;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jetbrains.annotations.Nullable;

/**
 * Parses and validates a dynamic‐command pattern string into a list of Tokens.
 */
public class DynamicCommandParser {
  // matches code like "s[foo|bar|baz]"
  private static final Pattern ENUM_PATTERN = Pattern.compile("^([a-zA-Z]+)\\[(.+)]$");

  /**
   * Splits a pattern (e.g. "%s foo:%i bar") into Tokens, ensures every placeholder is one of the
   * supported types, and enforces “all-or-none” naming.
   */
  public static @Nullable ParseResult parse(String pattern) {
    if (pattern == null || pattern.isBlank()) {
      return null;
    }

    String[] parts = pattern.trim().split("\\s+");
    List<Token> tokens = new ArrayList<>(parts.length);

    int totalPlaceholders = 0;
    int namedPlaceholders = 0;

    for (String part : parts) {
      if (part.startsWith("%")) {
        // unnamed placeholder, possibly enum
        String rawCode = part.substring(1);
        List<String> enumValues = null;
        String code = rawCode;
        String node = rawCode;

        Matcher m = ENUM_PATTERN.matcher(rawCode);
        if (m.matches()) {
          code = m.group(1);
          enumValues = List.of(m.group(2).split("\\|"));
        }

        Placeholder ph = Placeholder.byCode(code);
        tokens.add(new Token(part, true, null, node, ph, enumValues));
        totalPlaceholders++;

      } else if (part.contains(":%")) {
        // named placeholder, possibly enum
        int idx = part.indexOf(":%");
        String name = part.substring(0, idx);
        if (name.isEmpty()) {
          throw new IllegalArgumentException(
              "Placeholder missing name before ':%' in '" + part + "'");
        }
        String node = name;

        String rawCode = part.substring(idx + 2);
        List<String> enumValues = null;
        String code = rawCode;

        Matcher m = ENUM_PATTERN.matcher(rawCode);
        if (m.matches()) {
          code = m.group(1);
          enumValues = List.of(m.group(2).split("\\|"));
        }

        Placeholder ph = Placeholder.byCode(code);
        tokens.add(new Token(part, true, name, node, ph, enumValues));
        totalPlaceholders++;
        namedPlaceholders++;

      } else {
        // plain literal
        String nodeName = part;
        tokens.add(new Token(part, false, null, nodeName, null, null));
      }
    }

    // enforce that either all placeholders are named or none are
    if (namedPlaceholders > 0 && namedPlaceholders < totalPlaceholders) {
      throw new IllegalArgumentException(
          "Invalid command pattern. Either name all placeholders or none of them.");
    }
    return new ParseResult(tokens);
  }
}
