package net.wizardsoflua.filesystem;

import static com.google.common.base.Preconditions.checkNotNull;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileStore;
import java.nio.file.FileSystem;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.WatchService;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.nio.file.spi.FileSystemProvider;
import java.util.List;
import java.util.Set;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;

/**
 * This FileSystem only supports the {@link #getPath(String, String...)} and {@link #getSeparator()}
 * methods. All other methods throw a RuntimeException.
 */
public class RestrictedFileSystem extends DelegatingFileSystem {
  private Path rootFolder;

  public RestrictedFileSystem(FileSystem delegate, Path topmostDirectory) {
    super(delegate);
    this.rootFolder = checkNotNull(topmostDirectory, "topmostDirectory==null!");
  }

  public Path getRootFolder() {
    return rootFolder;
  }
  
  public boolean isRootFolder(Path path) {
    return rootFolder.toAbsolutePath().equals(path.toAbsolutePath());
  }

  @Override
  public Path getPath(String first, String... more) {
    if (first.startsWith("/")) {
      first = first.substring(1);
    }
    List<String> parts = Lists.asList(first, more);
    String filename = Joiner.on(File.separatorChar).join(parts);
    Path result = PathUtil.toPath(rootFolder, filename);
    return result;
  }

  @Override
  public String getSeparator() {
    return super.getSeparator();
  }

  private void throwUnsupported() {
    throw new RuntimeException(
        "Unsupported function in " + RestrictedFileSystem.class.getSimpleName());
  }

  @Override
  public int hashCode() {
    throwUnsupported();
    return super.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    throwUnsupported();
    return super.equals(obj);
  }

  @Override
  public FileSystemProvider provider() {
    throwUnsupported();
    return super.provider();
  }

  @Override
  public void close() throws IOException {
    throwUnsupported();
    super.close();
  }

  @Override
  public boolean isOpen() {
    throwUnsupported();
    return super.isOpen();
  }

  @Override
  public boolean isReadOnly() {
    throwUnsupported();
    return super.isReadOnly();
  }

  @Override
  public Iterable<Path> getRootDirectories() {
    throwUnsupported();
    return super.getRootDirectories();
  }

  @Override
  public Iterable<FileStore> getFileStores() {
    throwUnsupported();
    return super.getFileStores();
  }

  @Override
  public String toString() {
    throwUnsupported();
    return super.toString();
  }

  @Override
  public Set<String> supportedFileAttributeViews() {
    throwUnsupported();
    return super.supportedFileAttributeViews();
  }

  @Override
  public PathMatcher getPathMatcher(String syntaxAndPattern) {
    throwUnsupported();
    return super.getPathMatcher(syntaxAndPattern);
  }

  @Override
  public UserPrincipalLookupService getUserPrincipalLookupService() {
    throwUnsupported();
    return super.getUserPrincipalLookupService();
  }

  @Override
  public WatchService newWatchService() throws IOException {
    throwUnsupported();
    return super.newWatchService();
  }
}
