package net.wizardsoflua.spell;

import static java.util.Objects.requireNonNull;
import java.nio.file.FileSystem;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import net.minecraft.class_124;
import net.minecraft.class_1297;
import net.minecraft.class_2168;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_5250;
import net.minecraft.class_5251;
import net.sandius.rembulan.StateContext;
import net.sandius.rembulan.Table;
import net.sandius.rembulan.Variable;
import net.sandius.rembulan.env.RuntimeEnvironments;
import net.sandius.rembulan.exec.CallException;
import net.sandius.rembulan.exec.CallPausedException;
import net.sandius.rembulan.exec.Continuation;
import net.sandius.rembulan.impl.DefaultTable;
import net.sandius.rembulan.impl.StateContexts;
import net.sandius.rembulan.lib.BasicLib;
import net.sandius.rembulan.lib.CoroutineLib;
import net.sandius.rembulan.lib.IoLib;
import net.sandius.rembulan.lib.MathLib;
import net.sandius.rembulan.lib.ModuleLib;
import net.sandius.rembulan.lib.StringLib;
import net.sandius.rembulan.lib.TableLib;
import net.sandius.rembulan.lib.Utf8Lib;
import net.sandius.rembulan.load.LoaderException;
import net.sandius.rembulan.runtime.LuaFunction;
import net.wizardsoflua.WizardsOfLuaMod;
import net.wizardsoflua.event.SpellFinishCallback;
import net.wizardsoflua.event.SpellTerminatedCallback;
import net.wizardsoflua.extension.spell.api.resource.LuaTypes;
import net.wizardsoflua.lua.Converters;
import net.wizardsoflua.lua.compiler.ExtendedChunkLoader;
import net.wizardsoflua.lua.compiler.PatchedCompilerChunkLoader;
import net.wizardsoflua.lua.module.print.LogRedirector;
import net.wizardsoflua.lua.module.print.PrintRedirector;
import net.wizardsoflua.lua.module.searcher.ClasspathResourceSearcher;
import net.wizardsoflua.lua.module.searcher.LuaFunctionBinaryCache;
import net.wizardsoflua.lua.module.searcher.PatchedChunkLoadPathSearcher;
import net.wizardsoflua.lua.module.types.TypesModule;
import net.wizardsoflua.lua.module.wol.LogTarget;
import net.wizardsoflua.lua.module.wol.WolModule;
import net.wizardsoflua.lua.nbt.NbtConverter;
import net.wizardsoflua.lua.scheduling.CallFellAsleepException;
import net.wizardsoflua.lua.scheduling.LuaScheduler;

public class SpellProgram {
  public static final String ROOT_CLASS_PREFIX = "SpellByteCode";
  private final LuaFunctionBinaryCache luaFunctionCache;
  private final SpellScope spellScope;
  private final StateContext stateContext = StateContexts.newDefaultInstance();
  private final LuaScheduler scheduler = new LuaScheduler(stateContext);
  private final ExtendedChunkLoader loader = PatchedCompilerChunkLoader.of(ROOT_CLASS_PREFIX);
  private final SpellExceptionFactory exceptionFactory = new SpellExceptionFactory();
  private final LinkedHashMap<Object, Object> argsMap;

  private final Table env = stateContext.newTable();

  private enum State {
    NEW, SLEEPING, PAUSED, FINISHED, TERMINATED;
  }

  private State state = State.NEW;
  private Continuation continuation;

  /**
   * Max. number of Lua ticks a spell can run per game tick [default: 50000]
   */
  private long luaTicksLimit = 50000;
  /**
   * The totalWorldTime at which this program should stop sleeping.
   */
  private long wakeUpTime;

  private Spell spell;
  private String code;

  public SpellProgram(LuaFunctionBinaryCache luaFunctionCache, SpellScope spellScope, String code,
      LinkedHashMap<Object, Object> argsMap) {
    this.luaFunctionCache = Objects.requireNonNull(luaFunctionCache, "luaFunctionCache");
    this.spellScope = Objects.requireNonNull(spellScope, "spellScope");
    this.code = code;
    this.argsMap = argsMap;

    installSystemLibraries();
    PrintRedirector.installInto(env, message -> {
      spellScope.getCommandSource().method_9226(() -> class_2561.method_43470(message), false);
    });
    LogRedirector.installInto(env, message -> {
      LogTarget logTarget = spellScope.getWolModule().getLog();
      switch (logTarget) {
        case source:
          class_2168 commandSource = spellScope.getTopmostCommandSource();
          commandSource.method_45068(class_2561.method_43470(message));
          break;
        case console:
          WizardsOfLuaMod.LOGGER.info(message);
          break;
        case operators:
          for (class_3222 serverPlayerEntity : spellScope.getWorld().method_8503()
              .method_3760().method_14571()) {
            if (spellScope.getWorld().method_8503().method_3760()
                .method_14569(serverPlayerEntity.method_7334())) {
              serverPlayerEntity.method_64398(class_2561.method_30163(message));
            }
          }
          spellScope.getWorld().method_8503().method_3739().method_45068(class_2561.method_43470(message));
          break;
        case none:
          break;
      }
    });
    env.rawset(WolModule.NAME, getConverters().toLua(spellScope.getWolModule()));

    installTypesModule();
    spellScope.getTypes().installInto(env);
  }

  void setSpell(Spell spell) {
    this.spell = Objects.requireNonNull(spell, "spell");
    Object luaSpell = getConverters().toLua(spell);
    env.rawset("spell", luaSpell);
  }

  public Spell getSpell() {
    return spell;
  }

  public LuaTypes getTypes() {
    return spellScope.getTypes();
  }

  public Converters getConverters() {
    return spellScope.getConverters();
  }

  public NbtConverter getNbtConverter() {
    return spellScope.getNbtConverter();
  }

  public LuaScheduler getScheduler() {
    return scheduler;
  }

  public long getLuaTicksLimit() {
    return luaTicksLimit;
  }

  public void setLuaTicksLimit(long luaTicksLimit) {
    this.luaTicksLimit = luaTicksLimit;
  }

  private void installTypesModule() {
    TypesModule typesModule = new TypesModule(getTypes(), env, DefaultTable.factory());
    Object luaObject = getConverters().toLua(typesModule);
    env.rawset("Types", luaObject);
  }

  public String getCode() {
    return code;
  }

  public boolean isFinished() {
    return state == State.FINISHED || state == State.TERMINATED;
  }

  public boolean isTerminated() {
    return state == State.TERMINATED;
  }

  void terminate() {
    state = State.TERMINATED;
    SpellTerminatedCallback.EVENT.invoker().onSpellTerminated(this.spell);
  }

  void finish() {
    SpellFinishCallback.EVENT.invoker().onSpellFinish(spell);
    state = State.FINISHED;
  }

  private void installSystemLibraries() {
    ClassLoader classLoader = Thread.currentThread().getContextClassLoader();

    BasicLib.installInto(stateContext, env, /* runtimeEnv */ null, loader);

    // We don't pass the loader to the ModuleLib in order to prevent the installation of the
    // ChunkLoadPathSearcher
    ModuleLib.installInto(stateContext, env, /* runtimeEnv */ null, /* loader */ null,
        /* classLoader */ null);
    // Instead we install our own two searchers
    ClasspathResourceSearcher.installInto(env, loader, luaFunctionCache, classLoader);

    FileSystem realFileSystem = spellScope.getWolServerFileSystem().getDelegate();
    String luaSearchPath = spellScope.getDirectories().getLuaSearchPath(realFileSystem);
    PatchedChunkLoadPathSearcher.installInto(env, loader, luaFunctionCache, classLoader,
        realFileSystem, () -> luaSearchPath);

    CoroutineLib.installInto(stateContext, env);
    StringLib.installInto(stateContext, env);
    MathLib.installInto(stateContext, env);
    TableLib.installInto(stateContext, env);

    WolRuntimeEnvironment runtimeEnvironment = new WolRuntimeEnvironment(
        RuntimeEnvironments.system(), spellScope.getWolServerFileSystem());
    IoLib.installInto(stateContext, env, runtimeEnvironment);
    Utf8Lib.installInto(stateContext, env);
  }

  public void resume() {
    try {
      switch (state) {
        case NEW:
          compileAndRun();
          finish();
          break;
        case SLEEPING:
          if (wakeUpTime > spellScope.getWorld().method_8510()) {
            return;
          }
        case PAUSED:
          scheduler.resume(luaTicksLimit, continuation);
          finish();
          break;
        case FINISHED:
        case TERMINATED:
          return;
      }
    } catch (CallFellAsleepException ex) {
      int sleepDuration = ex.getSleepDuration();
      wakeUpTime = spellScope.getWorld().method_8510() + sleepDuration;
      continuation = ex.getContinuation();
      state = State.SLEEPING;
    } catch (CallPausedException ex) {
      continuation = ex.getContinuation();
      state = State.PAUSED;
    } catch (Exception ex) {
      handleException("Error during spell execution", ex);
    }
  }

  private void compileAndRun()
      throws LoaderException, CallException, CallPausedException, InterruptedException {
    // SpellModule.installInto(env, getConverters(), spellEntity);
    // SpellsModule.installInto(env, getConverters(), spellRegistry, spellEntity);
    //
    addGlobalRequirements();
    Object[] arguments = installArgs(argsMap);
    LuaFunction commandLineFunc = loader.loadTextChunk(new Variable(env), "command-line", code);
    scheduler.call(luaTicksLimit, commandLineFunc, arguments);
  }

  private void addGlobalRequirements() throws CallException, InterruptedException {
    LuaFunction requireFunction =
        requireNonNull((LuaFunction) env.rawget("require"), "Missing require function!");
    for (String module : Arrays.asList( //
        "wol.globals", //
        "wol.Vec3" //
    )) {
      scheduler.callUnpausable(Long.MAX_VALUE, requireFunction, module);
    }
  }

  private Object[] installArgs(Map<Object, Object> argMap) {
    List<Object> result = new ArrayList<>();
    if (argMap.size() > 0) {
      Table argsTable = DefaultTable.factory().newTable();
      for (Object key : argMap.keySet()) {
        Object value = spellScope.getConverters().toLua(argMap.get(key));
        argsTable.rawset(key, value);
        result.add(value);
      }
      env.rawset("args", argsTable);
    }
    return result.toArray();
  }

  public void handleException(String contextMessage, Throwable t) {
    // terminate();
    spell.setDead();
    SpellException s = exceptionFactory.create(t);
    String message = String.format("%s: %s", contextMessage, s.getMessage());
    WizardsOfLuaMod.LOGGER.error(message, s);
    class_1297 owner = spellScope.getOwner();
    class_3218 world = (class_3218) owner.method_37908();
    if (owner != null) {
      class_5250 txt = class_2561.method_43470(message) //
          .method_10862(class_2583.field_24360.method_27703( //
              class_5251.method_27718(class_124.field_1061)) //
              .method_10982(Boolean.valueOf(true)));
      owner.method_5671(world).method_45068(txt);
    }
  }
}
