/*
 * Decompiled with CFR 0.152.
 */
package org.skydream.sphinxriddle;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.ItemLike;
import org.skydream.sphinxriddle.ActiveQuestion;
import org.skydream.sphinxriddle.Config;
import org.skydream.sphinxriddle.Question;
import org.skydream.sphinxriddle.QuizScoreboard;
import org.skydream.sphinxriddle.Reward;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QuizManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(QuizManager.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private final MinecraftServer server;
    private final File questionsFile;
    private final File rewardsFile;
    private List<Question> questions = new ArrayList<Question>();
    private List<Reward> rewards = new ArrayList<Reward>();
    private ScheduledExecutorService scheduler;
    private final QuizScoreboard scoreboard;
    private ActiveQuestion currentQuestion;
    private int tickCounter = 0;
    private final Set<UUID> answeredPlayers = ConcurrentHashMap.newKeySet();

    public QuizManager(MinecraftServer server) {
        this.server = server;
        File configDir = new File("config/sphinxriddle");
        this.questionsFile = new File(configDir, "questions.json");
        this.rewardsFile = new File(configDir, "rewards.json");
        this.scoreboard = new QuizScoreboard(configDir);
        if (!configDir.exists()) {
            configDir.mkdirs();
        }
    }

    public void loadConfig() {
        this.loadQuestions();
        this.loadRewards();
        this.scoreboard.loadScoreboard();
        LOGGER.info("Loaded {} questions and {} rewards", (Object)this.questions.size(), (Object)this.rewards.size());
    }

    private void loadQuestions() {
        if (!this.questionsFile.exists()) {
            this.createDefaultQuestions();
            return;
        }
        try (FileReader reader = new FileReader(this.questionsFile);){
            Question[] questionArray = (Question[])GSON.fromJson((Reader)reader, Question[].class);
            this.questions = new ArrayList<Question>(Arrays.asList(questionArray));
        }
        catch (IOException e) {
            LOGGER.error("Failed to load questions", (Throwable)e);
            this.createDefaultQuestions();
        }
    }

    private void loadRewards() {
        if (!this.rewardsFile.exists()) {
            this.createDefaultRewards();
            return;
        }
        try (FileReader reader = new FileReader(this.rewardsFile);){
            Reward[] rewardArray = (Reward[])GSON.fromJson((Reader)reader, Reward[].class);
            this.rewards = new ArrayList<Reward>(Arrays.asList(rewardArray));
        }
        catch (IOException e) {
            LOGGER.error("Failed to load rewards", (Throwable)e);
            this.createDefaultRewards();
        }
    }

    private void createDefaultQuestions() {
        this.questions = new ArrayList<Question>();
        this.questions.add(new Question("Minecraft\u4e2d\u54ea\u79cd\u751f\u7269\u4f1a\u7206\u70b8\uff1f", "\u82e6\u529b\u6015"));
        this.questions.add(new Question("\u7528\u6765\u5408\u6210\u706b\u628a\u7684\u4e24\u79cd\u6750\u6599\u662f\u4ec0\u4e48\uff1f", "\u7164\u70ad\u548c\u6728\u68cd"));
        this.saveQuestions();
    }

    private void createDefaultRewards() {
        this.rewards = new ArrayList<Reward>();
        this.rewards.add(new Reward("minecraft:diamond", 3));
        this.rewards.add(new Reward("minecraft:emerald", 5));
        this.rewards.add(new Reward("minecraft:iron_ingot", 10));
        this.saveRewards();
    }

    public void saveQuestions() {
        try (FileWriter writer = new FileWriter(this.questionsFile);){
            GSON.toJson(this.questions, (Appendable)writer);
        }
        catch (IOException e) {
            LOGGER.error("Failed to save questions", (Throwable)e);
        }
    }

    public void saveRewards() {
        try (FileWriter writer = new FileWriter(this.rewardsFile);){
            GSON.toJson(this.rewards, (Appendable)writer);
        }
        catch (IOException e) {
            LOGGER.error("Failed to save rewards", (Throwable)e);
        }
    }

    public void startAutoQuestionTimer() {
        if (this.scheduler != null && !this.scheduler.isShutdown()) {
            this.scheduler.shutdown();
        }
        if (((Boolean)Config.AUTO_QUESTION_ENABLED.get()).booleanValue()) {
            this.scheduler = Executors.newSingleThreadScheduledExecutor();
            int interval = (Integer)Config.QUESTION_INTERVAL.get();
            this.scheduler.scheduleAtFixedRate(this::askRandomQuestion, interval, interval, TimeUnit.SECONDS);
            LOGGER.info("Auto question timer started with {} second interval", (Object)interval);
        }
    }

    public void stopAutoQuestionTimer() {
        if (this.scheduler != null) {
            this.scheduler.shutdown();
            LOGGER.info("Auto question timer stopped");
        }
    }

    public void askRandomQuestion() {
        if (this.questions.isEmpty()) {
            LOGGER.warn("No questions available");
            return;
        }
        Question question = this.questions.get(new Random().nextInt(this.questions.size()));
        this.askQuestion(question);
    }

    public void askQuestion(Question question) {
        if (this.currentQuestion != null) {
            this.currentQuestion = null;
        }
        this.currentQuestion = new ActiveQuestion(question, System.currentTimeMillis());
        this.answeredPlayers.clear();
        String message = ((String)Config.NEW_QUESTION_MESSAGE.get()).replace("%question%", question.getQuestion());
        this.broadcastMessage(message);
        LOGGER.info("New question: {}", (Object)question.getQuestion());
    }

    public void handlePlayerAnswer(ServerPlayer player, String message) {
        if (this.currentQuestion == null || this.answeredPlayers.contains(player.getUUID())) {
            return;
        }
        String answer = this.currentQuestion.getQuestion().getAnswer().toLowerCase();
        if (message.toLowerCase().contains(answer)) {
            this.answeredPlayers.add(player.getUUID());
            this.rewardPlayer(player);
            this.currentQuestion = null;
        }
    }

    private void rewardPlayer(ServerPlayer player) {
        if (this.rewards.isEmpty()) {
            LOGGER.warn("No rewards available");
            return;
        }
        Reward reward = this.rewards.get(new Random().nextInt(this.rewards.size()));
        Item item = (Item)BuiltInRegistries.ITEM.get(reward.getItemResourceLocation());
        int amount = new Random().nextInt(reward.getMaxAmount()) + 1;
        ItemStack itemStack = new ItemStack((ItemLike)item, amount);
        if (player.getInventory().add(itemStack)) {
            String itemId = reward.getItemId();
            String[] parts = itemId.split(":");
            String itemName = parts.length > 1 ? parts[1] : itemId;
            itemName = itemName.replace("_", " ");
            String[] words = itemName.split(" ");
            StringBuilder displayName = new StringBuilder();
            for (String word : words) {
                if (word.isEmpty()) continue;
                displayName.append(Character.toUpperCase(word.charAt(0))).append(word.substring(1)).append(" ");
            }
            String finalName = displayName.toString().trim();
            this.scoreboard.addScore(player.getScoreboardName(), 1);
            this.updateScoreboardForAllPlayers();
            String rewardMessage = ((String)Config.REWARD_MESSAGE.get()).replace("%player%", player.getScoreboardName()).replace("%reward%", amount + "x " + finalName);
            this.broadcastMessage(rewardMessage);
            LOGGER.info("Player {} received reward: {}x {} and 1 point", new Object[]{player.getScoreboardName(), amount, finalName});
        }
    }

    public void tick() {
        if (this.currentQuestion != null) {
            ++this.tickCounter;
            if (this.tickCounter >= 20) {
                this.tickCounter = 0;
                long currentTime = System.currentTimeMillis();
                long elapsed = (currentTime - this.currentQuestion.getStartTime()) / 1000L;
                if (elapsed >= (long)((Integer)Config.QUESTION_TIMEOUT.get()).intValue()) {
                    this.currentQuestion = null;
                    this.answeredPlayers.clear();
                    LOGGER.info("Question expired due to timeout");
                }
            }
        }
    }

    private void broadcastMessage(String message) {
        String cleanMessage = message.replaceAll("&[0-9a-f]", "");
        MutableComponent component = Component.literal((String)message.replace("&", "\u00a7"));
        this.server.getPlayerList().getPlayers().forEach(arg_0 -> QuizManager.lambda$broadcastMessage$0((Component)component, arg_0));
        LOGGER.info(cleanMessage);
    }

    public void initializeScoreboard() {
        this.scoreboard.initializeScoreboard(this.server);
        this.updateScoreboardForAllPlayers();
    }

    public void updateScoreboardForAllPlayers() {
        this.scoreboard.updateScoreboardDisplay(this.server);
    }

    public void resetScoreboard() {
        this.scoreboard.resetScoreboard();
        this.updateScoreboardForAllPlayers();
    }

    public List<Question> getQuestions() {
        return this.questions;
    }

    public List<Reward> getRewards() {
        return this.rewards;
    }

    public ActiveQuestion getCurrentQuestion() {
        return this.currentQuestion;
    }

    public QuizScoreboard getScoreboard() {
        return this.scoreboard;
    }

    public void addQuestion(Question question) {
        this.questions.add(question);
        this.saveQuestions();
    }

    public void addReward(Reward reward) {
        this.rewards.add(reward);
        this.saveRewards();
    }

    public void removeQuestion(String questionText) {
        this.questions.removeIf(question -> question.getQuestion().equals(questionText));
        this.saveQuestions();
    }

    public void removeReward(String itemId) {
        this.rewards.removeIf(reward -> reward.getItemId().equals(itemId));
        this.saveRewards();
    }

    public void reload() {
        this.loadConfig();
        this.stopAutoQuestionTimer();
        this.startAutoQuestionTimer();
        this.currentQuestion = null;
        this.answeredPlayers.clear();
        this.updateScoreboardForAllPlayers();
    }

    private static /* synthetic */ void lambda$broadcastMessage$0(Component component, ServerPlayer player) {
        player.sendSystemMessage(component);
    }
}

