/*
 *     Highly configurable PaperDoll mod. Forked from Extra Player Renderer.
 *     Copyright (C) 2024-2025  LucunJi(Original author), HappyRespawnanchor
 *
 *     This file is part of Ayame PaperDoll.
 *
 *     Ayame PaperDoll is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Ayame PaperDoll is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with Ayame PaperDoll.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.ayamemc.ayamepaperdoll.config;

import ;
import com.google.common.collect.ImmutableList;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_342;
import net.minecraft.class_3532;
import net.minecraft.class_357;
import net.minecraft.class_364;
import net.minecraft.class_4185;
import net.minecraft.class_6379;
import net.minecraft.class_7842;
import net.minecraft.class_7919;
import net.minecraft.client.gui.components.*;
import org.ayamemc.ayamepaperdoll.AyamePaperDoll;
import org.ayamemc.ayamepaperdoll.config.model.ConfigOption;
import org.ayamemc.ayamepaperdoll.config.model.RangedConfigOption;
import org.ayamemc.ayamepaperdoll.config.view.ListWidget;
import org.ayamemc.ayamepaperdoll.config.view.Retextured;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

public class ConfigWidgetRegistry {
    public static final int RESET_BUTTON_WIDTH = 50;
    public static final ConfigWidgetRegistry DEFAULT = new ConfigWidgetRegistry();
    private static final String ON_LANGKEY = "config.%s.on".formatted(AyamePaperDoll.MOD_ID);
    private static final String OFF_LANGKEY = "config.%s.off".formatted(AyamePaperDoll.MOD_ID);
    private static final String RESET_LANGKEY = "config.%s.reset".formatted(AyamePaperDoll.MOD_ID);
    private static final int WIDGET_WIDTH = 150;
    private static final int WIDGET_HEIGHT = 20;
    private final List<OptionWidgetEntry<?, ?>> defaultWidgets = new ArrayList<>();

    private ConfigWidgetRegistry() {
        defaultWidgets.add(new OptionWidgetEntry<>(Boolean.class, ConfigOption.class, ConfigWidgetRegistry::getOnOffButton));
        defaultWidgets.add(new OptionWidgetEntry<>(Enum.class, ConfigOption.class, ConfigWidgetRegistry::getEnumCycleButton));
        defaultWidgets.add(new OptionWidgetEntry<>(Double.class, RangedConfigOption.class, ConfigWidgetRegistry::getDoubleSlider));
        defaultWidgets.add(new OptionWidgetEntry<>(Integer.class, RangedConfigOption.class, ConfigWidgetRegistry::getIntegerSlider));
        defaultWidgets.add(new OptionWidgetEntry<>(String.class, ConfigOption.class, ConfigWidgetRegistry::getStringField));
    }

    private static <T> class_4185 getResetButton(ConfigOption<T> option, ConfigValueUpdater<T> setter) {
        var reset = new ConfigButton(RESET_BUTTON_WIDTH, WIDGET_HEIGHT, class_2561.method_43471(RESET_LANGKEY), self -> setter.setValue(option.getDefaultValue()));
        reset.field_22763 = !option.isValueDefault();
        return reset;
    }

    private static ListWidget.ListEntry getStringField(ConfigOption<String> option) {
        var setter = new ConfigValueUpdater<String>();
        var field = new ConfigTextField(WIDGET_WIDTH, WIDGET_HEIGHT);
        field.method_1852(option.getValue());
        field.method_1863(setter::setValue);
        var reset = getResetButton(option, setter);
        setter.setUpdateHandler(val -> {
            option.setValue(val);
            field.method_1852(option.getValue());
            reset.field_22763 = !option.isValueDefault();
        });
        return new ConfigEntry(option.getName(), class_7919.method_47407(option.getDescription()), field, reset);
    }

    private static ListWidget.ListEntry getDoubleSlider(RangedConfigOption<Double> option) {
        var setter = new ConfigValueUpdater<Double>();
        var slider = new ConfigSlider(
                WIDGET_WIDTH, WIDGET_HEIGHT,
                (option.getValue() - option.getMin()) / (option.getMax() - option.getMin()),
                val -> setter.setValue(class_3532.method_16436(val, option.getMin(), option.getMax())),
                val -> class_2561.method_30163("%.2f".formatted(option.getValue())));
        var reset = getResetButton(option, setter);
        setter.setUpdateHandler(val -> {
            option.setValue(val);
            slider.setValueNoEvent((option.getValue() - option.getMin()) / (option.getMax() - option.getMin()));
            reset.field_22763 = !option.isValueDefault();
        });
        return new ConfigEntry(option.getName(), class_7919.method_47407(option.getDescription()), slider, reset);
    }

    private static ListWidget.ListEntry getIntegerSlider(RangedConfigOption<Integer> option) {
        var setter = new ConfigValueUpdater<Integer>();
        double steps = option.getMax() - option.getMin();
        var slider = new ConfigSlider(
                WIDGET_WIDTH, WIDGET_HEIGHT,
                (option.getValue() - option.getMin()) / steps,
                val -> setter.setValue((int) Math.round(val * steps) + option.getMin()),
                val -> class_2561.method_30163(Integer.toString(option.getValue())));
        var reset = getResetButton(option, setter);
        setter.setUpdateHandler(val -> {
            option.setValue(val);
            slider.setValueNoEvent((option.getValue() - option.getMin()) / steps);
            reset.field_22763 = !option.isValueDefault();
        });
        return new ConfigEntry(option.getName(), class_7919.method_47407(option.getDescription()), slider, reset);
    }

    private static ListWidget.ListEntry getOnOffButton(ConfigOption<Boolean> option) {
        var setter = new ConfigValueUpdater<Boolean>();
        var btn = new ConfigButton(
                WIDGET_WIDTH, WIDGET_HEIGHT,
                class_2561.method_43471(option.getValue() ? ON_LANGKEY : OFF_LANGKEY),
                self -> setter.setValue(!option.getValue()));
        var reset = getResetButton(option, setter);
        setter.setUpdateHandler(val -> {
            option.setValue(val);
            btn.method_25355(class_2561.method_43471(option.getValue() ? ON_LANGKEY : OFF_LANGKEY));
            reset.field_22763 = !option.isValueDefault();
        });
        return new ConfigEntry(option.getName(), class_7919.method_47407(option.getDescription()), btn, reset);
    }

    private static <T extends Enum<?>> ListWidget.ListEntry getEnumCycleButton(ConfigOption<T> option) {
        var setter = new ConfigValueUpdater<T>();
        var values = option.getType().getEnumConstants();
        var translations = new class_2561[values.length];
        for (int i = 0; i < values.length; i++) {
            translations[i] = class_2561.method_43471("config.%s.enum.%s.%s".formatted(AyamePaperDoll.MOD_ID, option.getType().getSimpleName(), values[i].name()));
        }
        var btn = new ConfigButton(
                WIDGET_WIDTH, WIDGET_HEIGHT,
                translations[option.getValue().ordinal()],
                self -> setter.setValue(values[(option.getValue().ordinal() + 1) % values.length]));
        var reset = getResetButton(option, setter);
        setter.setUpdateHandler(val -> {
            option.setValue(val);
            btn.method_25355(translations[option.getValue().ordinal()]);
            reset.field_22763 = !option.isValueDefault();
        });
        return new ConfigEntry(option.getName(), class_7919.method_47407(option.getDescription()), btn, reset);
    }

    public <T, U extends ConfigOption<T>> Optional<ListWidget.ListEntry> getConfigEntry(U option) {
        for (var entry : defaultWidgets) {
            if (entry.type.isAssignableFrom(option.getType())) {
                //noinspection unchecked
                var entryCast = (OptionWidgetEntry<T, U>) entry;
                return Optional.of(entryCast.widgetProvider.apply(option));
            }
        }
        return Optional.empty();
    }

    record OptionWidgetEntry<T, U extends ConfigOption<T>>(Class<T> type, Class<U> optionType,
                                                           Function<U, ListWidget.ListEntry> widgetProvider) {
    }

    public static class ConfigButton extends class_4185 implements Retextured {
        public ConfigButton(int width, int height, class_2561 message, class_4241 action) {
            super(0, 0, width, height, message, action, field_40754);
        }

        @Override
        public class_2960 retexture(class_2960 oldTexture) {
            return AyamePaperDoll.path(oldTexture.method_12832());
        }
    }

    private static class ConfigTextField extends class_342 implements Retextured {
        public ConfigTextField(int width, int height) {
            super(class_310.method_1551().field_1772, width, height, class_2561.method_43473());
        }

        @Override
        public class_2960 retexture(class_2960 oldTexture) {
            return AyamePaperDoll.path(oldTexture.method_12832());
        }
    }

    private static class ConfigSlider extends class_357 implements Retextured {
        @NotNull
        private final Consumer<Double> changeListener;
        @NotNull
        private final Function<Double, class_2561> messageProvider;

        public ConfigSlider(int width, int height, double value,
                            @NotNull Consumer<Double> changeListener, @NotNull Function<Double, class_2561> messageProvider) {
            super(0, 0, width, height, class_2561.method_43473(), value);
            this.changeListener = changeListener;
            this.messageProvider = messageProvider;
            this.method_25346();
        }

        @Override
        public class_2960 retexture(class_2960 oldTexture) {
            return AyamePaperDoll.path(oldTexture.method_12832());
        }

        @Override
        protected void method_25344() {
            this.changeListener.accept(this.field_22753);
        }

        @Override
        protected void method_25346() {
            this.method_25355(this.messageProvider.apply(this.field_22753));
        }

        public void setValueNoEvent(double value) {
            this.field_22753 = value;
            this.method_25346();
        }
    }

    /**
     * Mediates operations of widgets, and decouples them from data models.
     */
    private static class ConfigValueUpdater<T> {
        private Consumer<T> updateHandler = null;
        private boolean preventUpdates = false;

        public void setUpdateHandler(Consumer<T> updateHandler) {
            this.updateHandler = updateHandler;
        }

        public void setValue(T newValue) {
            if (this.preventUpdates || this.updateHandler == null) return;
            this.preventUpdates = true;
            this.updateHandler.accept(newValue);
            this.preventUpdates = false;
        }
    }

    private static class ConfigEntry extends ListWidget.ListEntry {
        private static final int LABEL_Y_OFFSET = 7;
        private static final int GAP_WIDTH = 10;

        private final List<class_339> children;
        private final class_7842 label;
        private final class_339 widget;
        private final class_4185 reset;

        private ConfigEntry(class_2561 label, class_7919 labelTooltip, class_339 widget, class_4185 resetButton) {
            var labelWidget = new class_7842(label, class_310.method_1551().field_1772);
            labelWidget.method_47400(labelTooltip);
            this.label = labelWidget;
            this.widget = widget;
            this.reset = resetButton;
            this.children = ImmutableList.of(this.widget, this.reset);
        }

        @Override
        public List<? extends class_6379> method_37025() {
            return this.children;
        }

        @Override
        public List<? extends class_364> method_25396() {
            return this.children;
        }

        @Override
        public void method_25343(class_332 guiGraphics, int mouseX, int mouseY, boolean hovered, float tickDelta) {
            label.method_48229(method_73380(), method_73382() + LABEL_Y_OFFSET);
            reset.method_48229(method_73380() + method_73387() - reset.method_25368(), method_73382());
            widget.method_48229(method_73380()+ method_73387() - reset.method_25368() - widget.method_25368() - GAP_WIDTH,method_73382());
            widget.method_25394(guiGraphics, mouseX, mouseY, tickDelta);
            reset.method_25394(guiGraphics, mouseX, mouseY, tickDelta);
            label.method_25394(guiGraphics, mouseX, mouseY, tickDelta);
        }
    }

}
