package net.myitian.no_caves;

import net.minecraft.core.Holder;
import net.minecraft.util.Tuple;
import net.minecraft.world.level.levelgen.DensityFunction;
import net.minecraft.world.level.levelgen.DensityFunctions;
import net.minecraft.world.level.levelgen.NoiseChunk;
import net.minecraft.world.level.levelgen.synth.NormalNoise;
import net.myitian.no_caves.config.Config;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Function;
import java.util.function.Predicate;

public final class DensityFunctionCaveCleaner {
    private static final Map<String, Tuple<Predicate<DensityFunction>, Function<DensityFunction, DensityFunction>>> customTransformerRegistry = new LinkedHashMap<>();

    /**
     * @return a mutable map that preserves insertion order
     */
    public static Map<String, Tuple<Predicate<DensityFunction>, Function<DensityFunction, @Nullable DensityFunction>>> getCustomTransformerRegistry() {
        return customTransformerRegistry;
    }

    /**
     * @return null if the original DensityFunction is a cave DensityFunction,
     * otherwise the original DensityFunction will be transformed and returned.
     * @apiNote Some in-place transformations will be done, so the original DensityFunction may change!
     */
    @Nullable
    public static DensityFunction transform(DensityFunction original) {
        if (!customTransformerRegistry.isEmpty()) {
            for (var entry : customTransformerRegistry.entrySet()) {
                String id = entry.getKey();
                try {
                    var pair = entry.getValue();
                    if (pair != null
                            && pair.m_14418_() != null
                            && pair.m_14419_() != null
                            && pair.m_14418_().test(original)) {
                        return pair.m_14419_().apply(original);
                    }
                } catch (Exception e) {
                    throw new RuntimeException("An unhandled exception occurred in transformer " + id, e);
                }
            }
        } else if (original instanceof DensityFunctions.HolderHolder holder) {
            return transformRegistryEntryHolder(holder);
        } else if (original instanceof DensityFunctions.Noise noise) {
            return transformNoise(noise);
        } else if (original instanceof DensityFunctions.RangeChoice rangeChoice) {
            return transformRangeChoice(rangeChoice);
        } else if (original instanceof DensityFunctions.MulOrAdd linear) {
            return transformLinear(linear);
        } else if (original instanceof DensityFunctions.TwoArgumentSimpleFunction binary) {
            return transformBinary(binary);
        } else if (original instanceof DensityFunctions.PureTransformer unary) {
            return transformUnary(unary);
        } else if (original instanceof DensityFunctions.MarkerOrMarked wrapper) {
            return transformWrapper(wrapper);
        } else if (original instanceof DensityFunctions.TransformerWithContext positional) {
            return transformPositional(positional);
        }
        return original;
    }

    public static boolean isCaveDensityFunction(DensityFunction densityFunction) {
        return densityFunction instanceof DensityFunctions.HolderHolder holder
                && isCaveDensityFunction(holder.f_208636_());
    }

    public static boolean isCaveDensityFunction(Holder<DensityFunction> densityFunction) {
        return densityFunction instanceof Holder.Reference<DensityFunction> reference
                && Config.getDensityFunctionCavePatterns().matches(reference.m_205785_().m_135782_().toString());
    }

    public static boolean isCaveNoise(DensityFunction.NoiseHolder noise) {
        return isCaveNoise(noise.f_223997_());
    }

    public static boolean isCaveNoise(Holder<NormalNoise.NoiseParameters> noise) {
        return noise instanceof Holder.Reference<NormalNoise.NoiseParameters> reference
                && Config.getNoiseCavePatterns().matches(reference.m_205785_().m_135782_().toString());
    }

    @Nullable
    public static DensityFunction transformRegistryEntryHolder(DensityFunctions.HolderHolder holder) {
        Holder<DensityFunction> function = holder.f_208636_();
        if (Config.isEnableDensityFunctionCaveFilter() && isCaveDensityFunction(function)) {
            return null;
        } else if (function instanceof Holder.Direct<DensityFunction>) {
            return transform(function.m_203334_());
        }
        return holder;
    }

    @Nullable
    public static DensityFunction transformNoise(DensityFunctions.Noise noise) {
        return Config.isEnableNoiseCaveFilter() && isCaveNoise(noise.f_208787_()) ? null : noise;
    }

    @Nullable
    private static DensityFunction transformRangeChoice(DensityFunctions.RangeChoice rangeChoice) {
        DensityFunction input = rangeChoice.f_208823_();
        if (isCaveDensityFunction(input)) {
            // Might not cover all cases, but good enough for vanilla worldgen json.
            return transform(rangeChoice.f_208827_());
        } else if (input instanceof DensityFunctions.Constant constant) {
            double v = constant.f_208607_();
            if (v >= rangeChoice.f_208824_() && v < rangeChoice.f_208825_()) {
                return transform(rangeChoice.f_208826_());
            } else {
                return transform(rangeChoice.f_208827_());
            }
        } else {
            DensityFunction originalChild1 = rangeChoice.f_208826_();
            DensityFunction transformedChild1 = transform(originalChild1);
            DensityFunction originalChild2 = rangeChoice.f_208827_();
            DensityFunction transformedChild2 = transform(originalChild2);
            if (transformedChild1 == null) {
                return transformedChild2;
            } else if (transformedChild2 == null) {
                return transformedChild1;
            } else {
                rangeChoice.f_208826_ = transformedChild1;
                rangeChoice.f_208827_ = transformedChild2;
                return rangeChoice;
            }
        }
    }

    private static DensityFunction transformLinear(DensityFunctions.MulOrAdd linear) {
        DensityFunction originalChild = linear.m_207305_();
        DensityFunction transformedChild = transform(originalChild);
        if (transformedChild == null) {
            return DensityFunctions.m_208264_(linear.f_208750_());
        } else if (transformedChild instanceof DensityFunctions.Constant constant) {
            return DensityFunctions.m_208264_(linear.m_207382_(constant.f_208607_()));
        } else {
            linear.f_208747_ = transformedChild;
            return linear;
        }
    }

    @Nullable
    private static DensityFunction transformBinary(DensityFunctions.TwoArgumentSimpleFunction binary) {
        DensityFunction originalChild1 = binary.m_207185_();
        DensityFunction transformedChild1 = transform(originalChild1);
        DensityFunction originalChild2 = binary.m_207190_();
        DensityFunction transformedChild2 = transform(originalChild2);
        if (transformedChild1 == null) {
            return transformedChild2;
        } else if (transformedChild2 == null) {
            return transformedChild1;
        } else if (transformedChild1 instanceof DensityFunctions.Constant constant1
                && transformedChild2 instanceof DensityFunctions.Constant constant2) {
            double v1 = constant1.f_208607_();
            switch (binary.m_207119_()) {
                case ADD -> {
                    return DensityFunctions.m_208264_(v1 + constant2.f_208607_());
                }
                case MUL -> {
                    return DensityFunctions.m_208264_(v1 == 0.0 ? 0.0 : v1 * constant2.f_208607_());
                }
                case MIN -> {
                    return DensityFunctions.m_208264_(Math.min(v1, constant2.f_208607_()));
                }
                case MAX -> {
                    return DensityFunctions.m_208264_(Math.max(v1, constant2.f_208607_()));
                }
                default -> {
                }
            }
        }
        if (originalChild1 != transformedChild1 || originalChild2 != transformedChild2) {
            return DensityFunctions.TwoArgumentSimpleFunction.m_209073_(binary.m_207119_(), transformedChild1, transformedChild2);
        }
        return binary;
    }

    @Nullable
    private static DensityFunction transformUnary(DensityFunctions.PureTransformer unary) {
        DensityFunction originalChild = unary.m_207305_();
        DensityFunction transformedChild = transform(originalChild);
        if (transformedChild == null) {
            return null;
        } else if (transformedChild instanceof DensityFunctions.Constant constant) {
            return DensityFunctions.m_208264_(unary.m_207382_(constant.f_208607_()));
        } else if (transformedChild != originalChild) {
            if (unary instanceof DensityFunctions.Clamp typedUnary) {
                typedUnary.f_208584_ = transformedChild;
            } else if (unary instanceof DensityFunctions.Mapped typedUnary) {
                return DensityFunctions.Mapped.m_208671_(typedUnary.f_208654_(), transformedChild);
            }
            // Unknown types will be left as is
        }
        return unary;
    }

    @Nullable
    private static DensityFunction transformWrapper(DensityFunctions.MarkerOrMarked wrapper) {
        DensityFunction originalChild = wrapper.m_207056_();
        DensityFunction transformedChild = transform(originalChild);
        if (transformedChild == null) {
            return null;
        } else if (transformedChild != originalChild) {
            if (wrapper instanceof NoiseChunk.Cache2D typedWrapper) {
                typedWrapper.f_209284_ = transformedChild;
            } else if (wrapper instanceof NoiseChunk.Cache2D typedWrapper) {
                typedWrapper.f_209284_ = transformedChild;
            } else if (wrapper instanceof NoiseChunk.CacheOnce typedWrapper) {
                typedWrapper.f_209310_ = transformedChild;
            } else if (wrapper instanceof NoiseChunk.CacheAllInCell typedWrapper) {
                typedWrapper.f_209297_ = transformedChild;
            } else if (wrapper instanceof NoiseChunk.NoiseInterpolator typedWrapper) {
                typedWrapper.f_188830_ = transformedChild;
            } else if (wrapper instanceof NoiseChunk.FlatCache typedWrapper) {
                typedWrapper.f_209326_ = transformedChild;
            } else if (wrapper instanceof DensityFunctions.Marker typedWrapper) {
                typedWrapper.f_208706_ = transformedChild;
            }
            // Unknown types will be left as is
        }
        return wrapper;
    }

    @Nullable
    private static DensityFunction transformPositional(DensityFunctions.TransformerWithContext positional) {
        DensityFunction originalChild = positional.m_207189_();
        DensityFunction transformedChild = transform(originalChild);
        if (transformedChild == null) {
            return null;
        } else if (transformedChild != originalChild) {
            if (positional instanceof DensityFunctions.BlendDensity typedPositional) {
                typedPositional.f_208546_ = transformedChild;
            } else if (positional instanceof DensityFunctions.WeirdScaledSampler typedPositional) {
                typedPositional.f_208425_ = transformedChild;
            }
            // Unknown types will be left as is
        }
        return positional;
    }
}
