package io.github.diiiaz.better_hammers.api;

// Copied and/or modified from: "https://github.com/Draylar/magna/blob/1.20.1/src/main/java/dev/draylar/magna/api/MagnaTool.java"

import io.github.diiiaz.better_hammers.item.HammerItem;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1922;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2680;
import net.minecraft.class_3965;

/**
 * Represents a tool that should break in a certain radius.
 * <p>
 * Implementers can either make a custom tool using this interface,
 * or use one of the provided base item classes,
 * {@link HammerItem}.
 */
public interface HammerTool {

    /**
     * Returns the base breaking radius of this {@link HammerTool}.
     * <p>
     * The full area of a break is calculated with: 1 + (2 * getRadius()), or:
     *   - 3x3 for a radius of 1
     *   - 5x5 for a radius of 2
     *   - 7x7 for a radius of 3
     * and so on.
     * Tools that have a dynamic radius can either change their radius based on {@link class_1799},
     * or add a listener to {@link ToolRadiusCallback}.
     *
     * @param stack  current {@link HammerTool} stack being used
     * @return       breaking radius of stack
     */
    int getRadius(class_1799 stack);

    /**
     * Returns the depth of this {@link HammerTool}.
     * <p>
     * Unlike radius, which defines the number of perpendicular blocks broken, around a central point, to the player,
     *   depth is how far back it goes. A depth of 0 (default) means a single layer is broken, while a depth of 3 with a
     *   radius of 3 represents a 3x3 cube with the origin 1 block away from the central block being broken.
     *
     * @param stack   current {@link HammerTool} stack being used
     * @return        breaking depth of stack
     */
    default int getDepth(class_1799 stack) {
        return 0;
    }

    /**
     * Modifies which block will be considered the center of the radius.
     * <p>
     * This is useful for tools with a big radius to avoid breaking blocks under the player.
     *
     * @param world           world the block is breaking
     * @param player          player that is breaking
     * @param blockHitResult  raycast result from where the player is looking to the block being mined
     * @param toolStack       {@link HammerTool} currently being held by the player
     * @return                a {@link class_2338} that will define the center of the radius
     */
    default class_2338 getCenterPosition(class_1937 world, class_1657 player, class_3965 blockHitResult, class_1799 toolStack) {
        return blockHitResult.method_17777();
    }

    /**
     * Defines behavior about how this {@link HammerTool} should process block drops.
     * <p>
     * This is useful for mechanics such as auto-smelt or removing stacks that shouldn't be dropped while using a certain tool.
     *
     * @param world      world the stack is being dropped in
     * @param player     player that caused the stack to drop
     * @param pos        position of the block dropping the stack
     * @param heldStack  {@link HammerTool} currently being held by the player
     * @return           a {@link BlockProcessor} that defines information about how this tool should process dropped items
     */
    default BlockProcessor getProcessor(class_1937 world, class_1657 player, class_2338 pos, class_1799 heldStack) {
        return (tool, input) -> input;
    }

    /**
     * Returns whether the block at the given position in the given world is a valid breaking target for this tool.
     * <p>
     * Whether a block is valid for breaking is a rough definition of effectiveness mixed with speed and tool requirements.
     *
     * @param view   world to look in
     * @param pos    position to look at
     * @param stack  stack to try to break with
     * @return       whether the stack is roughly effective on the given location
     */
    default boolean isBlockValidForBreaking(class_1922 view, class_2338 pos, class_1799 stack) {
        class_2680 blockState = view.method_8320(pos);

        if (blockState.method_26214(view, pos) <= 0.0) { return false; }
        if (stack.method_7951(blockState)) { return true; }
        if (blockState.method_29291()) { return false; }

        return stack.method_7924(blockState) > 1.0F;
    }

    /**
     * Provides simple functionality for tools attempting to break blocks in a certain radius.
     * <p>
     * Before breaking, config options are checked, hardness is checked, effectiveness is checked,
     * and radius events are triggered.
     *
     * @param world        world to attempt to break blocks in
     * @param pos          center position to break at
     * @param player       player breaking the blocks
     * @param breakRadius  radius to break blocks in, 1 is 3x3, 2 is 5x5
     * @return             whether the break was successful
     */
    default boolean attemptBreak(class_1937 world, class_2338 pos, class_1657 player, int breakRadius, BlockProcessor processor) {
        class_1799 stack = player.method_6047();
        if (ignoreRadiusBreak(stack, player) || !isBlockValidForBreaking(world, pos, stack)) { return false; }

        // only do a 3x3 break if the player's tool is effective on the block they are breaking
        // this makes it so breaking gravel doesn't break nearby stone
        int radius = ToolRadiusCallback.EVENT.invoker().getRadius(stack, breakRadius);
        int depth = getDepth(stack);

        // break blocks
        BlockBreaker.breakInRadius(world, player, radius, depth, getBlockFinder(), (view, breakPos) -> isBlockValidForBreaking(view, breakPos, stack), processor, true);
        return true;

    }

    /**
     * Returns {@code true} if this {@link HammerTool} is NOT allowed to break in a larger radius based on player & global config state.
     *
     * <p>
     * Implementers should call {@code super} when overriding this method unless they intend to change tool sneak functionality.
     *
     * @param stack main hand stack of player
     * @param player player using this {@link HammerTool}
     * @return {@code true} if this {@link HammerTool} is NOT allowed to break bonus blocks with respect to config/player
     */
    default boolean ignoreRadiusBreak(class_1799 stack, class_1657 player) {
        return false;
    }

    /**
     * Provides the {@link BlockFinder} that determines the valid positions to break.
     * @return the {@link BlockFinder} that this tool uses
     */
    default BlockFinder getBlockFinder() {
        return BlockFinder.DEFAULT;
    }
}