package arm32x.minecraft.commandblockide.server.function;

import arm32x.minecraft.commandblockide.mixin.server.DirectoryResourcePackAccessor;
import arm32x.minecraft.commandblockide.mixin.server.FunctionLoaderAccessor;
import com.mojang.datafixers.util.Either;
import com.mojang.serialization.DataResult;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3258;
import net.minecraft.class_3259;
import net.minecraft.class_3264;
import net.minecraft.class_4239;
import net.minecraft.server.MinecraftServer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class FunctionIO {
    /**
     * Loads a function from the filesystem.
     *
     * @param server The Minecraft server the function is loaded in.
     * @param functionId The namespaced ID of the function to save.
     * @return Either the lines read from the {@code .mcfunction} file, or a
     *         feedback message to show to the user indicating what went wrong.
     */
    public static Either<List<String>, class_2561> loadFunction(MinecraftServer server, class_2960 functionId) {
        // TODO: Use proper error handling instead of returning Text.
        // TODO: Make loading functions use a CompletableFuture so a loading
        //       screen can be shown.

        // Convert the function ID ('some_datapack:some_function') to a resource
        // path ('some_datapack:functions/some_function.mcfunction').
        var resourceFinder = FunctionLoaderAccessor.getResourceFinder();
        var functionResourcePath = resourceFinder.method_45112(functionId);

        // Figure out which resource pack the function is in.
        var resourceManager = server.method_34864();
        var functionResource = resourceManager.method_14486(functionResourcePath);
        if (functionResource.isEmpty()) {
            return Either.right(class_2561.method_43469("commandBlockIDE.loadFunction.failed.noResourcePack", functionId));
        }
        var pack = functionResource.get().method_45304();

        // Only directory-based resource packs are supported.
        if (pack instanceof class_3258) {
            return Either.right(class_2561.method_43469("commandBlockIDE.loadFunction.failed.zipNotSupported", functionId).method_27692(class_124.field_1061));
        } else if (!(pack instanceof class_3259)) {
            return Either.right(class_2561.method_43469("commandBlockIDE.loadFunction.failed.packClassNotSupported", functionId, pack.getClass().getSimpleName()).method_27692(class_124.field_1061));
        }
        var directoryPack = (class_3259)pack;

        // Get the path to the function resource in the filesystem.
        DataResult<Path> pathResult = getFilesystemPathOfResource(directoryPack, class_3264.field_14190, functionResourcePath);
        if (pathResult.result().isEmpty()) {
            String errorMessage = pathResult.error().get().message();
            return Either.right(class_2561.method_43469("commandBlockIDE.loadFunction.failed.invalidPath", functionId, functionResourcePath, errorMessage));
        }
        Path path = pathResult.result().get();

        // Read the content of the mcfunction file.
        try {
            return Either.left(Files.readAllLines(path));
        } catch (IOException e) {
            LOGGER.error("IO exception occurred while loading function '" + functionId.toString() + "':", e);
            return Either.right(class_2561.method_43469("commandBlockIDE.loadFunction.failed.ioException", functionId).method_27692(class_124.field_1061));
        }
    }

    /**
     * Saves a function to the filesystem.
     *
     * @param server The Minecraft server the function is loaded in.
     * @param functionId The namespaced ID of the function to save.
     * @param lines The lines to write into the {@code .mcfunction} file.
     * @return The feedback message to show to the user.
     */
    public static class_2561 saveFunction(MinecraftServer server, class_2960 functionId, List<String> lines) {
        // TODO: Use proper error handling instead of returning Text.
        // TODO: Make saving functions use a CompletableFuture so errors can be
        //       properly shown to the user.

        // Convert the function ID ('some_datapack:some_function') to a resource
        // path ('some_datapack:functions/some_function.mcfunction').
        var resourceFinder = FunctionLoaderAccessor.getResourceFinder();
        var functionResourcePath = resourceFinder.method_45112(functionId);

        // Figure out which resource pack the function is in.
        var resourceManager = server.method_34864();
        var functionResource = resourceManager.method_14486(functionResourcePath);
        if (functionResource.isEmpty()) {
            // Error saving function '...': Not found in any datapack.
            return class_2561.method_43469("commandBlockIDE.saveFunction.failed.noResourcePack", functionId);
        }
        var pack = functionResource.get().method_45304();

        // Only directory-based resource packs are supported.
        if (pack instanceof class_3258) {
            return class_2561.method_43469("commandBlockIDE.saveFunction.failed.zipNotSupported", functionId).method_27692(
                class_124.field_1061);
        } else if (!(pack instanceof class_3259)) {
            return class_2561.method_43469("commandBlockIDE.saveFunction.failed.packClassNotSupported", functionId, pack.getClass().getSimpleName()).method_27692(class_124.field_1061);
        }
        var directoryPack = (class_3259)pack;

        // Get the path to the function resource in the filesystem.
        DataResult<Path> pathResult = getFilesystemPathOfResource(directoryPack, class_3264.field_14190, functionResourcePath);
        if (pathResult.result().isEmpty()) {
            String errorMessage = pathResult.error().get().message();
            // Error saving function '...': Invalid path '...': ...
            return class_2561.method_43469("commandBlockIDE.saveFunction.failed.invalidPath", functionId, functionResourcePath, errorMessage);
        }
        Path path = pathResult.result().get();

        // Replace the content of the mcfunction file.
        try {
            Files.write(path, lines, StandardOpenOption.TRUNCATE_EXISTING);
        } catch (IOException e) {
            LOGGER.error("IO exception occurred while saving function '" + functionId.toString() + "':", e);
            return class_2561.method_43469("commandBlockIDE.saveFunction.failed.ioException", functionId).method_27692(class_124.field_1061);
        }

        return class_2561.method_43469("commandBlockIDE.saveFunction.success.file", functionId);
    }

    /**
     * Determines the filesystem path to a resource in a directory-based
     * resource pack.
     *
     * <p>This code implements the same logic as DirectoryResourcePack.open,
     * except without opening the file at the end. If an error occurs, it is
     * returned as a {@link DataResult}.</p>
     *
     * <p>Since {@code DataResult} is a part of the open-source DataFixerUpper,
     * it should be more stable between updates than other Minecraft code.</p>
     *
     * @param pack The resource pack containing the resource.
     * @param resourceType Whether the resource is from a client-side resource
     *                     pack or server-side datapack.
     * @param resourcePath The path to the resource inside the resource pack.
     * @return A filesystem path to the same resource as {@code resourcePath}.
     */
    @SuppressWarnings("SameParameterValue")
    private static DataResult<Path> getFilesystemPathOfResource(class_3259 pack, class_3264 resourceType, class_2960 resourcePath) {
        Path root = ((DirectoryResourcePackAccessor)pack).getRoot();
        Path namespaceDir = root.resolve(resourceType.method_14413()).resolve(resourcePath.method_12836());

        return class_4239.method_46346(resourcePath.method_12832())
            .map(segments -> class_4239.method_46344(namespaceDir, segments));
    }

    private static final Logger LOGGER = LogManager.getLogger();
}
