package arm32x.minecraft.commandblockide.util;

import java.util.Iterator;
import java.util.NoSuchElementException;
import net.minecraft.class_2540;
import io.netty.buffer.ByteBuf;
import org.jetbrains.annotations.NotNull;

/**
 * Splits a {@link class_2540} into several chunks to avoid size limitations.
 */
public final class PacketSplitter implements Iterable<ByteBuf> {
	public static final int CHUNK_SIZE = 32500;
	public static final int HEADER_MAGIC = 1397771337 /* SPLI */;

	private final class_2540 source;

	public PacketSplitter(class_2540 source) {
		this.source = source;
	}

	/**
	 * Writes a {@code PacketSplitter} header to the provided {@link
	 * class_2540}. This should be written to the {@code PacketByteBuf}
	 * first, before any other data.
	 * @param buf The {@code PacketByteBuf} to write the header to.
	 */
	public static void writeHeader(class_2540 buf) {
		buf.method_53002(HEADER_MAGIC);
		buf.method_53002(0);
	}

	/**
	 * Updates the chunk count in the {@code PacketSplitter} header of the
	 * provided {@link class_2540}.
	 * @param buf The {@code PacketByteBuf} to update the chunk count of. It
	 *            must contain a header written by {@link
	 *            PacketSplitter#writeHeader(class_2540)}.
	 * @throws MissingHeaderException if {@code buf} does not contain a {@code
	 *         PacketSplitter} header.
	 */
	public static void updateChunkCount(class_2540 buf) {
		int start = buf.readerIndex();
		if (buf.readInt() != HEADER_MAGIC) {
			throw new MissingHeaderException();
		}
		int chunkCount = (int)Math.ceil(buf.readableBytes() / (double)CHUNK_SIZE);
		buf.method_52989(start + 4, chunkCount);
		buf.method_52988(start);
	}

	@Override
	public @NotNull Iterator<ByteBuf> iterator() {
		if (source.getInt(source.readerIndex()) != HEADER_MAGIC) {
			throw new MissingHeaderException();
		}
		return new BufferIterator();
	}

	private class BufferIterator implements Iterator<ByteBuf> {
		@Override
		public boolean hasNext() {
			return source.readableBytes() > 0;
		}

		@Override
		public ByteBuf next() {
			int length = Math.min(CHUNK_SIZE, source.readableBytes());
			if (length <= 0) {
				throw new NoSuchElementException();
			}
			return source.readSlice(length);
		}
	}

	public static class MissingHeaderException extends RuntimeException {
		@Override
		public String getMessage() {
			return "Missing PacketSplitter header.";
		}
	}
}
