# Fishing Journal Book System

A flexible, template-based UI system for creating paginated content in the Fishing Journal using owo-ui.

## Table of Contents

- [Overview](#overview)
- [Architecture](#architecture)
- [Quick Start: Creating a New Page](#quick-start-creating-a-new-page)
- [Content Types](#content-types)
- [File Structure](#file-structure)
- [Creating a BookPage](#creating-a-bookpage)
- [Creating a BookEntry](#creating-a-bookentry)
- [Creating a BookIndex](#creating-a-bookindex)
- [Troubleshooting](#troubleshooting)

---

## Overview

The book system separates **structure** (templates) from **content** (component definitions) from **data** (Java binding):

- **Templates** (`templates/*.xml`) - Define wrapper structure (sizing, padding, navigation)
- **Content XML** (`pages/content/*.xml`) - Define components (labels, images, containers)
- **Java Classes** - Bind data to components using `childById()`

### Benefits

✅ Single source of truth for page structure
✅ Create new pages by writing XML + minimal Java
✅ Easy to maintain consistent styling
✅ Hot-reloadable XML during development

---

## Architecture

### Three Content Types

```
BookContent (interface)
├── BookPage     - Single static page (e.g., Player Stats)
├── BookEntry    - Multi-page with navigation (e.g., Fish Collection Grid)
└── BookIndex    - Scrollable list menu (e.g., Table of Contents)
```

### Render Flow

```
1. Load Template XML    → Get wrapper structure (140x180, padding, etc.)
2. Load Content XML     → Get component definitions (labels, images)
3. Inject Content       → Add content to template
4. Bind Data (Java)     → Find components by ID, set text/images
```

---

## Quick Start: Creating a New Page

Want to add a new page showing fish rarity statistics? Here's how:

### Step 1: Create Content XML

**File:** `pages/content/fish_rarity_stats.xml`

```xml
<?xml version="1.0" encoding="UTF-8"?>
<owo-ui xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
        xsi:noNamespaceSchemaLocation="https://raw.githubusercontent.com/wisp-forest/owo-lib/1.20/owo-ui.xsd">
    <templates>
        <template name="content">
            <flow-layout direction="vertical">
                <sizing>
                    <horizontal method="content"/>
                    <vertical method="content"/>
                </sizing>

                <children>
                    <label id="title">
                        <text>Rarity Stats</text>
                        <color>#000000</color>
                        <margins><top>5</top><bottom>5</bottom></margins>
                    </label>

                    <label id="common-count">
                        <text>Common: 0</text>
                        <color>#000000</color>
                        <margins><bottom>5</bottom></margins>
                    </label>

                    <label id="rare-count">
                        <text>Rare: 0</text>
                        <color>#000000</color>
                        <margins><bottom>5</bottom></margins>
                    </label>
                </children>
            </flow-layout>
        </template>
    </templates>
</owo-ui>
```

### Step 2: Create Java Class

**File:** `src/client/java/.../pages/FishRarityStatsPage.java`

```java
public class FishRarityStatsPage extends BookPage {
    private final JournalData journalData;

    public FishRarityStatsPage(JournalData journalData) {
        super("fishing_journal/pages/content/fish_rarity_stats");
        this.journalData = journalData;
    }

    @Override
    protected void buildContent() {
        // Get components from content XML
        LabelComponent commonLabel = contentRoot.childById(LabelComponent.class, "common-count");
        LabelComponent rareLabel = contentRoot.childById(LabelComponent.class, "rare-count");

        // Calculate stats
        long commonCount = journalData.caughtFishIds().stream()
            .filter(id -> FishDictionary.get(id).getRarity() == FishRarity.COMMON)
            .count();

        long rareCount = journalData.caughtFishIds().stream()
            .filter(id -> FishDictionary.get(id).getRarity() == FishRarity.RARE)
            .count();

        // Bind data
        commonLabel.text(Text.literal("Common: " + commonCount));
        rareLabel.text(Text.literal("Rare: " + rareCount));
    }
}
```

### Step 3: Wire It Up

In `FishingJournalScreen.java`, add your page to a section:

```java
private void showStatsSection() {
    FishRarityStatsPage rarityStats = new FishRarityStatsPage(journalData);
    setLeftContent(rarityStats);
}
```

Done! 🎉

---

## Content Types

### 1. BookPage

**Use when:** You need a single static page with no internal navigation.

**Examples:** Player Stats, Basic Fish Info, Settings

**Key methods:**
- Constructor: Pass content XML path to `super()`
- `buildContent()`: Use `contentRoot.childById()` to get components and bind data

**Template used:** `templates/page.xml` (automatically loaded)

### 2. BookEntry

**Use when:** You need multiple pages with forward/back navigation.

**Examples:** Fish Collection Grid (48 fish per page), Catch History

**Key methods:**
- `getPageCount()`: Return total number of pages
- `buildPage(pageIndex, container)`: Build content for specific page

**Template used:** `templates/entry.xml` (automatically loaded, includes prev/next buttons)

### 3. BookIndex

**Use when:** You need a scrollable menu/table of contents.

**Examples:** Fish species list, Achievement categories

**Key methods:**
- `getIndexItems()`: Return list of `IndexItem(id, displayName)`
- `onItemClick(itemId)`: Handle item selection

**Template used:** `templates/index.xml` (automatically loaded, includes scroll container)

---

## File Structure

```
fishing_journal/
├── templates/
│   ├── page.xml        # 140x180 container with padding
│   ├── entry.xml       # Container + prev/next buttons
│   └── index.xml       # Scrollable list container
│
├── pages/
│   └── content/
│       ├── player_stats.xml      # Component definitions
│       ├── basic_info.xml
│       └── [your_page].xml       # ← Add new content XMLs here
│
└── entries/
    └── collection_entry.xml
```

**Java structure:**
```
fishingjournal/
├── content/
│   ├── BookContent.java     # Base interface
│   ├── BookPage.java        # Single page base
│   ├── BookEntry.java       # Multi-page base
│   └── BookIndex.java       # Index/menu base
│
├── pages/
│   ├── PlayerStatsPage.java
│   ├── BasicInfoPage.java
│   └── [YourPage].java      # ← Add new page classes here
│
└── entries/
    └── CollectionEntry.java
```

---

## Creating a BookPage

### Complete Example: Player Stats Page

#### Content XML (`pages/content/player_stats.xml`)

```xml
<template name="content">
    <flow-layout direction="vertical">
        <children>
            <label id="title">
                <text>Player Stats</text>
                <color>#000000</color>
            </label>

            <flow-layout id="favorite-fish-image" direction="vertical">
                <sizing><horizontal method="content"/></sizing>
            </flow-layout>

            <label id="total-catches">
                <text>Total Catches: 0</text>
                <color>#000000</color>
            </label>

            <label id="total-time">
                <text>Total Time: 0s</text>
                <color>#000000</color>
            </label>
        </children>
    </flow-layout>
</template>
```

#### Java Class

```java
public class PlayerStatsPage extends BookPage {
    private final JournalData journalData;

    public PlayerStatsPage(JournalData journalData) {
        super("fishing_journal/pages/content/player_stats"); // ← Content XML path
        this.journalData = journalData;
    }

    @Override
    protected void buildContent() {
        // 1. Get components from content XML
        FlowLayout fishImageContainer = contentRoot.childById(FlowLayout.class, "favorite-fish-image");
        LabelComponent totalCatchesLabel = contentRoot.childById(LabelComponent.class, "total-catches");
        LabelComponent totalTimeLabel = contentRoot.childById(LabelComponent.class, "total-time");

        // 2. Null check
        if (fishImageContainer == null || totalCatchesLabel == null || totalTimeLabel == null) {
            Hooked.LOGGER.error("Could not find required components");
            return;
        }

        // 3. Calculate data
        int totalCatches = journalData.catchHistory().size();
        long totalMillis = journalData.catchHistory().stream()
            .mapToLong(CatchRecord::fishingDuration)
            .sum();

        // 4. Bind data to components
        totalCatchesLabel.text(Text.literal("Total Catches: " + totalCatches));
        totalTimeLabel.text(Text.literal("Total Time: " + formatTime(totalMillis)));

        // 5. Add dynamic content (images, etc.)
        fishImageContainer.clearChildren();
        var fishImage = Components.texture(fishTexture, 0, 0, 16, 16, 16, 16);
        fishImageContainer.child(fishImage);
    }
}
```

### Key Points

- **`super(xmlPath)`**: Pass your content XML path (relative to `assets/hooked/owo_ui/`)
- **`contentRoot.childById()`**: Find components by their `id` attribute
- **Always null-check**: Components might be missing if XML has errors
- **`update()` is automatic**: Called when fish selection changes

---

## Creating a BookEntry

### Example: Paginated Fish Collection

```java
public class CollectionEntry extends BookEntry {
    private static final int FISH_PER_PAGE = 48;
    private final List<FishData> allFish;

    public CollectionEntry(JournalData journalData) {
        // No super() call needed - template loaded automatically
        this.allFish = new ArrayList<>(FishDictionary.getAll().values());
    }

    @Override
    protected int getPageCount() {
        return (int) Math.ceil((double) allFish.size() / FISH_PER_PAGE);
    }

    @Override
    protected void buildPage(int pageIndex, FlowLayout container) {
        // Clear previous page
        container.clearChildren();

        // Calculate which fish to display
        int startIndex = pageIndex * FISH_PER_PAGE;
        int endIndex = Math.min(startIndex + FISH_PER_PAGE, allFish.size());

        // Create grid
        GridLayout fishGrid = Containers.grid(Sizing.fixed(140), Sizing.fixed(140), 8, 6);

        // Add fish to grid
        for (int i = startIndex; i < endIndex; i++) {
            FishData fish = allFish.get(i);
            var fishSlot = createFishSlot(fish); // Your helper method
            fishGrid.child(fishSlot, col, row);
        }

        container.child(fishGrid);
    }
}
```

### How It Works

- Template automatically provides prev/next buttons
- `refreshPage()` is called automatically on button click
- You just implement `buildPage()` to fill the content container

---

## Creating a BookIndex

### Example: Fish Species Menu

```java
public class FishSpeciesIndex extends BookIndex {
    private final Consumer<String> onSpeciesSelected;

    public FishSpeciesIndex(Consumer<String> onSpeciesSelected) {
        // No super() call needed - template loaded automatically
        this.onSpeciesSelected = onSpeciesSelected;
    }

    @Override
    protected List<IndexItem> getIndexItems() {
        // Return list of menu items
        return List.of(
            new IndexItem("freshwater", "Freshwater Fish"),
            new IndexItem("saltwater", "Saltwater Fish"),
            new IndexItem("rare", "Rare Species")
        );
    }

    @Override
    protected void onItemClick(String itemId) {
        // Handle selection
        onSpeciesSelected.accept(itemId);
    }
}
```

### How It Works

- Template provides scrollable list container
- Buttons are automatically created from `IndexItem` list
- Click handlers route to `onItemClick()`

---

## Troubleshooting

### ❌ "Could not find required components in content XML"

**Cause:** Component ID mismatch between XML and Java

**Fix:** Check that `id="my-component"` in XML matches `childById(... , "my-component")` in Java

### ❌ Hot reload error: `NoSuchFileException`

**Cause:** Old XML file path cached by hot reload

**Solution:** Restart the game to clear file watchers

### ❌ Components not showing up

**Checklist:**
1. Did you call `pageRoot.child(component)` or add it to container?
2. Is sizing correct? (`Sizing.content()` vs `Sizing.fixed()`)
3. Check margins - component might be offscreen

### ❌ "Template not found" error

**Cause:** XML path is incorrect

**Fix:** Paths are relative to `assets/hooked/owo_ui/`:
- ✅ `"fishing_journal/pages/content/my_page"`
- ❌ `"pages/content/my_page"` (missing `fishing_journal/`)
- ❌ `"/fishing_journal/pages/content/my_page"` (no leading slash)

### 🔍 Debugging Tips

**Enable verbose logging:**
```java
Hooked.LOGGER.info("Component found: {}", component != null);
```

**Check XML syntax:**
- All tags must be closed
- IDs must be unique within the template
- Color format: `#000000` (hex) or `#FF000000` (ARGB)

**Test incrementally:**
1. Create minimal XML with one label
2. Verify it loads
3. Add more components one at a time

---

## Common Patterns

### Dynamic Images

```java
fishImageContainer.clearChildren(); // Clear old images first
Identifier texture = Identifier.of("hooked", "textures/item/fish/" + fishId + ".png");
var image = Components.texture(texture, 0, 0, 16, 16, 16, 16)
    .sizing(Sizing.fixed(32));
fishImageContainer.child(image);
```

### Conditional Content

```java
if (fishData != null) {
    nameLabel.text(Text.translatable(fishData.getTranslationKey()));
} else {
    nameLabel.text(Text.literal("Unknown Fish"));
}
```

### Click Handlers

```java
itemComponent.mouseDown().subscribe((mouseX, mouseY, button) -> {
    onFishClicked.accept(fishId);
    return true; // Consume event
});
```

---

## Additional Resources

- **owo-ui Documentation:** https://docs.wispforest.io/owo/ui/
- **Example Pages:** See `PlayerStatsPage.java`, `BasicInfoPage.java`
- **Example Entry:** See `CollectionEntry.java`

---

**Questions?** Check the existing pages in `src/client/java/.../fishingjournal/pages/` for more examples.
