import os
import sys
from PIL import Image
import numpy as np

def find_sprites(image_path):
    """Find all sprite regions in an image based on non-transparent pixels"""
    
    # Load image
    img = Image.open(image_path).convert('RGBA')
    width, height = img.size
    pixels = np.array(img)
    
    print(f"Image: {os.path.basename(image_path)}")
    print(f"Dimensions: {width} x {height} pixels\n")
    
    # Get alpha channel
    alpha = pixels[:, :, 3]
    
    # Track visited pixels
    visited = np.zeros((height, width), dtype=bool)
    sprites = []
    
    def flood_fill(start_x, start_y):
        """Flood fill to find connected non-transparent region"""
        stack = [(start_x, start_y)]
        min_x = max_x = start_x
        min_y = max_y = start_y
        
        while stack:
            x, y = stack.pop()
            
            if x < 0 or x >= width or y < 0 or y >= height:
                continue
            if visited[y, x]:
                continue
            if alpha[y, x] == 0:  # transparent
                continue
                
            visited[y, x] = True
            min_x = min(min_x, x)
            max_x = max(max_x, x)
            min_y = min(min_y, y)
            max_y = max(max_y, y)
            
            # Check 4-directional neighbors
            stack.extend([(x+1, y), (x-1, y), (x, y+1), (x, y-1)])
        
        return {
            'x': min_x,
            'y': min_y,
            'width': max_x - min_x + 1,
            'height': max_y - min_y + 1
        }
    
    # Find all sprites
    for y in range(height):
        for x in range(width):
            if not visited[y, x] and alpha[y, x] > 0:
                sprite = flood_fill(x, y)
                sprites.append(sprite)
    
    # Sort sprites by position (top to bottom, left to right)
    sprites.sort(key=lambda s: (s['y'], s['x']))
    
    return sprites

def main():
    # Check if image path provided as argument
    if len(sys.argv) > 1:
        image_path = sys.argv[1]
        if not os.path.exists(image_path):
            print(f"Error: File '{image_path}' not found!")
            return
    else:
        # Look for image files in current directory
        current_dir = os.getcwd()
        print(f"Searching in: {current_dir}\n")
        
        valid_extensions = ('.png', '.jpg', '.jpeg', '.gif', '.bmp', '.webp')
        image_files = [f for f in os.listdir('.') if f.lower().endswith(valid_extensions)]
        
        if not image_files:
            print("No image files found in current directory!")
            print("\nUsage:")
            print("  python sprite_mapper.py                    # Search current directory")
            print("  python sprite_mapper.py path/to/image.png  # Specify image path")
            print("\nSupported formats: .png, .jpg, .jpeg, .gif, .bmp, .webp")
            return
        
        # If multiple images, let user choose or process first one
        if len(image_files) > 1:
            print("Multiple images found:")
            for i, f in enumerate(image_files, 1):
                print(f"  {i}. {f}")
            print(f"\nProcessing: {image_files[0]}")
            print("(Specify filename as argument to analyze a different image)\n")
        
        image_path = image_files[0]
    
    # Find sprites
    try:
        sprites = find_sprites(image_path)
    except Exception as e:
        print(f"Error processing image: {e}")
        return
    
    # Print results
    print("="*50)
    print("UV MAPPINGS")
    print("="*50)
    
    for i, sprite in enumerate(sprites, 1):
        print(f"\nSprite {i}:")
        print(f"  Position: ({sprite['x']}, {sprite['y']})")
        print(f"  Size: {sprite['width']} x {sprite['height']} pixels")
    
    print(f"\n{'='*50}")
    print(f"Total sprites found: {len(sprites)}")

if __name__ == "__main__":
    main()