package net.nullved.pmweatherapi.network;

import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.neoforge.network.PacketDistributor;
import net.neoforged.neoforge.network.handling.IPayloadHandler;
import net.neoforged.neoforge.network.registration.PayloadRegistrar;
import net.nullved.pmweatherapi.radar.storage.RadarServerStorage;

import java.util.function.BiConsumer;
import java.util.function.Function;

/**
 * The base networking class for PMWeatherAPI
 * @since 0.14.15.3
 */
public class PMWNetworking {
    /**
     * Registers all of PMWeatherAPI's packets. Should not be called
     * @param args The arguments to pass
     * @since 0.14.15.3
     */
    public static void register(Object... args) {
        registerClientboundPacket(S2CRadarPacket.TYPE, S2CRadarPacket.STREAM_CODEC, S2CRadarPacket::handle, args);
        registerClientboundPacket(S2CMetarPacket.TYPE, S2CMetarPacket.STREAM_CODEC, S2CMetarPacket::handle, args);
        registerClientboundPacket(S2CWSRPacket.TYPE, S2CWSRPacket.STREAM_CODEC, S2CWSRPacket::handle, args);
    }

    /**
     * Registers a serverbound packet payload
     * @param type The {@link CustomPacketPayload.Type} of the packet
     * @param codec The {@link StreamCodec} for the packet
     * @param handler The handle method for the packet
     * @param args Additional arguments. args[0] should be the {@link PayloadRegistrar}
     * @param <T> A packet extending {@link CustomPacketPayload}
     * @since 0.14.15.3
     */
    public static <T extends CustomPacketPayload> void registerServerboundPacket(CustomPacketPayload.Type<T> type, StreamCodec<? super RegistryFriendlyByteBuf, T> codec, BiConsumer<T, Player> handler, Object... args) {
        PayloadRegistrar registrar = (PayloadRegistrar) args[0];
        IPayloadHandler<T> serverHandler = (pkt, ctx) -> ctx.enqueueWork(() -> handler.accept(pkt, ctx.player()));
        registrar.playToServer(type, codec, serverHandler);
    }

    /**
     * Registers a clientbound packet payload
     * @param type The {@link CustomPacketPayload.Type} of the packet
     * @param codec The {@link StreamCodec} for the packet
     * @param handler The handle method for the packet
     * @param args Additional arguments. args[0] should be the {@link PayloadRegistrar}
     * @param <T> A packet extending {@link CustomPacketPayload}
     * @since 0.14.15.3
     */
    public static <T extends CustomPacketPayload> void registerClientboundPacket(CustomPacketPayload.Type<T> type, StreamCodec<? super RegistryFriendlyByteBuf, T> codec, BiConsumer<T, Player> handler, Object... args) {
        PayloadRegistrar registrar = (PayloadRegistrar) args[0];
        IPayloadHandler<T> clientHandler = (pkt, ctx) -> ctx.enqueueWork(() -> handler.accept(pkt, ctx.player()));
        registrar.playToClient(type, codec, clientHandler);
    }

    /**
     * Sends all clients a {@link S2CStoragePacket}. The type of packet is dependent on the caller of this method
     * @param tag The tag to send, generated by the {@link RadarServerStorage}
     * @since 0.15.3.3
     */
    public static void serverSendStorageToAll(CompoundTag tag, Function<CompoundTag, S2CStoragePacket<?>> pkt) {
        PacketDistributor.sendToAllPlayers(pkt.apply(tag));
    }

    /**
     * Sends a {@link S2CStoragePacket} to a specific player. The type of packet is dependent on the caller of this method
     * @param tag The tag to send with the packet
     * @since 0.15.3.3
     */
    public static void serverSendStorageToPlayer(CompoundTag tag, Function<CompoundTag, S2CStoragePacket<?>> pkt, Player player) {
        PacketDistributor.sendToPlayer((ServerPlayer) player, pkt.apply(tag));
    }
}
