package net.nullved.pmweatherapi.network;

import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.neoforge.network.PacketDistributor;
import net.neoforged.neoforge.network.handling.IPayloadHandler;
import net.neoforged.neoforge.network.registration.PayloadRegistrar;
import net.nullved.pmweatherapi.PMWeatherAPI;

import java.util.function.BiConsumer;

/**
 * The base networking class for PMWeatherAPI
 * @since 0.14.15.3
 */
public class PMWNetworking {
    public static final ResourceLocation RADARS_ID = PMWeatherAPI.rl("radars");

    /**
     * Registers all of PMWeatherAPI's packets. Should not be called
     * @param args The arguments to pass
     * @since 0.14.15.3
     */
    public static void register(Object... args) {
        registerClientboundPacket(S2CRadarsPacket.TYPE, S2CRadarsPacket.STREAM_CODEC, S2CRadarsPacket::handle, args);
    }

    /**
     * Registers a serverbound packet payload
     * @param type The {@link CustomPacketPayload.Type} of the packet
     * @param codec The {@link StreamCodec} for the packet
     * @param handler The handle method for the packet
     * @param args Additional arguments. args[0] should be the {@link PayloadRegistrar}
     * @param <T> A packet extending {@link CustomPacketPayload}
     * @since 0.14.15.3
     */
    public static <T extends CustomPacketPayload> void registerServerboundPacket(CustomPacketPayload.Type<T> type, StreamCodec<? super RegistryFriendlyByteBuf, T> codec, BiConsumer<T, Player> handler, Object... args) {
        PayloadRegistrar registrar = (PayloadRegistrar) args[0];
        IPayloadHandler<T> serverHandler = (pkt, ctx) -> ctx.enqueueWork(() -> handler.accept(pkt, ctx.player()));
        registrar.playToServer(type, codec, serverHandler);
    }

    /**
     * Registers a clientbound packet payload
     * @param type The {@link CustomPacketPayload.Type} of the packet
     * @param codec The {@link StreamCodec} for the packet
     * @param handler The handle method for the packet
     * @param args Additional arguments. args[0] should be the {@link PayloadRegistrar}
     * @param <T> A packet extending {@link CustomPacketPayload}
     * @since 0.14.15.3
     */
    public static <T extends CustomPacketPayload, B extends FriendlyByteBuf> void registerClientboundPacket(CustomPacketPayload.Type<T> type, StreamCodec<? super RegistryFriendlyByteBuf, T> codec, BiConsumer<T, Player> handler, Object... args) {
        PayloadRegistrar registrar = (PayloadRegistrar) args[0];
        IPayloadHandler<T> clientHandler = (pkt, ctx) -> ctx.enqueueWork(() -> handler.accept(pkt, ctx.player()));
        registrar.playToClient(type, codec, clientHandler);
    }

    /**
     * Sends all radars to all clients
     * @param tag The tag to send, generated by the {@link net.nullved.pmweatherapi.radar.RadarServerStorage}
     * @since 0.14.15.3
     */
    public static void serverSendRadarsToAll(CompoundTag tag) {
        PacketDistributor.sendToAllPlayers(new S2CRadarsPacket(tag));
    }

    /**
     * Sends all radars to a specific player
     * @param tag The tag to send, generated by the {@link net.nullved.pmweatherapi.radar.RadarServerStorage}
     * @since 0.14.15.3
     */
    public static void serverSendRadarsToPlayer(CompoundTag tag, Player player) {
        PacketDistributor.sendToPlayer((ServerPlayer) player, new S2CRadarsPacket(tag));
    }
}
