package me.emafire003.dev.particleanimationlib.effects;

import me.emafire003.dev.particleanimationlib.EffectType;
import me.emafire003.dev.particleanimationlib.effects.base.TargetedEffect;
import me.emafire003.dev.particleanimationlib.util.EffectModifier;
import net.minecraft.class_1297;
import net.minecraft.class_2394;
import net.minecraft.class_2398;
import net.minecraft.class_243;
import net.minecraft.class_3218;

@SuppressWarnings("unused")
public class ArcEffect extends TargetedEffect {


    /**
     * Height of the arc in blocks
     */
    public float height = 2;

    /**
     * Particles per arc
     */
    public int particles = 100;


    /**
     * Internal counter
     */
    protected int step = 0;

    /**
     * Creates a new arc effect between two points
     *
     * @param world The world the particles are going to spawn in
     * @param particle The particle effect that is going to be spawned. You can use {@link class_2398}
     * @param origin The origin position of the effect, aka the initial point of the arc
     * @param target The target position of the effect, aka the finial point of the arc
     * @param count The number of particles to spread between the two points
     * @param height The height (in blocks) of the arc, aka its curvature.
     */
    public ArcEffect(class_3218 world, class_2394 particle, class_243 origin, class_243 target, int count, float height) {
        super(world, EffectType.REPEATING, particle, origin);
        this.setTargetPos(target);
        this.particles = count;
        this.height = height;
    }

    /**
     * Creates a new arc effect between two points
     *
     * @param world The world the particles are going to spawn in
     * @param particle The particle effect that is going to be spawned. You can use {@link class_2398}
     * @param origin The origin position of the effect, aka the initial point of the arc
     * @param target The target position of the effect, aka the finial point of the arc
     */
    public ArcEffect(class_3218 world, class_2394 particle, class_243 origin, class_243 target) {
        super(world, EffectType.REPEATING, particle, origin);
        this.setTargetPos(target);
    }

    private ArcEffect(Builder builder) {
        super(builder.world, EffectType.REPEATING, builder.particle, builder.originPos);
        setIterations(builder.iterations);
        setOriginPos(builder.originPos);
        setUpdatePositions(builder.updatePositions);
        setEntityOrigin(builder.entityOrigin);
        setOriginOffset(builder.originOffset);
        world = builder.world;
        particle = builder.particle;
        setHeight(builder.height);
        setParticles(builder.particles);
        setTargetPos(builder.targetPos);
        setUpdateTargetPositions(builder.updateTargetPositions);
        setEntityTarget(builder.entityTarget);
        setTargetOffset(builder.targetOffset);
        setUseEyePosAsOrigin(builder.useEyePosAsOrigin);
        setUseEyePosAsTarget(builder.useEyePosAsTarget);
        setExecuteOnStop(builder.executeOnStop);
        setParticleLimit(builder.particleLimit);
        setShouldSpawnParticlesEveryNIteration(builder.shouldSpawnParticlesEveryNIteration);
        setSpawnParticlesEveryNIteration(builder.spawnParticlesEveryNIteration);
        setShouldLimitParticlesSpawnedPerIteration(builder.shouldLimitParticlesSpawnedPerIteration);
        setShouldLimitParticlesEveryNIterations(builder.shouldLimitParticlesEveryNIterations);
        setLimitParticlesEveryNIterations(builder.limitParticlesEveryNIterations);
        setForced(builder.forced);
    }

    public static void copy(ArcEffect original, ArcEffect copy) {
        TargetedEffect.copy(original, copy);
        copy.setHeight(original.getHeight());
        copy.setParticles(original.getParticles());
        copy.step = original.step;
    }

    /**
     * Creates a new arc effect between two points
     *
     * @param world The world the particles are going to spawn in
     * @param particle The particle effect that is going to be spawned. You can use {@link class_2398}
     * @param origin The origin position of the effect, aka the initial point of the arc
     * @param target The target position of the effect, aka the finial point of the arc
     * @param count The number of particles to spread between the two points
     */
    public ArcEffect(class_3218 world, class_2394 particle, class_243 origin, class_243 target, int count) {
        super(world, EffectType.REPEATING, particle, origin);
        this.setTargetPos(target);
        this.particles = count;
    }



    /** Returns a builder for the effect.
     *
     * @param world The world the particles are going to spawn in
     * @param particle The particle effect that is going to be spawned. You can use {@link class_2398}
     * @param originPos The origin position of the effect
     *<p>
     *  Setting a world, a particle effect and an origin position is ALWAYS mandatory, hence their presence in this method!
     * If this is an effect that uses Yaw and Pitch, remember to set those as well!
     * */
    public static Builder builder(class_3218 world, class_2394 particle, class_243 originPos) {
        return new Builder().world(world).particle(particle).originPos(originPos);
    }


    @Override
    public void onRun() {
        class_243 origin = this.getOriginPos();
        class_243 target = this.getTargetPos();

        if (target == null) {
            return;
        }

        if (origin == null) {
            return;
        }

        class_243 link = target.method_1020(origin);
        float length = (float) link.method_1033();
        float pitch = (float) (4 * height / Math.pow(length, 2));

        class_243 v;
        float x;
        float y;

        for (int i = 0; i < particles; i++) {

            step++;
            v = new class_243(link.method_10216(), link.method_10214(), link.method_10215()).method_1029().method_1021(length * i / particles);
            //ParticleAnimationLib.LOGGER.info("The v is: " + v);
            x = ((float) i / particles) * length - length / 2;
            y = (float) (-pitch * Math.pow(x, 2) + height);

            this.displayParticle(particle, origin.method_1019(v).method_1031(0, y, 0));

            step++;
        }
    }

    public float getHeight() {
        return height;
    }

    public void setHeight(float height) {
        this.height = height;
    }

    public int getParticles() {
        return particles;
    }

    public void setParticles(int particles) {
        this.particles = particles;
    }

    /**
     * {@code ArcEffect} builder static inner class.
     */
    public static final class Builder {
        private int iterations;
        private class_243 originPos;
        private boolean updatePositions;
        private class_1297 entityOrigin;
        private class_243 originOffset;
        private class_3218 world;
        private class_2394 particle;
        /**
         * Height of the arc in blocks
         */
        private float height = 2;

        /**
         * Particles per arc
         */
        private int particles = 100;
        private class_243 targetPos;
        private boolean updateTargetPositions = true;
        private class_1297 entityTarget;
        private class_243 targetOffset;
        private boolean useEyePosAsOrigin;
        private boolean useEyePosAsTarget;
        private EffectModifier executeOnStop;
        private boolean shouldSpawnParticlesEveryNIteration = false;
        private int spawnParticlesEveryNIteration = 5;
        private boolean shouldLimitParticlesSpawnedPerIteration = true;
        private int particleLimit = 5000;
        private boolean shouldLimitParticlesEveryNIterations = false;
        private int limitParticlesEveryNIterations = 5;

        private Builder() {
        }


        /**
         * Sets the {@code iterations} and returns a reference to this Builder enabling method chaining.
         *
         * @param iterations the {@code iterations} to set
         * @return a reference to this Builder
         */
        public Builder iterations(int iterations) {
            this.iterations = iterations;
            return this;
        }

        /**
         * Sets the {@code originPos} and returns a reference to this Builder enabling method chaining.
         *
         * @param originPos the {@code originPos} to set
         * @return a reference to this Builder
         */
        public Builder originPos(class_243 originPos) {
            this.originPos = originPos;
            return this;
        }

        /**
         * Sets the {@code updatePositions} and returns a reference to this Builder enabling method chaining.
         *
         * @param updatePositions the {@code updatePositions} to set
         * @return a reference to this Builder
         */
        public Builder updatePositions(boolean updatePositions) {
            this.updatePositions = updatePositions;
            return this;
        }


        /**
         * Sets the {@code executeOnStop} and returns a reference to this Builder enabling method chaining.
         *
         * @param executeOnStop the {@code executeOnStop} to set
         * @return a reference to this Builder
         */
        public Builder executeOnStop(EffectModifier executeOnStop) {
            this.executeOnStop = executeOnStop;
            return this;
        }

        /**
         * Sets the {@code useEyePosAsOrigin} and returns a reference to this Builder enabling method chaining.
         *
         * @param useEyePos the {@code useEyePosAsOrigin} to set
         * @return a reference to this Builder
         */
        public Builder useEyePosAsOrigin(boolean useEyePos) {
            this.useEyePosAsOrigin = useEyePos;
            return this;
        }

        /**
         * Sets the {@code useEyePosAsTarget} and returns a reference to this Builder enabling method chaining.
         *
         * @param useEyePos the {@code useEyePosAsTarget} to set
         * @return a reference to this Builder
         */
        public Builder useEyePosAsTarget(boolean useEyePos) {
            this.useEyePosAsTarget = useEyePos;
            return this;
        }

        /**
         * Sets the {@code entityOrigin} and returns a reference to this Builder enabling method chaining.
         *
         * @param entityOrigin the {@code entityOrigin} to set
         * @return a reference to this Builder
         */
        public Builder entityOrigin(class_1297 entityOrigin) {
            this.entityOrigin = entityOrigin;
            return this;
        }

        /**
         * Sets the {@code originOffset} and returns a reference to this Builder enabling method chaining.
         *
         * @param originOffset the {@code originOffset} to set
         * @return a reference to this Builder
         */
        public Builder originOffset(class_243 originOffset) {
            this.originOffset = originOffset;
            return this;
        }

        /**
         * Sets the {@code world} and returns a reference to this Builder enabling method chaining.
         *
         * @param world the {@code world} to set
         * @return a reference to this Builder
         */
        public Builder world(class_3218 world) {
            this.world = world;
            return this;
        }

        /**
         * Sets the {@code particle} and returns a reference to this Builder enabling method chaining.
         *
         * @param particle the {@code particle} to set
         * @return a reference to this Builder
         */
        public Builder particle(class_2394 particle) {
            this.particle = particle;
            return this;
        }

        /**
         * Sets the {@code height} and returns a reference to this Builder enabling method chaining.
         *
         * @param height the {@code height} to set
         * @return a reference to this Builder
         */
        public Builder height(float height) {
            this.height = height;
            return this;
        }

        /**
         * Sets the {@code particles} and returns a reference to this Builder enabling method chaining.
         *
         * @param particles the {@code particles} to set
         * @return a reference to this Builder
         */
        public Builder particles(int particles) {
            this.particles = particles;
            return this;
        }

        /**
         * Sets the {@code targetPos} and returns a reference to this Builder enabling method chaining.
         *
         * @param targetPos the {@code targetPos} to set
         * @return a reference to this Builder
         */
        public Builder targetPos(class_243 targetPos) {
            this.targetPos = targetPos;
            return this;
        }

        /**
         * Sets the {@code updateTargetPositions} and returns a reference to this Builder enabling method chaining.
         *
         * @param updateTargetPositions the {@code updateTargetPositions} to set
         * @return a reference to this Builder
         */
        public Builder updateTargetPositions(boolean updateTargetPositions) {
            this.updateTargetPositions = updateTargetPositions;
            return this;
        }

        /**
         * Sets the {@code entityTarget} and returns a reference to this Builder enabling method chaining.
         *
         * @param entityTarget the {@code entityTarget} to set
         * @return a reference to this Builder
         */
        public Builder entityTarget(class_1297 entityTarget) {
            this.entityTarget = entityTarget;
            return this;
        }

        /**
         * Sets the {@code targetOffset} and returns a reference to this Builder enabling method chaining.
         *
         * @param targetOffset the {@code targetOffset} to set
         * @return a reference to this Builder
         */
        public Builder targetOffset(class_243 targetOffset) {
            this.targetOffset = targetOffset;
            return this;
        }

        /**
         * Returns a {@code ArcEffect} built from the parameters previously set.
         *
         * @return a {@code ArcEffect} built with parameters of this {@code ArcEffect.Builder}
         */
        public ArcEffect build() {
            return new ArcEffect(this);
        }

        /**
         * Sets the {@code particleLimit} and returns a reference to this Builder enabling method chaining.
         *
         * @param val the {@code particleLimit} to set
         * @return a reference to this Builder
         */
        public Builder particleLimit(int val) {
            particleLimit = val;
            return this;
        }

        /**
         * Sets the {@code shouldSpawnParticlesEveryNIteration} and returns a reference to this Builder enabling method chaining.
         *
         * @param val the {@code shouldSpawnParticlesEveryNIteration} to set
         * @return a reference to this Builder
         */
        public Builder shouldSpawnParticlesEveryNIteration(boolean val) {
            shouldSpawnParticlesEveryNIteration = val;
            return this;
        }

        /**
         * Sets the {@code spawnParticlesEveryNIteration} and returns a reference to this Builder enabling method chaining.
         *
         * @param val the {@code spawnParticlesEveryNIteration} to set
         * @return a reference to this Builder
         */
        public Builder spawnParticlesEveryNIteration(int val) {
            spawnParticlesEveryNIteration = val;
            return this;
        }

        /**
         * Sets the {@code shouldLimitParticlesSpawnedPerIteration} and returns a reference to this Builder enabling method chaining.
         *
         * @param val the {@code shouldLimitParticlesSpawnedPerIteration} to set
         * @return a reference to this Builder
         */
        public Builder shouldLimitParticlesSpawnedPerIteration(boolean val) {
            shouldLimitParticlesSpawnedPerIteration = val;
            return this;
        }

        /**
         * Sets the {@code shouldLimitParticlesEveryNIterations} and returns a reference to this Builder enabling method chaining.
         *
         * @param val the {@code shouldLimitParticlesEveryNIterations} to set
         * @return a reference to this Builder
         */
        public Builder shouldLimitParticlesEveryNIterations(boolean val) {
            shouldLimitParticlesEveryNIterations = val;
            return this;
        }

        /**
         * Sets the {@code limitParticlesEveryNIterations} and returns a reference to this Builder enabling method chaining.
         *
         * @param val the {@code limitParticlesEveryNIterations} to set
         * @return a reference to this Builder
         */
        public Builder limitParticlesEveryNIterations(int val) {
            limitParticlesEveryNIterations = val;
            return this;
        }

        private boolean forced = false;
        /**
         * Sets the {@code forced} and returns a reference to this Builder enabling method chaining.
         *
         * @param forced the {@code forced} to set
         * @return a reference to this Builder
         */
        public Builder forced(boolean forced) {
            this.forced = forced;
            return this;
        }
    }
}
