/*
 * Copyright (C) 2025  DragonsPlus
 * SPDX-License-Identifier: LGPL-3.0-or-later
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package plus.dragons.createenchantmentindustry.common.fluids.printer.behaviour;

import com.mojang.serialization.DataResult;
import com.simibubi.create.foundation.blockEntity.behaviour.fluid.SmartFluidTankBehaviour;
import com.simibubi.create.foundation.utility.CreateLang;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.minecraft.ChatFormatting;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Holder;
import net.minecraft.core.component.DataComponents;
import net.minecraft.network.chat.Component;
import net.minecraft.tags.ItemTags;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.entity.BannerPattern;
import net.minecraft.world.level.block.entity.BannerPatternLayers;
import net.neoforged.neoforge.fluids.FluidStack;
import plus.dragons.createdragonsplus.common.fluids.dye.DyeFluidType;
import plus.dragons.createenchantmentindustry.common.CEICommon;
import plus.dragons.createenchantmentindustry.common.fluids.printer.PrinterBlockEntity;
import plus.dragons.createenchantmentindustry.common.registry.CEIDataMaps;
import plus.dragons.createenchantmentindustry.config.CEIConfig;
import plus.dragons.createenchantmentindustry.util.CEILang;

public class BannerPatternPrintingBehavior implements PrintingBehaviour {
    private final SmartFluidTankBehaviour tank;
    private final Holder<BannerPattern> pattern;

    public BannerPatternPrintingBehavior(SmartFluidTankBehaviour tank, Holder<BannerPattern> pattern) {
        this.tank = tank;
        this.pattern = pattern;
    }

    public static Optional<DataResult<PrintingBehaviour>> create(Level level, SmartFluidTankBehaviour tank, ItemStack stack) {
        if (!stack.is(ItemTags.BANNERS))
            return Optional.empty();
        BannerPatternLayers layers = stack.get(DataComponents.BANNER_PATTERNS);
        if (layers.layers().isEmpty())
            return Optional.of(DataResult.error(() -> CEICommon.asLocalization("gui.printer.banner_pattern.no_pattern")));
        if (layers.layers().size() > 1)
            return Optional.of(DataResult.error(() -> CEICommon.asLocalization("gui.printer.banner_pattern.multiple_pattern")));
        return Optional.of(DataResult.success(new BannerPatternPrintingBehavior(tank, layers.layers().getFirst().pattern())));
    }

    @Override
    public int getRequiredItemCount(Level level, ItemStack stack) {
        if (stack.is(ItemTags.BANNERS)) {
            BannerPatternLayers layers = stack.get(DataComponents.BANNER_PATTERNS);
            if (layers.layers().isEmpty())
                return 1;
            if (layers.layers().getLast().pattern().value().assetId().equals(pattern.value().assetId()))
                return 0;
            return 1;
        }
        return 0;
    }

    @Override
    public int getRequiredFluidAmount(Level level, ItemStack stack, FluidStack fluidStack) {
        var cost = fluidStack.getFluidHolder().getData(CEIDataMaps.PRINTING_BANNER_PATTERN_INGREDIENT);
        return cost == null ? 0 : cost;
    }

    @Override
    public ItemStack getResult(Level level, ItemStack stack, FluidStack fluidStack) {
        BannerPatternLayers layers = stack.get(DataComponents.BANNER_PATTERNS);
        ArrayList<BannerPatternLayers.Layer> l = new ArrayList<>();
        l.addAll(layers.layers());
        l.add(new BannerPatternLayers.Layer(pattern, ((DyeFluidType) fluidStack.getFluidType()).getColor()));
        var result = stack.copy();
        result.set(DataComponents.BANNER_PATTERNS, new BannerPatternLayers(l));
        return result;
    }

    @Override
    public void onFinished(Level level, BlockPos pos, PrinterBlockEntity printer) {
        // Plays SoundEvents.BOOK_PAGE_TURN
        level.levelEvent(1043, pos.below(), 0);
    }

    @Override
    public boolean isSafeNBT() {
        return false;
    }

    @Override
    public boolean addToGoggleTooltip(List<Component> tooltip, boolean isPlayerSneaking) {
        CEILang.translate("gui.goggles.printing.banner_pattern").forGoggles(tooltip);
        var amount = tank.getPrimaryHandler().getFluid().getFluidHolder().getData(CEIDataMaps.PRINTING_BANNER_PATTERN_INGREDIENT);
        if (amount != null) {
            var p = Component.literal("→ ").append(Component.translatable(pattern.value().translationKey() + "." + ((DyeFluidType) tank.getPrimaryHandler().getFluid().getFluidType()).getColor().getName())).withStyle(ChatFormatting.GOLD);
            CEILang.builder().add(p).forGoggles(tooltip, 1);
            CEILang.translate("gui.goggles.printing.cost",
                    CEILang.number(amount)
                            .add(CreateLang.translate("generic.unit.millibuckets"))
                            .style(amount <= CEIConfig.fluids().printerFluidCapacity.get()
                                    ? ChatFormatting.GREEN
                                    : ChatFormatting.RED))
                    .forGoggles(tooltip, 1);
        } else if (!tank.getPrimaryHandler().getFluid().isEmpty()) {
            CEILang.translate("gui.goggles.printing.incorrect_liquid").style(ChatFormatting.RED).forGoggles(tooltip);
        }
        return true;
    }
}
