class ServerMonitorDashboard {
    constructor() {
        this.charts = {};
        this.dataHistory = {
            tps: [],
            cpu: [],
            ram: [],
            storage: [],
            players: [],
            timestamps: []
        };
        this.maxDataPoints = 20; // Show last 10 minutes (20 * 30s = 10min)
        this.ws = null;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 5;

        this.init();
    }

    init() {
        this.initCharts();
        this.connectWebSocket();
        this.loadInitialData();
    }

    initCharts() {
        const commonOptions = {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                x: {
                    display: false
                },
                y: {
                    beginAtZero: true,
                    grid: {
                        color: 'rgba(0,0,0,0.05)'
                    }
                }
            },
            elements: {
                point: {
                    radius: 3,
                    hoverRadius: 6
                },
                line: {
                    borderWidth: 2,
                    fill: true
                }
            },
            animation: {
                duration: 500
            }
        };

        // TPS Chart
        this.charts.tps = new Chart(document.getElementById('tpsChart'), {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'TPS',
                    data: [],
                    borderColor: '#4ecdc4',
                    backgroundColor: 'rgba(78, 205, 196, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    ...commonOptions.scales,
                    y: {
                        ...commonOptions.scales.y,
                        max: 20.5,
                        min: 0
                    }
                }
            }
        });

        // CPU Chart
        this.charts.cpu = new Chart(document.getElementById('cpuChart'), {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'CPU %',
                    data: [],
                    borderColor: '#45b7d1',
                    backgroundColor: 'rgba(69, 183, 209, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    ...commonOptions.scales,
                    y: {
                        ...commonOptions.scales.y,
                        max: 100,
                        min: 0
                    }
                }
            }
        });

        // RAM Chart
        this.charts.ram = new Chart(document.getElementById('ramChart'), {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'RAM %',
                    data: [],
                    borderColor: '#96ceb4',
                    backgroundColor: 'rgba(150, 206, 180, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    ...commonOptions.scales,
                    y: {
                        ...commonOptions.scales.y,
                        max: 100,
                        min: 0
                    }
                }
            }
        });

        // Storage Chart
        this.charts.storage = new Chart(document.getElementById('storageChart'), {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'Storage %',
                    data: [],
                    borderColor: '#feca57',
                    backgroundColor: 'rgba(254, 202, 87, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    ...commonOptions.scales,
                    y: {
                        ...commonOptions.scales.y,
                        max: 100,
                        min: 0
                    }
                }
            }
        });

        // Players Chart
        this.charts.players = new Chart(document.getElementById('playersChart'), {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'Players',
                    data: [],
                    borderColor: '#ff9ff3',
                    backgroundColor: 'rgba(255, 159, 243, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    ...commonOptions.scales,
                    y: {
                        ...commonOptions.scales.y,
                        beginAtZero: true
                    }
                }
            }
        });
    }

    connectWebSocket() {
        const protocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
        const wsUrl = `${protocol}//${window.location.host}/ws`;

        try {
            this.ws = new WebSocket(wsUrl);

            this.ws.onopen = () => {
                this.setConnectionStatus(true, 'Connected');
                this.reconnectAttempts = 0;
            };

            this.ws.onmessage = (event) => {
                try {
                    const data = JSON.parse(event.data);
                    this.updateData(data);
                } catch (e) {
                    console.error('Error parsing WebSocket data:', e);
                }
            };

            this.ws.onclose = () => {
                this.setConnectionStatus(false, 'Disconnected');
                this.attemptReconnect();
            };

            this.ws.onerror = (error) => {
                console.error('WebSocket error:', error);
                this.setConnectionStatus(false, 'Connection Error');
            };

        } catch (e) {
            console.error('Failed to create WebSocket connection:', e);
            this.setConnectionStatus(false, 'Connection Failed');
        }
    }

    attemptReconnect() {
        if (this.reconnectAttempts < this.maxReconnectAttempts) {
            this.reconnectAttempts++;
            this.setConnectionStatus(false, `Reconnecting... (${this.reconnectAttempts}/${this.maxReconnectAttempts})`);

            setTimeout(() => {
                this.connectWebSocket();
            }, 5000 * this.reconnectAttempts); // Exponential backoff
        } else {
            this.setConnectionStatus(false, 'Failed to reconnect');
        }
    }

    setConnectionStatus(connected, text) {
        const statusDot = document.getElementById('connectionStatus');
        const statusText = document.getElementById('connectionText');

        statusDot.classList.toggle('connected', connected);
        statusText.textContent = text;
    }

    async loadInitialData() {
        try {
            const response = await fetch('/api/history');
            if (response.ok) {
                const historyData = await response.json();
                historyData.forEach(data => this.updateData(data));
            }
        } catch (e) {
            console.error('Failed to load initial data:', e);
        }
    }

    updateData(data) {
        if (!data || typeof data !== 'object') return;

        // Update current values
        this.updateCurrentValues(data);

        // Add to history
        const timestamp = data.timestamp || Date.now();
        this.addDataPoint('tps', data.tps || 0, timestamp);
        this.addDataPoint('cpu', data.cpu || 0, timestamp);
        this.addDataPoint('ram', data.ramPercentage || 0, timestamp);
        this.addDataPoint('storage', data.storagePercentage || 0, timestamp);
        this.addDataPoint('players', data.onlinePlayers || 0, timestamp);

        // Update charts
        this.updateCharts();
    }

    updateCurrentValues(data) {
        // TPS
        const tpsValue = document.getElementById('tpsValue');
        const tps = data.tps || 0;
        tpsValue.textContent = tps.toFixed(1);
        this.setUsageColor(tpsValue, tps, 20, 15); // Good if >= 15, warning if < 15

        // CPU
        const cpuValue = document.getElementById('cpuValue');
        const cpu = data.cpu || 0;
        cpuValue.textContent = cpu.toFixed(1);
        this.setUsageColor(cpuValue, cpu, 100, 80); // Warning if > 80, critical if > 90

        // RAM
        const ramValue = document.getElementById('ramValue');
        const ramDetails = document.getElementById('ramDetails');
        const ramPercentage = data.ramPercentage || 0;
        const ramUsed = data.ramUsedMB || 0;
        ramValue.textContent = ramPercentage.toFixed(1);
        ramDetails.textContent = `${ramUsed.toFixed(0)} MB`;
        this.setUsageColor(ramValue, ramPercentage, 100, 85); // Warning if > 85, critical if > 95

        // Storage
        const storageValue = document.getElementById('storageValue');
        const storageDetails = document.getElementById('storageDetails');
        const storagePercentage = data.storagePercentage || 0;
        const storageUsed = data.storageUsedGB || 0;
        storageValue.textContent = storagePercentage.toFixed(1);
        storageDetails.textContent = `${storageUsed.toFixed(1)} GB`;
        this.setUsageColor(storageValue, storagePercentage, 100, 90); // Warning if > 90, critical if > 95

        // Players
        const playersValue = document.getElementById('playersValue');
        const playersUnit = document.querySelector('.stat-card:nth-child(5) .stat-unit');
        const onlinePlayers = data.onlinePlayers || 0;
        const maxPlayers = data.maxPlayers || 0;
        playersValue.textContent = onlinePlayers;
        playersUnit.textContent = `/ ${maxPlayers}`;
    }

    setUsageColor(element, value, maxValue, warningThreshold) {
        element.classList.remove('high-usage', 'medium-usage');

        if (value >= maxValue * 0.95) {
            element.classList.add('high-usage'); // Critical
        } else if (value >= maxValue * (warningThreshold / 100)) {
            element.classList.add('medium-usage'); // Warning
        }
    }

    addDataPoint(metric, value, timestamp) {
        this.dataHistory[metric].push(value);
        this.dataHistory.timestamps.push(new Date(timestamp).toLocaleTimeString());

        // Keep only the last maxDataPoints
        if (this.dataHistory[metric].length > this.maxDataPoints) {
            this.dataHistory[metric].shift();
            this.dataHistory.timestamps.shift();
        }
    }

    updateCharts() {
        Object.keys(this.charts).forEach(metric => {
            const chart = this.charts[metric];
            chart.data.labels = this.dataHistory.timestamps.slice(-this.maxDataPoints);
            chart.data.datasets[0].data = this.dataHistory[metric].slice(-this.maxDataPoints);
            chart.update('none'); // Update without animation for better performance
        });
    }
}

// Initialize the dashboard when the page loads
document.addEventListener('DOMContentLoaded', () => {
    new ServerMonitorDashboard();
});
