/*
 * Decompiled with CFR 0.152.
 */
package cm.chunkManager;

import cm.chunkManager.api.ChunkManagerAPI;
import cm.chunkManager.components.ChunkAnalyzer;
import cm.chunkManager.components.ChunkCache;
import cm.chunkManager.components.ChunkPrefetcher;
import cm.chunkManager.components.ChunkProcessor;
import cm.chunkManager.components.ConfigurationManager;
import cm.chunkManager.components.LanguageManager;
import cm.chunkManager.components.PerformanceMonitor;
import cm.chunkManager.components.RegionOptimizer;
import cm.chunkManager.components.SmartChunkUnloader;
import cm.chunkManager.components.TaskManager;
import cm.chunkManager.events.ChunkEventManager;
import cm.chunkManager.gui.GuiManager;
import cm.chunkManager.metrics.MetricsCollector;
import cm.chunkManager.shaded.bstats.bukkit.Metrics;
import cm.chunkManager.utils.ChunkUtils;
import cm.chunkManager.utils.MemoryManager;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.Chunk;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.command.Command;
import org.bukkit.command.CommandExecutor;
import org.bukkit.command.CommandSender;
import org.bukkit.command.PluginCommand;
import org.bukkit.command.TabCompleter;
import org.bukkit.entity.Player;
import org.bukkit.event.EventHandler;
import org.bukkit.event.EventPriority;
import org.bukkit.event.Listener;
import org.bukkit.event.player.PlayerMoveEvent;
import org.bukkit.event.player.PlayerQuitEvent;
import org.bukkit.event.player.PlayerTeleportEvent;
import org.bukkit.event.world.ChunkLoadEvent;
import org.bukkit.event.world.ChunkUnloadEvent;
import org.bukkit.generator.WorldInfo;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;

public class ChunkManager
extends JavaPlugin
implements Listener,
TabCompleter,
ChunkManagerAPI {
    private TaskManager taskManager;
    private ChunkCache chunkCache;
    private ConfigurationManager configManager;
    private PerformanceMonitor performanceMonitor;
    private MemoryManager memoryManager;
    private ChunkProcessor chunkProcessor;
    private SmartChunkUnloader smartUnloader;
    private LanguageManager languageManager;
    private ChunkAnalyzer chunkAnalyzer;
    private RegionOptimizer regionOptimizer;
    private ChunkPrefetcher chunkPrefetcher;
    private ChunkEventManager eventManager;
    private MetricsCollector metricsCollector;
    private GuiManager guiManager;
    private final DecimalFormat df = new DecimalFormat("#.##");
    private final Map<UUID, Integer> playerRadiusOverrides = new ConcurrentHashMap<UUID, Integer>();
    private final Map<UUID, Long> lastPlayerActivity = new ConcurrentHashMap<UUID, Long>();
    private boolean maintenanceMode;
    private boolean autoMaintenanceActive;
    private boolean manualOverride;
    private boolean advancedFeaturesEnabled;

    public void onEnable() {
        this.initializeComponents();
        this.getServer().getPluginManager().registerEvents((Listener)this, (Plugin)this);
        this.registerCommands();
        this.startTasks();
        this.setupEventListeners();
        this.guiManager = new GuiManager(this);
        this.getServer().getPluginManager().registerEvents((Listener)this.guiManager, (Plugin)this);
        new Metrics((Plugin)this, 22717);
        this.getLogger().info("ChunkManager v" + this.getDescription().getVersion() + " enabled");
        this.getLogger().info("Loaded " + this.languageManager.getAvailableLanguages().size() + " languages");
        this.getLogger().info("Advanced features: " + (this.advancedFeaturesEnabled ? "ENABLED" : "DISABLED"));
    }

    private void initializeComponents() {
        this.taskManager = new TaskManager(this);
        this.configManager = new ConfigurationManager(this);
        this.languageManager = new LanguageManager(this);
        this.chunkCache = new ChunkCache(this, this.configManager.getMaxCacheSize(), this.configManager.getCacheExpirationTime());
        this.performanceMonitor = new PerformanceMonitor(this);
        this.memoryManager = new MemoryManager(this, 1000);
        int coreThreads = Math.max(2, Runtime.getRuntime().availableProcessors() / 2);
        int maxThreads = Math.max(4, Runtime.getRuntime().availableProcessors());
        this.chunkProcessor = new ChunkProcessor(this, coreThreads, maxThreads);
        this.smartUnloader = new SmartChunkUnloader(this, this.chunkProcessor);
        this.advancedFeaturesEnabled = this.getConfig().getBoolean("advanced-features.enabled", true);
        if (this.advancedFeaturesEnabled) {
            this.chunkAnalyzer = new ChunkAnalyzer(this);
            this.regionOptimizer = new RegionOptimizer(this);
            this.chunkPrefetcher = new ChunkPrefetcher(this);
            this.eventManager = new ChunkEventManager(this);
            this.metricsCollector = new MetricsCollector(this);
            boolean exportMetrics = this.getConfig().getBoolean("performance.export-metrics", false);
            int metricsInterval = this.getConfig().getInt("performance.metrics-interval", 300);
            this.metricsCollector.startExporting(exportMetrics, metricsInterval);
        }
        this.maintenanceMode = this.configManager.isMaintenanceModeEnabled();
        if (this.maintenanceMode) {
            this.configManager.setDynamicRadius(this.configManager.getMaintenanceModeRadius());
            this.autoMaintenanceActive = true;
        }
        this.performanceMonitor.setPerformanceHandler(this::handlePerformanceChange);
    }

    private void registerCommands() {
        String[] commands;
        for (String cmdName : commands = new String[]{"chunkstatus", "chunkreload", "optimizechunks", "chunkgc", "chunksummary", "maintenance", "chunklang", "chunkanalyze", "chunkregions", "chunkmetrics", "cm"}) {
            PluginCommand cmd = this.getCommand(cmdName);
            if (cmd == null) continue;
            cmd.setExecutor((CommandExecutor)this);
            cmd.setTabCompleter((TabCompleter)this);
        }
    }

    private void startTasks() {
        this.taskManager.startTask("chunkLoader", this::processChunkQueue, 1L, 1L);
        this.taskManager.startTask("smartUnloader", this::performSmartUnload, 200L, 200L);
        this.taskManager.startTask("cacheCleaner", () -> this.chunkCache.cleanExpiredCache(), 600L, 600L);
        this.taskManager.startTask("tickUpdater", () -> this.performanceMonitor.updateTickTime(), 1L, 1L);
        this.taskManager.startTask("performanceMonitor", this::monitorPerformance, 200L, 200L);
        this.taskManager.startAsyncTask("memoryMonitor", () -> this.memoryManager.checkMemoryStatus(), 400L, 400L);
        if (this.advancedFeaturesEnabled) {
            this.taskManager.startTask("regionDecay", () -> this.regionOptimizer.decayActivityScores(), 1200L, 1200L);
            this.taskManager.startTask("analyzerCleanup", () -> this.chunkAnalyzer.cleanExpiredCache(), 6000L, 6000L);
            this.taskManager.startTask("metricsUpdate", this::updateMetrics, 100L, 100L);
        }
    }

    public void onDisable() {
        this.taskManager.cancelAllTasks();
        this.chunkCache.clearAll();
        this.memoryManager.cleanup();
        this.chunkProcessor.shutdown();
        this.smartUnloader.cleanup();
        if (this.guiManager != null) {
            this.guiManager.shutdown();
        }
        if (this.advancedFeaturesEnabled) {
            this.regionOptimizer.cleanup();
            this.chunkPrefetcher.shutdown();
            this.chunkAnalyzer.clearCache();
            this.metricsCollector.shutdown();
        }
        this.getLogger().info("ChunkManager disabled");
    }

    public boolean onCommand(CommandSender sender, Command command, String label, String[] args) {
        String cmdName = command.getName().toLowerCase(Locale.ROOT);
        if (cmdName.equals("chunklang")) {
            if (this.requireAdmin(sender)) {
                this.handleLanguageCommand(sender, args);
            }
            return true;
        }
        if (!this.requireAdmin(sender)) {
            return true;
        }
        switch (cmdName) {
            case "chunkstatus": {
                this.displayChunkInfo(sender);
                break;
            }
            case "chunkreload": {
                this.reloadConfigValues(sender);
                break;
            }
            case "optimizechunks": {
                this.optimizeChunks(sender);
                break;
            }
            case "chunkgc": {
                this.runChunkGC(sender);
                break;
            }
            case "chunksummary": {
                this.generateChunkSummary(sender);
                break;
            }
            case "maintenance": {
                this.handleMaintenanceCommand(sender, args);
                break;
            }
            case "chunkanalyze": {
                if (!this.advancedFeaturesEnabled) break;
                this.analyzeChunks(sender, args);
                break;
            }
            case "chunkregions": {
                if (!this.advancedFeaturesEnabled) break;
                this.displayRegionInfo(sender);
                break;
            }
            case "chunkmetrics": {
                if (!this.advancedFeaturesEnabled) break;
                this.displayMetrics(sender);
                break;
            }
            case "cm": {
                if (sender instanceof Player) {
                    Player p = (Player)sender;
                    this.guiManager.openMainMenu(p);
                    break;
                }
                sender.sendMessage(String.valueOf(ChatColor.RED) + "Only players can use the GUI!");
                break;
            }
            default: {
                return false;
            }
        }
        return true;
    }

    private void handleLanguageCommand(CommandSender sender, String[] args) {
        if (args.length == 0) {
            sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Current language: " + String.valueOf(ChatColor.WHITE) + this.languageManager.getCurrentLanguageName());
            sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Available languages:");
            this.languageManager.getAvailableLanguages().forEach((code, name) -> sender.sendMessage(String.valueOf(ChatColor.GRAY) + "  - " + String.valueOf(ChatColor.WHITE) + code + String.valueOf(ChatColor.GRAY) + " (" + name + ")"));
            sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "Usage: /chunklang <language-code>");
        } else {
            String langCode = args[0].toLowerCase();
            if (this.languageManager.getAvailableLanguages().containsKey(langCode)) {
                this.languageManager.setLanguage(langCode);
                this.getConfig().set("language", (Object)langCode);
                this.saveConfig();
                sender.sendMessage(String.valueOf(ChatColor.GREEN) + "Language changed to: " + this.languageManager.getCurrentLanguageName());
            } else {
                sender.sendMessage(String.valueOf(ChatColor.RED) + "Invalid language code: " + langCode);
            }
        }
    }

    private boolean requireAdmin(CommandSender sender) {
        if (!sender.hasPermission("chunkmanager.admin")) {
            sender.sendMessage(String.valueOf(ChatColor.RED) + this.languageManager.getMessage("no-permission"));
            return false;
        }
        return true;
    }

    private void handleMaintenanceCommand(CommandSender sender, String[] args) {
        if (args.length == 0) {
            this.showMaintenanceStatus(sender);
            return;
        }
        switch (args[0].toLowerCase(Locale.ROOT)) {
            case "on": {
                this.setMaintenanceMode(true, sender, false);
                break;
            }
            case "off": {
                this.setMaintenanceMode(false, sender, false);
                break;
            }
            case "status": {
                this.showMaintenanceStatus(sender);
                break;
            }
            case "auto": {
                this.toggleAutoMaintenance(sender);
                break;
            }
            default: {
                sender.sendMessage(String.valueOf(ChatColor.RED) + "Usage: /maintenance <on|off|auto|status>");
            }
        }
    }

    private void toggleAutoMaintenance(CommandSender sender) {
        this.manualOverride = !this.manualOverride;
        HashMap<String, String> placeholders = new HashMap<String, String>();
        placeholders.put("state", this.manualOverride ? "disabled" : "enabled");
        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + this.languageManager.getMessage("auto-maintenance-toggle", placeholders));
        this.configManager.saveAutoMaintenance(!this.manualOverride);
    }

    private void setMaintenanceMode(boolean enable, CommandSender sender, boolean autoMode) {
        if (this.maintenanceMode == enable) {
            this.showMaintenanceStatus(sender);
            return;
        }
        this.maintenanceMode = enable;
        if (enable) {
            this.configManager.setDynamicRadius(this.configManager.getMaintenanceModeRadius());
            this.autoMaintenanceActive = autoMode;
            HashMap<String, String> placeholders = new HashMap<String, String>();
            placeholders.put("radius", String.valueOf(this.configManager.getDynamicRadius()));
            this.sendColoredMessage(sender, this.languageManager.getMessage("maintenance-mode-enabled", placeholders));
        } else {
            this.configManager.setDynamicRadius(this.configManager.getDefaultRadius());
            if (!autoMode) {
                this.autoMaintenanceActive = false;
                this.manualOverride = true;
            }
            HashMap<String, String> placeholders = new HashMap<String, String>();
            placeholders.put("radius", String.valueOf(this.configManager.getDynamicRadius()));
            this.sendColoredMessage(sender, this.languageManager.getMessage("maintenance-mode-disabled", placeholders));
        }
        this.configManager.saveMaintenanceMode(enable);
    }

    private void sendColoredMessage(CommandSender sender, String message) {
        if (sender == Bukkit.getConsoleSender()) {
            this.getLogger().info(ChatColor.stripColor((String)message));
        } else {
            sender.sendMessage(message);
            if (this.configManager.shouldBroadcastMaintenanceChanges()) {
                Bukkit.getServer().broadcastMessage(message);
            }
        }
    }

    private void showMaintenanceStatus(CommandSender sender) {
        HashMap<String, String> placeholders = new HashMap<String, String>();
        placeholders.put("status", this.maintenanceMode ? "ENABLED" : "DISABLED");
        placeholders.put("radius", String.valueOf(this.configManager.getDynamicRadius()));
        placeholders.put("auto", this.manualOverride ? "DISABLED" : "ENABLED");
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "--- " + String.valueOf(ChatColor.AQUA) + "Maintenance Status " + String.valueOf(ChatColor.GRAY) + "---");
        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + this.languageManager.getMessage("maintenance-mode-status", placeholders));
        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + this.languageManager.getMessage("auto-maintenance-status", placeholders));
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "---------------------------");
    }

    private void reloadConfigValues(CommandSender sender) {
        boolean previousMaintenanceMode = this.maintenanceMode;
        boolean previousManualOverride = this.manualOverride;
        this.configManager.reload();
        this.languageManager.reloadLanguages();
        this.chunkCache = new ChunkCache(this, this.configManager.getMaxCacheSize(), this.configManager.getCacheExpirationTime());
        if (previousManualOverride) {
            this.maintenanceMode = previousMaintenanceMode;
            this.manualOverride = previousManualOverride;
            if (this.maintenanceMode) {
                this.configManager.setDynamicRadius(this.configManager.getMaintenanceModeRadius());
            }
        }
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("config-reloaded"));
    }

    private void displayChunkInfo(CommandSender sender) {
        Map<String, String> info = this.performanceMonitor.getPerformanceInfo();
        MemoryManager.MemoryStats memStats = this.memoryManager.getMemoryStats();
        Map<String, Object> heatStats = this.smartUnloader.getHeatMapStats();
        double tickTime = this.performanceMonitor.getAverageTickTime();
        double tps = tickTime > 50.0 ? Math.max(0.0, 20.0 - (tickTime - 50.0) / 50.0) : 20.0;
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "--- " + String.valueOf(ChatColor.AQUA) + "ChunkManager v" + this.getDescription().getVersion() + " Status " + String.valueOf(ChatColor.GRAY) + "---");
        ChatColor tpsColor = tps >= 19.0 ? ChatColor.GREEN : (tps >= 17.0 ? ChatColor.YELLOW : (tps >= 15.0 ? ChatColor.GOLD : ChatColor.RED));
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Server TPS: " + String.valueOf(tpsColor) + this.df.format(tps) + "/20.0 " + String.valueOf(ChatColor.GRAY) + "(" + this.df.format(tickTime) + "ms tick)");
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("chunks-in-queue", Collections.singletonMap("size", String.valueOf(this.chunkCache.getQueueSize()))));
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("chunks-kept-loaded", Collections.singletonMap("size", String.valueOf(this.chunkCache.getKeepLoadedSize()))));
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("online-players", Collections.singletonMap("count", info.get("online_players"))));
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("total-loaded-chunks", Collections.singletonMap("count", info.get("total_chunks"))));
        ChatColor memColor = memStats.usagePercentage > 80.0 ? ChatColor.RED : (memStats.usagePercentage > 60.0 ? ChatColor.YELLOW : ChatColor.GREEN);
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Memory: " + String.valueOf(memColor) + this.df.format(memStats.usedMB) + "/" + this.df.format(memStats.maxMB) + " MB " + String.valueOf(ChatColor.GRAY) + "(" + this.df.format(memStats.usagePercentage) + "%)");
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("dynamic-radius", Collections.singletonMap("radius", String.valueOf(this.configManager.getDynamicRadius()))));
        if (this.maintenanceMode) {
            sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "MAINTENANCE MODE ACTIVE");
        }
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Heat Map: " + String.valueOf(ChatColor.WHITE) + "Hot: " + String.valueOf(heatStats.get("hotChunks")) + ", Cold: " + String.valueOf(heatStats.get("coldChunks")) + ", Total: " + String.valueOf(heatStats.get("totalChunks")));
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Processor Queue: " + String.valueOf(ChatColor.WHITE) + this.chunkProcessor.getQueueSize() + " (Active Threads: " + this.chunkProcessor.getActiveThreads() + ")");
        if (this.advancedFeaturesEnabled) {
            sender.sendMessage(String.valueOf(ChatColor.DARK_AQUA) + "Advanced Features: " + String.valueOf(ChatColor.GREEN) + "ENABLED");
        }
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "---------------------------");
    }

    private void optimizeChunks(CommandSender sender) {
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Optimizing chunks...");
        this.taskManager.runAsyncTask("optimize", () -> {
            int totalUnloaded = 0;
            for (World world : Bukkit.getWorlds()) {
                if (this.isWorldBlacklisted(world)) continue;
                totalUnloaded += this.smartUnloader.performSmartUnload(world, 50);
            }
            int unloaded = totalUnloaded;
            PerformanceMonitor.PerformanceData perfData = this.performanceMonitor.getCurrentPerformance();
            HashMap<String, String> placeholders = new HashMap<String, String>();
            placeholders.put("unloaded_chunks", String.valueOf(unloaded));
            placeholders.put("memory_usage", this.df.format(perfData.memoryUsage));
            placeholders.put("average_tick_time", this.df.format(perfData.tickTime));
            Bukkit.getScheduler().runTask((Plugin)this, () -> {
                sender.sendMessage(String.valueOf(ChatColor.GRAY) + "--- " + String.valueOf(ChatColor.AQUA) + "Optimization Results " + String.valueOf(ChatColor.GRAY) + "---");
                sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("unloaded-chunks", placeholders));
                sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("current-memory-usage", placeholders));
                sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("current-average-tick-time", placeholders));
                sender.sendMessage(String.valueOf(ChatColor.GRAY) + "---------------------------");
            });
        });
    }

    private void runChunkGC(CommandSender sender) {
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Cleaning memory...");
        this.performanceMonitor.runGarbageCollection(results -> {
            HashMap<String, String> placeholders = new HashMap<String, String>();
            placeholders.put("before_gc", this.df.format(results.get("before")));
            placeholders.put("after_gc", this.df.format(results.get("after")));
            placeholders.put("freed_memory", this.df.format(results.get("freed")));
            sender.sendMessage(String.valueOf(ChatColor.GRAY) + "--- " + String.valueOf(ChatColor.AQUA) + "Memory Cleanup Results " + String.valueOf(ChatColor.GRAY) + "---");
            sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("memory-before-gc", placeholders));
            sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("memory-after-gc", placeholders));
            sender.sendMessage(String.valueOf(ChatColor.AQUA) + this.languageManager.getMessage("freed-memory", placeholders));
            sender.sendMessage(String.valueOf(ChatColor.GRAY) + "---------------------------");
        });
    }

    private void generateChunkSummary(CommandSender sender) {
        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Generating summary...");
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this, () -> {
            HashMap worldStats = new HashMap();
            Bukkit.getScheduler().runTask((Plugin)this, () -> {
                for (World world : Bukkit.getWorlds()) {
                    if (this.isWorldBlacklisted(world)) continue;
                    worldStats.put(world, this.performanceMonitor.gatherWorldStatistics(world));
                }
                Bukkit.getScheduler().runTaskAsynchronously((Plugin)this, () -> {
                    StringBuilder summary = new StringBuilder();
                    summary.append(ChatColor.GRAY).append("--- ").append(ChatColor.AQUA).append("World Analysis").append(ChatColor.GRAY).append(" ---\n");
                    for (World world : Bukkit.getWorlds()) {
                        if (this.isWorldBlacklisted(world)) {
                            summary.append(ChatColor.RED).append("World '").append(world.getName()).append("' is blacklisted\n\n");
                            continue;
                        }
                        if (!worldStats.containsKey(world)) continue;
                        this.appendWorldSummary(summary, world, (PerformanceMonitor.WorldStatistics)worldStats.get(world));
                    }
                    summary.append(ChatColor.GRAY).append("---------------------------");
                    Bukkit.getScheduler().runTask((Plugin)this, () -> sender.sendMessage(summary.toString()));
                });
            });
        });
    }

    private void appendWorldSummary(StringBuilder summary, World world, PerformanceMonitor.WorldStatistics stats) {
        summary.append(ChatColor.AQUA).append(this.languageManager.getMessage("world-name", Collections.singletonMap("world", world.getName()))).append("\n");
        summary.append(ChatColor.AQUA).append(this.languageManager.getMessage("loaded-chunks", Collections.singletonMap("count", String.valueOf(world.getLoadedChunks().length)))).append("\n");
        Map<String, String> placeholders = stats.toPlaceholderMap();
        summary.append(ChatColor.AQUA).append(this.languageManager.getMessage("active-chunks", placeholders)).append("\n");
        summary.append(ChatColor.AQUA).append(this.languageManager.getMessage("total-entities", placeholders)).append("\n");
        summary.append(ChatColor.AQUA).append(this.languageManager.getMessage("total-tile-entities", placeholders)).append("\n");
        summary.append(ChatColor.AQUA).append(this.languageManager.getMessage("total-players", placeholders)).append("\n\n");
    }

    @EventHandler
    public void onPlayerMove(PlayerMoveEvent event) {
        if (event.getTo() == null) {
            return;
        }
        if (event.getFrom().getChunk().equals((Object)event.getTo().getChunk())) {
            return;
        }
        Player player = event.getPlayer();
        World world = player.getWorld();
        if (!this.isWorldBlacklisted(world)) {
            this.preloadChunksAround(event.getTo(), player);
            this.lastPlayerActivity.put(player.getUniqueId(), System.currentTimeMillis());
            if (this.advancedFeaturesEnabled) {
                this.regionOptimizer.onPlayerMove(player, event.getFrom(), event.getTo());
                this.chunkPrefetcher.trackPlayerMovement(player);
                this.metricsCollector.recordMetric("player_movements", 1L);
            }
        }
    }

    @EventHandler
    public void onPlayerQuit(PlayerQuitEvent event) {
        Player player = event.getPlayer();
        this.playerRadiusOverrides.remove(player.getUniqueId());
        this.lastPlayerActivity.remove(player.getUniqueId());
        this.smartUnloader.onPlayerQuit(player);
        if (this.advancedFeaturesEnabled) {
            this.regionOptimizer.onPlayerQuit(player);
            this.chunkPrefetcher.onPlayerQuit(player);
        }
    }

    @EventHandler
    public void onChunkLoad(ChunkLoadEvent event) {
        Chunk chunk = event.getChunk();
        if (!this.isWorldBlacklisted(chunk.getWorld())) {
            this.smartUnloader.onChunkLoad(chunk);
            if (this.advancedFeaturesEnabled) {
                this.eventManager.fireLoadEvent(chunk, event.isNewChunk(), null);
                this.metricsCollector.recordMetric("chunks_loaded", 1L);
            }
        }
    }

    @EventHandler
    public void onChunkUnload(ChunkUnloadEvent event) {
        Chunk c = event.getChunk();
        if (this.isWorldBlacklisted(c.getWorld())) {
            return;
        }
        if (this.chunkCache.shouldKeepLoaded(c)) {
            this.chunkCache.removeFromKeepLoaded(c);
            this.chunkCache.addToQueue(c);
        } else if (!this.chunkCache.isCached(c)) {
            this.chunkCache.addToCache(c);
        }
        if (this.advancedFeaturesEnabled) {
            this.eventManager.fireUnloadEvent(c, ChunkEventManager.ChunkUnloadEvent.UnloadReason.AUTOMATIC);
            this.metricsCollector.recordMetric("chunks_unloaded", 1L);
        }
    }

    private void preloadChunksAround(Location location, Player player) {
        World world = location.getWorld();
        if (world == null) {
            return;
        }
        int radius = this.getEffectiveRadius(player);
        int centerX = location.getBlockX() >> 4;
        int centerZ = location.getBlockZ() >> 4;
        for (int x = -radius; x <= radius; ++x) {
            for (int z = -radius; z <= radius; ++z) {
                int chunkX = centerX + x;
                int chunkZ = centerZ + z;
                ChunkUtils.getChunkAtAsync(world, chunkX, chunkZ, this).thenAccept(chunk -> this.smartUnloader.onChunkAccess((Chunk)chunk));
            }
        }
    }

    private int getEffectiveRadius(Player player) {
        if (this.memoryManager.isLowMemoryMode()) {
            return Math.max(1, this.configManager.getDynamicRadius() - 1);
        }
        Integer override = this.playerRadiusOverrides.get(player.getUniqueId());
        return override != null ? override.intValue() : this.configManager.getDynamicRadius();
    }

    private void processChunkQueue() {
        int maxChunks;
        int chunksLoaded = 0;
        int n = maxChunks = this.memoryManager.isLowMemoryMode() ? Math.max(1, this.configManager.getMaxChunksToLoad() / 2) : this.configManager.getMaxChunksToLoad();
        while (chunksLoaded < maxChunks && !this.chunkCache.isQueueEmpty()) {
            Chunk chunk = this.chunkCache.pollFromQueue();
            if (chunk == null) continue;
            this.chunkProcessor.submitLoadTask(chunk);
            ++chunksLoaded;
        }
    }

    private void performSmartUnload() {
        for (World world : Bukkit.getWorlds()) {
            int unloaded;
            if (this.isWorldBlacklisted(world) || (unloaded = this.smartUnloader.performSmartUnload(world, 10)) <= 0) continue;
            this.getLogger().fine("Smart unloaded " + unloaded + " cold chunks from " + world.getName());
        }
    }

    private void monitorPerformance() {
        if (this.manualOverride) {
            return;
        }
        this.performanceMonitor.analyzePerformance();
        this.adjustSettingsBasedOnPerformance();
    }

    private void handlePerformanceChange(PerformanceMonitor.PerformanceStatus status) {
        if (this.manualOverride) {
            return;
        }
        switch (status) {
            case CRITICAL: {
                if (this.maintenanceMode) break;
                this.setMaintenanceMode(true, (CommandSender)Bukkit.getConsoleSender(), true);
                break;
            }
            case POOR: {
                int currentRadius;
                if (this.maintenanceMode || !this.configManager.isDynamicRadiusEnabled() || (currentRadius = this.configManager.getDynamicRadius()) <= 1) break;
                this.configManager.setDynamicRadius(currentRadius - 1);
                this.getLogger().fine(this.languageManager.getMessage("radius-reduced", Collections.singletonMap("radius", String.valueOf(currentRadius - 1))));
                break;
            }
            case EXCELLENT: {
                int currentRadius;
                if (this.maintenanceMode && this.autoMaintenanceActive) {
                    this.setMaintenanceMode(false, (CommandSender)Bukkit.getConsoleSender(), true);
                    break;
                }
                if (!this.configManager.isDynamicRadiusEnabled() || (currentRadius = this.configManager.getDynamicRadius()) >= 6) break;
                this.configManager.setDynamicRadius(currentRadius + 1);
                this.getLogger().fine(this.languageManager.getMessage("radius-increased", Collections.singletonMap("radius", String.valueOf(currentRadius + 1))));
                break;
            }
        }
    }

    private void adjustSettingsBasedOnPerformance() {
        PerformanceMonitor.PerformanceData data = this.performanceMonitor.getCurrentPerformance();
        if (this.maintenanceMode) {
            if (!this.autoMaintenanceActive) {
                return;
            }
            if (data.tickTime < this.configManager.getMaintenanceModeTickTimeThreshold() && data.memoryPercentage < this.configManager.getMaintenanceModeMemoryThreshold()) {
                this.setMaintenanceMode(false, (CommandSender)Bukkit.getConsoleSender(), true);
            }
            return;
        }
        if (data.tickTime > this.configManager.getMaintenanceModeTickTimeThreshold() || data.memoryPercentage > this.configManager.getMaintenanceModeMemoryThreshold()) {
            this.setMaintenanceMode(true, (CommandSender)Bukkit.getConsoleSender(), true);
        }
    }

    @Override
    public boolean isWorldBlacklisted(World world) {
        if (world == null) {
            return true;
        }
        return this.configManager.getWorldBlacklist().stream().anyMatch(blacklisted -> blacklisted.equalsIgnoreCase(world.getName()));
    }

    private void setupEventListeners() {
        if (!this.advancedFeaturesEnabled) {
            return;
        }
        this.eventManager.registerListener(ChunkEventManager.ChunkEventType.PRE_LOAD, event -> this.metricsCollector.recordMetric("cache_misses", 1L));
        this.eventManager.registerListener(ChunkEventManager.ChunkEventType.CACHE_HIT, event -> this.metricsCollector.recordMetric("cache_hits", 1L));
        this.eventManager.registerListener(ChunkEventManager.ChunkEventType.OPTIMIZATION_START, event -> this.metricsCollector.recordMetric("optimization_runs", 1L));
    }

    @EventHandler(priority=EventPriority.MONITOR)
    public void onPlayerTeleport(PlayerTeleportEvent event) {
        Location to;
        if (this.advancedFeaturesEnabled && (to = event.getTo()) != null && !this.isWorldBlacklisted(to.getWorld())) {
            this.chunkPrefetcher.prefetchArea(to.getWorld(), to, 3);
        }
    }

    private void updateMetrics() {
        if (!this.advancedFeaturesEnabled) {
            return;
        }
        double tickTime = this.performanceMonitor.getAverageTickTime();
        double memoryUsage = this.memoryManager.getMemoryUsagePercentage() * 100.0;
        int loadedChunks = this.performanceMonitor.getTotalLoadedChunks();
        int onlinePlayers = Bukkit.getOnlinePlayers().size();
        double tps = tickTime > 50.0 ? Math.max(0.0, 20.0 - (tickTime - 50.0) / 50.0) : 20.0;
        Runtime runtime = Runtime.getRuntime();
        long usedMemoryMB = (runtime.totalMemory() - runtime.freeMemory()) / 0x100000L;
        this.metricsCollector.recordTimeSeriesPoint("tps", tps);
        this.metricsCollector.recordTimeSeriesPoint("memory_mb", usedMemoryMB);
        this.metricsCollector.recordTimeSeriesPoint("loaded_chunks", loadedChunks);
        this.metricsCollector.recordTimeSeriesPoint("online_players", onlinePlayers);
        this.metricsCollector.recordMetric("tick_time", (long)tickTime);
        this.metricsCollector.recordMetric("memory_usage", (long)memoryUsage);
    }

    private void analyzeChunks(CommandSender sender, String[] args) {
        if (!this.advancedFeaturesEnabled) {
            sender.sendMessage(String.valueOf(ChatColor.RED) + "Advanced features are disabled in config!");
            return;
        }
        if (args.length == 0) {
            sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Analyzing all worlds...");
            this.taskManager.runAsyncTask("analyze", () -> {
                for (World world : Bukkit.getWorlds()) {
                    if (this.isWorldBlacklisted(world)) continue;
                    Map<String, Object> stats = this.chunkAnalyzer.getWorldStatistics(world);
                    Bukkit.getScheduler().runTask((Plugin)this, () -> {
                        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "\n" + world.getName() + " Analysis:");
                        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "----------------");
                        if (stats.containsKey("status")) {
                            sender.sendMessage(String.valueOf(ChatColor.RED) + "  " + String.valueOf(stats.get("status")));
                        } else {
                            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Total Chunks: " + String.valueOf(stats.get("totalChunks")));
                            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Avg Complexity: " + this.df.format(stats.get("averageComplexity")) + " | Avg Impact: " + this.df.format(stats.get("averagePerformanceImpact")) + "%");
                            sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "  High Priority: " + String.valueOf(stats.get("highPriorityChunks")) + " | Extreme: " + String.valueOf(stats.get("extremeComplexityChunks")));
                            sender.sendMessage(String.valueOf(ChatColor.GREEN) + "  Structure Chunks: " + String.valueOf(stats.get("structureChunks")) + " | Redstone: " + String.valueOf(stats.get("redstoneChunks")) + " | Farms: " + String.valueOf(stats.get("farmChunks")));
                            sender.sendMessage(String.valueOf(ChatColor.RED) + "  Total Mobs: " + String.valueOf(stats.get("totalMobs")) + " (avg " + this.df.format(stats.get("averageMobsPerChunk")) + "/chunk)");
                            sender.sendMessage(String.valueOf(ChatColor.GOLD) + "  Tile Entities: " + String.valueOf(stats.get("totalTileEntities")) + " (avg " + this.df.format(stats.get("averageTileEntitiesPerChunk")) + "/chunk)");
                            sender.sendMessage(String.valueOf(ChatColor.AQUA) + "  Memory Usage: " + String.valueOf(stats.get("totalMemoryUsageMB")) + " MB");
                        }
                    });
                }
            });
        } else {
            World world = Bukkit.getWorld((String)args[0]);
            if (world != null) {
                Location location;
                if (sender instanceof Player) {
                    Player p = (Player)sender;
                    location = p.getLocation();
                } else {
                    location = new Location(world, 0.0, 64.0, 0.0);
                }
                Location loc = location;
                Chunk chunk = loc.getChunk();
                this.taskManager.runAsyncTask("analyze-chunk", () -> {
                    ChunkAnalyzer.ChunkAnalysisData data = this.chunkAnalyzer.analyzeChunkSync(chunk);
                    Bukkit.getScheduler().runTask((Plugin)this, () -> {
                        sender.sendMessage(String.valueOf(ChatColor.GREEN) + "\nChunk Analysis [" + chunk.getX() + ", " + chunk.getZ() + "]");
                        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "----------------");
                        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "Complexity: " + data.complexity + " (" + data.getComplexityLevel() + ")");
                        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "Priority: " + data.priority + "/100");
                        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "Performance Impact: " + this.df.format(data.performanceImpact) + "%");
                        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "Memory Usage: " + data.estimatedMemoryUsage / 1024 + " KB");
                        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "Structures: " + data.hasPlayerStructures + " | Redstone: " + data.hasRedstone + " | Farmland: " + data.hasFarmland);
                        sender.sendMessage(String.valueOf(ChatColor.RED) + "Mobs: " + data.mobCount + " | Passive: " + data.passiveEntityCount + " | Items: " + data.itemCount);
                        sender.sendMessage(String.valueOf(ChatColor.GOLD) + "Tile Entities: " + data.tileEntityCount);
                        sender.sendMessage(String.valueOf(ChatColor.AQUA) + "Entity Density: " + this.df.format(data.entityDensity) + " per chunk");
                        if (!data.blockCategories.isEmpty()) {
                            sender.sendMessage(String.valueOf(ChatColor.DARK_AQUA) + "Block Categories: ");
                            data.blockCategories.forEach((category, count) -> sender.sendMessage(String.valueOf(ChatColor.GRAY) + "  " + category + ": " + count));
                        }
                    });
                });
            } else {
                sender.sendMessage(String.valueOf(ChatColor.RED) + "World not found: " + args[0]);
            }
        }
    }

    private void displayRegionInfo(CommandSender sender) {
        if (!this.advancedFeaturesEnabled) {
            sender.sendMessage(String.valueOf(ChatColor.RED) + "Advanced features are disabled in config!");
            return;
        }
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "--- " + String.valueOf(ChatColor.AQUA) + "Region Statistics " + String.valueOf(ChatColor.GRAY) + "---");
        for (World world : Bukkit.getWorlds()) {
            if (this.isWorldBlacklisted(world)) continue;
            Map<String, Object> stats = this.regionOptimizer.getRegionStatistics(world);
            sender.sendMessage(String.valueOf(ChatColor.YELLOW) + world.getName() + ":");
            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Total Regions: " + String.valueOf(stats.get("totalRegions")));
            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Active Regions: " + String.valueOf(stats.get("activeRegions")));
            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Optimized Regions: " + String.valueOf(stats.get("optimizedRegions")));
            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  High Activity: " + String.valueOf(stats.get("highActivityRegions")));
        }
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "---------------------------");
    }

    private void displayMetrics(CommandSender sender) {
        if (!this.advancedFeaturesEnabled) {
            sender.sendMessage(String.valueOf(ChatColor.RED) + "Advanced features are disabled in config!");
            return;
        }
        Map<String, Object> metrics = this.metricsCollector.getCurrentMetrics();
        Map<String, Object> prefetchStats = this.chunkPrefetcher.getPrefetchStatistics();
        double tickTime = this.performanceMonitor.getAverageTickTime();
        double tps = tickTime > 50.0 ? Math.max(0.0, 20.0 - (tickTime - 50.0) / 50.0) : 20.0;
        Runtime runtime = Runtime.getRuntime();
        long maxMemory = runtime.maxMemory() / 0x100000L;
        long totalMemory = runtime.totalMemory() / 0x100000L;
        long freeMemory = runtime.freeMemory() / 0x100000L;
        long usedMemory = totalMemory - freeMemory;
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "--- " + String.valueOf(ChatColor.AQUA) + "Advanced Metrics " + String.valueOf(ChatColor.GRAY) + "---");
        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "Server Performance:");
        ChatColor tpsColor = tps >= 19.0 ? ChatColor.GREEN : (tps >= 17.0 ? ChatColor.YELLOW : (tps >= 15.0 ? ChatColor.GOLD : ChatColor.RED));
        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  TPS: " + String.valueOf(tpsColor) + this.df.format(tps) + "/20.0");
        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Tick Time: " + this.df.format(tickTime) + "ms (target: 50ms)");
        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Memory: " + usedMemory + "/" + maxMemory + " MB (" + this.df.format((double)usedMemory / (double)maxMemory * 100.0) + "%)");
        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "Cache Performance:");
        if (metrics.containsKey("cache_hits")) {
            long misses;
            Map cacheHits = (Map)metrics.get("cache_hits");
            Map cacheMisses = (Map)metrics.get("cache_misses");
            long hits = (Long)cacheHits.get("count");
            double hitRate = hits + (misses = ((Long)cacheMisses.get("count")).longValue()) > 0L ? (double)hits / (double)(hits + misses) * 100.0 : 0.0;
            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Hits: " + hits + " | Misses: " + misses);
            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Hit Rate: " + this.df.format(hitRate) + "%");
        } else {
            sender.sendMessage(String.valueOf(ChatColor.GRAY) + "  No cache data available yet");
        }
        sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "Prefetch Statistics:");
        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Total Prefetched: " + String.valueOf(prefetchStats.get("totalPrefetched")));
        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Cache Size: " + String.valueOf(prefetchStats.get("cacheSize")));
        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Tracked Players: " + String.valueOf(prefetchStats.get("trackedPlayers")));
        sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Avg Confidence: " + this.df.format((Double)prefetchStats.get("averageConfidence") * 100.0) + "%");
        if (metrics.containsKey("optimization_runs")) {
            sender.sendMessage(String.valueOf(ChatColor.YELLOW) + "Optimization Stats:");
            Map optRuns = (Map)metrics.get("optimization_runs");
            sender.sendMessage(String.valueOf(ChatColor.WHITE) + "  Total Runs: " + String.valueOf(optRuns.get("count")));
        }
        sender.sendMessage(String.valueOf(ChatColor.GRAY) + "---------------------------");
    }

    public List<String> onTabComplete(CommandSender sender, Command command, String alias, String[] args) {
        if (!sender.hasPermission("chunkmanager.admin")) {
            return null;
        }
        String cmdName = command.getName().toLowerCase();
        if (cmdName.equals("maintenance") && args.length == 1) {
            return Arrays.asList("on", "off", "auto", "status");
        }
        if (cmdName.equals("chunklang") && args.length == 1) {
            return new ArrayList<String>(this.languageManager.getAvailableLanguages().keySet());
        }
        if (cmdName.equals("chunkanalyze") && args.length == 1) {
            return Bukkit.getWorlds().stream().map(WorldInfo::getName).collect(Collectors.toList());
        }
        return null;
    }

    @Override
    public int getCurrentRadius() {
        return this.configManager.getDynamicRadius();
    }

    @Override
    public boolean isMaintenanceMode() {
        return this.maintenanceMode;
    }

    @Override
    public void setMaintenanceMode(boolean enabled) {
        this.setMaintenanceMode(enabled, (CommandSender)Bukkit.getConsoleSender(), false);
    }

    @Override
    public void setPlayerRadius(Player player, int radius) {
        this.playerRadiusOverrides.put(player.getUniqueId(), radius);
    }

    @Override
    public int getPlayerRadius(Player player) {
        return this.getEffectiveRadius(player);
    }

    @Override
    public CompletableFuture<PerformanceMonitor.WorldStatistics> getWorldStatistics(World world) {
        return CompletableFuture.supplyAsync(() -> this.performanceMonitor.gatherWorldStatistics(world));
    }

    @Override
    public Map<String, String> getPerformanceInfo() {
        return this.performanceMonitor.getPerformanceInfo();
    }

    @Override
    public void forceChunkOptimization() {
        for (World world : Bukkit.getWorlds()) {
            if (this.isWorldBlacklisted(world)) continue;
            this.smartUnloader.performSmartUnload(world, 100);
        }
    }

    @Override
    public void forceGarbageCollection() {
        this.memoryManager.forceGarbageCollection();
    }

    @Override
    public void reloadConfiguration() {
        this.configManager.reload();
        this.languageManager.reloadLanguages();
    }

    @Override
    public int getLoadedChunksCount() {
        return this.performanceMonitor.getTotalLoadedChunks();
    }

    @Override
    public int getQueuedChunksCount() {
        return this.chunkCache.getQueueSize();
    }

    @Override
    public double getAverageTickTime() {
        return this.performanceMonitor.getAverageTickTime();
    }

    @Override
    public double getMemoryUsagePercentage() {
        return this.memoryManager.getMemoryUsagePercentage() * 100.0;
    }

    public ConfigurationManager getConfigManager() {
        return this.configManager;
    }

    public PerformanceMonitor getPerformanceMonitor() {
        return this.performanceMonitor;
    }

    public MemoryManager getMemoryManager() {
        return this.memoryManager;
    }

    public ChunkCache getChunkCache() {
        return this.chunkCache;
    }

    public boolean isManualOverride() {
        return this.manualOverride;
    }

    public GuiManager getGuiManager() {
        return this.guiManager;
    }

    public RegionOptimizer getRegionOptimizer() {
        return this.regionOptimizer;
    }

    public ChunkAnalyzer getChunkAnalyzer() {
        return this.chunkAnalyzer;
    }
}

