/*
 * Decompiled with CFR 0.152.
 */
package cm.chunkManager.components;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.bukkit.Bukkit;
import org.bukkit.Chunk;
import org.bukkit.ChunkSnapshot;
import org.bukkit.Material;
import org.bukkit.World;
import org.bukkit.block.BlockState;
import org.bukkit.entity.Entity;
import org.bukkit.entity.EntityType;
import org.bukkit.entity.LivingEntity;
import org.bukkit.entity.Monster;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;

public class ChunkAnalyzer {
    private final JavaPlugin plugin;
    private final Map<ChunkIdentifier, ChunkAnalysisData> analysisCache;
    private final Random random = new Random();

    public ChunkAnalyzer(JavaPlugin plugin) {
        this.plugin = plugin;
        this.analysisCache = new ConcurrentHashMap<ChunkIdentifier, ChunkAnalysisData>();
    }

    public ChunkAnalysisData analyzeChunkSync(Chunk chunk) {
        ChunkIdentifier id = new ChunkIdentifier(chunk);
        ChunkAnalysisData cached = this.analysisCache.get(id);
        if (cached != null && !cached.isExpired()) {
            return cached;
        }
        if (Bukkit.isPrimaryThread()) {
            return this.performSmartAnalysis(chunk);
        }
        CompletableFuture future = new CompletableFuture();
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> future.complete(this.performSmartAnalysis(chunk)));
        try {
            return (ChunkAnalysisData)future.get(2L, TimeUnit.SECONDS);
        }
        catch (TimeoutException e) {
            this.plugin.getLogger().warning("Chunk analysis timed out after 2 seconds");
            return this.createEmptyAnalysis();
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to analyze chunk: " + e.getMessage());
            return this.createEmptyAnalysis();
        }
    }

    public CompletableFuture<ChunkAnalysisData> analyzeChunkAsync(Chunk chunk) {
        ChunkIdentifier id = new ChunkIdentifier(chunk);
        ChunkAnalysisData cached = this.analysisCache.get(id);
        if (cached != null && !cached.isExpired()) {
            return CompletableFuture.completedFuture(cached);
        }
        CompletableFuture<ChunkAnalysisData> future = new CompletableFuture<ChunkAnalysisData>();
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
            ChunkDataSnapshot dataSnapshot = this.captureChunkData(chunk);
            Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
                ChunkAnalysisData analysis = this.analyzeSnapshot(dataSnapshot, new ChunkIdentifier(chunk));
                future.complete(analysis);
            });
        });
        return future;
    }

    private ChunkDataSnapshot captureChunkData(Chunk chunk) {
        if (!chunk.isLoaded()) {
            return new ChunkDataSnapshot(new Entity[0], 0, new ArrayList<Material>(), chunk.getChunkSnapshot(true, false, false));
        }
        Entity[] entities = chunk.getEntities();
        ChunkSnapshot snapshot = chunk.getChunkSnapshot(true, false, false);
        ArrayList<Material> tileEntityTypes = new ArrayList<Material>();
        int tileEntityCount = 0;
        try {
            BlockState[] tiles = chunk.getTileEntities();
            tileEntityCount = tiles.length;
            for (BlockState tile : tiles) {
                tileEntityTypes.add(tile.getType());
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().fine("Could not get tile entities for chunk: " + e.getMessage());
        }
        return new ChunkDataSnapshot(entities, tileEntityCount, tileEntityTypes, snapshot);
    }

    private ChunkAnalysisData performSmartAnalysis(Chunk chunk) {
        ChunkIdentifier id = new ChunkIdentifier(chunk);
        if (!chunk.isLoaded()) {
            return this.createEmptyAnalysis();
        }
        ChunkDataSnapshot dataSnapshot = this.captureChunkData(chunk);
        return this.analyzeSnapshot(dataSnapshot, id);
    }

    private ChunkAnalysisData analyzeSnapshot(ChunkDataSnapshot dataSnapshot, ChunkIdentifier id) {
        HashMap<EntityType, Integer> entityCounts = new HashMap<EntityType, Integer>();
        HashMap<String, Integer> blockCategories = new HashMap<String, Integer>();
        int mobCount = 0;
        int passiveEntityCount = 0;
        int itemCount = 0;
        for (Entity entity : dataSnapshot.entities) {
            EntityType type = entity.getType();
            entityCounts.merge(type, 1, Integer::sum);
            if (entity instanceof Monster) {
                ++mobCount;
            } else if (entity instanceof LivingEntity) {
                ++passiveEntityCount;
            }
            if (type != EntityType.DROPPED_ITEM) continue;
            ++itemCount;
        }
        boolean hasPlayerStructures = false;
        boolean hasRedstone = false;
        boolean hasFarmland = false;
        HashSet<Material> redstoneTypes = new HashSet<Material>();
        for (Material type : dataSnapshot.tileEntityTypes) {
            String category = this.categorizeBlock(type);
            blockCategories.merge(category, 1, Integer::sum);
            if (this.isPlayerStructureBlock(type)) {
                hasPlayerStructures = true;
            }
            if (!this.isRedstoneComponent(type)) continue;
            hasRedstone = true;
            redstoneTypes.add(type);
        }
        ChunkSnapshot snapshot = dataSnapshot.snapshot;
        int sampleSize = 128;
        int sampledBlocks = 0;
        for (int i = 0; i < sampleSize; ++i) {
            int sampleY;
            int z;
            int x = this.random.nextInt(16);
            int y = snapshot.getHighestBlockYAt(x, z = this.random.nextInt(16));
            if (y <= -64) continue;
            for (int j = 0; j < 3 && (sampleY = y - j * 10) >= -64; ++j) {
                Material mat = snapshot.getBlockType(x, sampleY, z);
                if (mat == Material.AIR) continue;
                ++sampledBlocks;
                if (mat == Material.FARMLAND) {
                    hasFarmland = true;
                }
                if (!this.isRedstoneComponent(mat)) continue;
                hasRedstone = true;
                redstoneTypes.add(mat);
            }
        }
        double entityDensity = (double)dataSnapshot.entities.length / 256.0;
        int complexity = this.calculateSmartComplexity(entityCounts, dataSnapshot.tileEntityCount, mobCount, itemCount, hasRedstone, redstoneTypes.size(), sampledBlocks);
        double performanceImpact = this.calculatePerformanceImpact(dataSnapshot.tileEntityCount, mobCount, itemCount, entityDensity, hasRedstone);
        int estimatedMemoryUsage = this.estimateMemoryUsage(dataSnapshot.entities.length, dataSnapshot.tileEntityCount, sampledBlocks);
        int priority = this.calculateSmartPriority(complexity, performanceImpact, hasPlayerStructures, entityDensity);
        ChunkAnalysisData data = new ChunkAnalysisData(entityCounts, blockCategories, complexity, priority, hasPlayerStructures, hasRedstone, hasFarmland, dataSnapshot.tileEntityCount, entityDensity, mobCount, passiveEntityCount, itemCount, performanceImpact, estimatedMemoryUsage);
        this.analysisCache.put(id, data);
        return data;
    }

    private String categorizeBlock(Material material) {
        String name = material.name();
        if (name.contains("CHEST") || name.contains("BARREL") || name.contains("SHULKER")) {
            return "STORAGE";
        }
        if (name.contains("FURNACE") || name.contains("SMOKER") || name.contains("BLAST")) {
            return "PROCESSING";
        }
        if (name.contains("REDSTONE") || name.contains("PISTON") || name.contains("OBSERVER")) {
            return "REDSTONE";
        }
        if (name.contains("SPAWNER")) {
            return "SPAWNER";
        }
        if (name.contains("COMMAND") || name.contains("STRUCTURE")) {
            return "TECHNICAL";
        }
        if (name.contains("SIGN") || name.contains("BANNER") || name.contains("FRAME")) {
            return "DECORATIVE";
        }
        if (name.contains("HOPPER") || name.contains("DROPPER") || name.contains("DISPENSER")) {
            return "AUTOMATION";
        }
        return "OTHER";
    }

    private int calculateSmartComplexity(Map<EntityType, Integer> entities, int tileEntities, int mobs, int items, boolean hasRedstone, int redstoneTypes, int sampledBlocks) {
        int complexity = 0;
        complexity += tileEntities * 15;
        complexity += mobs * 8;
        complexity += items * 2;
        complexity += entities.size() * 5;
        if (hasRedstone) {
            complexity += 50 + redstoneTypes * 10;
        }
        if (tileEntities > 50) {
            complexity += (tileEntities - 50) * 5;
        }
        if (mobs > 30) {
            complexity += (mobs - 30) * 10;
        }
        if (items > 100) {
            complexity += items - 100;
        }
        return Math.min(complexity += Math.min(sampledBlocks / 10, 50), 1000);
    }

    private double calculatePerformanceImpact(int tileEntities, int mobs, int items, double entityDensity, boolean hasRedstone) {
        double impact = 0.0;
        impact += (double)tileEntities * 0.5;
        impact += (double)mobs * 0.3;
        impact += (double)items * 0.05;
        impact += entityDensity * 10.0;
        if (hasRedstone) {
            impact *= 1.5;
        }
        if (tileEntities > 100) {
            impact *= 2.0;
        }
        return Math.min(impact, 100.0);
    }

    private int estimateMemoryUsage(int entities, int tileEntities, int blocks) {
        int baseMemory = 8192;
        int entityMemory = entities * 512;
        int tileMemory = tileEntities * 1024;
        int blockMemory = blocks * 4;
        return baseMemory + entityMemory + tileMemory + blockMemory;
    }

    private int calculateSmartPriority(int complexity, double performanceImpact, boolean hasStructures, double entityDensity) {
        int priority = (int)((double)complexity * 0.4 + performanceImpact * 0.6);
        if (hasStructures) {
            priority += 20;
        }
        if (entityDensity > 1.0) {
            priority += (int)(entityDensity * 10.0);
        }
        return Math.min(priority, 100);
    }

    private boolean isPlayerStructureBlock(Material material) {
        String name = material.name();
        return name.contains("CHEST") || name.contains("FURNACE") || name.contains("CRAFTING") || name.contains("ANVIL") || name.contains("ENCHANTING") || name.contains("BREWING") || name.contains("BED") || name.contains("BEACON") || name.contains("ENDER_CHEST") || name.contains("SHULKER");
    }

    private boolean isRedstoneComponent(Material material) {
        String name = material.name();
        return name.contains("REDSTONE") || name.contains("PISTON") || name.contains("OBSERVER") || name.contains("HOPPER") || name.contains("COMPARATOR") || name.contains("REPEATER") || name.contains("DROPPER") || name.contains("DISPENSER") || name.contains("RAIL") || name.contains("LEVER") || name.contains("BUTTON") || name.contains("PRESSURE");
    }

    private ChunkAnalysisData createEmptyAnalysis() {
        return new ChunkAnalysisData(new HashMap<EntityType, Integer>(), new HashMap<String, Integer>(), 0, 0, false, false, false, 0, 0.0, 0, 0, 0, 0.0, 0);
    }

    public List<ChunkAnalysisData> analyzeWorld(World world) {
        Chunk[] loadedChunks;
        ArrayList<ChunkAnalysisData> results = new ArrayList<ChunkAnalysisData>();
        for (Chunk chunk : loadedChunks = world.getLoadedChunks()) {
            results.add(this.analyzeChunkSync(chunk));
        }
        return results;
    }

    public Map<String, Object> getWorldStatistics(World world) {
        if (world == null) {
            HashMap<String, Object> emptyStats = new HashMap<String, Object>();
            emptyStats.put("status", "World is null");
            return emptyStats;
        }
        List<ChunkAnalysisData> analyses = this.analyzeWorld(world);
        HashMap<String, Object> stats = new HashMap<String, Object>();
        if (analyses.isEmpty()) {
            stats.put("status", "No chunks loaded");
            return stats;
        }
        int totalComplexity = 0;
        int highPriorityChunks = 0;
        int extremeComplexityChunks = 0;
        int structureChunks = 0;
        int redstoneChunks = 0;
        int farmChunks = 0;
        double totalPerformanceImpact = 0.0;
        long totalMemoryUsage = 0L;
        int totalMobs = 0;
        int totalItems = 0;
        int totalTileEntities = 0;
        for (ChunkAnalysisData data : analyses) {
            totalComplexity += data.complexity;
            totalPerformanceImpact += data.performanceImpact;
            totalMemoryUsage += (long)data.estimatedMemoryUsage;
            totalMobs += data.mobCount;
            totalItems += data.itemCount;
            totalTileEntities += data.tileEntityCount;
            if (data.priority > 70) {
                ++highPriorityChunks;
            }
            if (data.complexity > 300) {
                ++extremeComplexityChunks;
            }
            if (data.hasPlayerStructures) {
                ++structureChunks;
            }
            if (data.hasRedstone) {
                ++redstoneChunks;
            }
            if (!data.hasFarmland) continue;
            ++farmChunks;
        }
        int chunkCount = analyses.size();
        stats.put("totalChunks", chunkCount);
        stats.put("averageComplexity", chunkCount > 0 ? totalComplexity / chunkCount : 0);
        stats.put("averagePerformanceImpact", chunkCount > 0 ? totalPerformanceImpact / (double)chunkCount : 0.0);
        stats.put("totalMemoryUsageMB", totalMemoryUsage / 0x100000L);
        stats.put("highPriorityChunks", highPriorityChunks);
        stats.put("extremeComplexityChunks", extremeComplexityChunks);
        stats.put("structureChunks", structureChunks);
        stats.put("redstoneChunks", redstoneChunks);
        stats.put("farmChunks", farmChunks);
        stats.put("totalMobs", totalMobs);
        stats.put("totalItems", totalItems);
        stats.put("totalTileEntities", totalTileEntities);
        stats.put("averageMobsPerChunk", chunkCount > 0 ? totalMobs / chunkCount : 0);
        stats.put("averageTileEntitiesPerChunk", chunkCount > 0 ? totalTileEntities / chunkCount : 0);
        return stats;
    }

    public void clearCache() {
        this.analysisCache.clear();
    }

    public void cleanExpiredCache() {
        this.analysisCache.entrySet().removeIf(entry -> ((ChunkAnalysisData)entry.getValue()).isExpired());
    }

    public static class ChunkIdentifier {
        public final String worldName;
        public final int x;
        public final int z;

        public ChunkIdentifier(Chunk chunk) {
            this.worldName = chunk.getWorld().getName();
            this.x = chunk.getX();
            this.z = chunk.getZ();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ChunkIdentifier)) {
                return false;
            }
            ChunkIdentifier that = (ChunkIdentifier)o;
            return this.x == that.x && this.z == that.z && this.worldName.equals(that.worldName);
        }

        public int hashCode() {
            return Objects.hash(this.worldName, this.x, this.z);
        }
    }

    public static class ChunkAnalysisData {
        public final Map<EntityType, Integer> entityCounts;
        public final Map<String, Integer> blockCategories;
        public final int complexity;
        public final int priority;
        public final long timestamp;
        public final boolean hasPlayerStructures;
        public final boolean hasRedstone;
        public final boolean hasFarmland;
        public final int tileEntityCount;
        public final double entityDensity;
        public final int mobCount;
        public final int passiveEntityCount;
        public final int itemCount;
        public final double performanceImpact;
        public final int estimatedMemoryUsage;

        public ChunkAnalysisData(Map<EntityType, Integer> entityCounts, Map<String, Integer> blockCategories, int complexity, int priority, boolean hasPlayerStructures, boolean hasRedstone, boolean hasFarmland, int tileEntityCount, double entityDensity, int mobCount, int passiveEntityCount, int itemCount, double performanceImpact, int estimatedMemoryUsage) {
            this.entityCounts = entityCounts;
            this.blockCategories = blockCategories;
            this.complexity = complexity;
            this.priority = priority;
            this.hasPlayerStructures = hasPlayerStructures;
            this.hasRedstone = hasRedstone;
            this.hasFarmland = hasFarmland;
            this.tileEntityCount = tileEntityCount;
            this.entityDensity = entityDensity;
            this.mobCount = mobCount;
            this.passiveEntityCount = passiveEntityCount;
            this.itemCount = itemCount;
            this.performanceImpact = performanceImpact;
            this.estimatedMemoryUsage = estimatedMemoryUsage;
            this.timestamp = System.currentTimeMillis();
        }

        public boolean isExpired() {
            return System.currentTimeMillis() - this.timestamp > 300000L;
        }

        public String getComplexityLevel() {
            if (this.complexity < 50) {
                return "LOW";
            }
            if (this.complexity < 150) {
                return "MEDIUM";
            }
            if (this.complexity < 300) {
                return "HIGH";
            }
            return "EXTREME";
        }
    }

    private static class ChunkDataSnapshot {
        public final Entity[] entities;
        public final int tileEntityCount;
        public final List<Material> tileEntityTypes;
        public final ChunkSnapshot snapshot;

        public ChunkDataSnapshot(Entity[] entities, int tileEntityCount, List<Material> tileEntityTypes, ChunkSnapshot snapshot) {
            this.entities = entities;
            this.tileEntityCount = tileEntityCount;
            this.tileEntityTypes = tileEntityTypes;
            this.snapshot = snapshot;
        }
    }
}

