/*
 * Decompiled with CFR 0.152.
 */
package cm.chunkManager.components;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryUsage;
import java.text.DecimalFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.logging.Logger;
import org.bukkit.Bukkit;
import org.bukkit.Chunk;
import org.bukkit.World;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;

public class PerformanceMonitor {
    private final JavaPlugin plugin;
    private final Logger logger;
    private final DecimalFormat df = new DecimalFormat("#.##");
    private long lastTickTime = System.currentTimeMillis();
    private double averageTickTime = 50.0;
    private Consumer<PerformanceStatus> performanceHandler;

    public PerformanceMonitor(JavaPlugin plugin) {
        this.plugin = plugin;
        this.logger = plugin.getLogger();
    }

    public void setPerformanceHandler(Consumer<PerformanceStatus> handler) {
        this.performanceHandler = handler;
    }

    public void updateTickTime() {
        long currentTime = System.currentTimeMillis();
        long timeSpent = currentTime - this.lastTickTime;
        this.averageTickTime = (this.averageTickTime * 19.0 + (double)timeSpent) / 20.0;
        this.lastTickTime = currentTime;
    }

    public PerformanceData getCurrentPerformance() {
        double tickTime = this.getAverageTickTime();
        MemoryUsage heapUsage = this.getHeapMemoryUsage();
        double memoryUsageMB = (double)heapUsage.getUsed() / 1048576.0;
        double memoryPercentage = (double)heapUsage.getUsed() / (double)heapUsage.getMax() * 100.0;
        int totalChunks = this.getTotalLoadedChunks();
        PerformanceStatus status = this.determinePerformanceStatus(tickTime, memoryPercentage);
        return new PerformanceData(tickTime, memoryUsageMB, memoryPercentage, totalChunks, status);
    }

    public PerformanceStatus checkPerformance(double tickThreshold, double memoryThreshold) {
        PerformanceData data = this.getCurrentPerformance();
        if (data.tickTime > tickThreshold || data.memoryPercentage > memoryThreshold) {
            return PerformanceStatus.POOR;
        }
        if (data.tickTime < 40.0 && data.memoryPercentage < 60.0) {
            return PerformanceStatus.EXCELLENT;
        }
        return PerformanceStatus.GOOD;
    }

    private PerformanceStatus determinePerformanceStatus(double tickTime, double memoryPercentage) {
        if (tickTime > 70.0 || memoryPercentage > 85.0) {
            return PerformanceStatus.CRITICAL;
        }
        if (tickTime > 55.0 || memoryPercentage > 75.0) {
            return PerformanceStatus.POOR;
        }
        if (tickTime > 45.0 || memoryPercentage > 65.0) {
            return PerformanceStatus.MODERATE;
        }
        if (tickTime > 35.0 || memoryPercentage > 50.0) {
            return PerformanceStatus.GOOD;
        }
        return PerformanceStatus.EXCELLENT;
    }

    public void analyzePerformance() {
        PerformanceData data = this.getCurrentPerformance();
        if (this.performanceHandler != null) {
            this.performanceHandler.accept(data.status);
        }
        if (data.status == PerformanceStatus.CRITICAL) {
            this.logger.warning("Critical performance detected! Tick time: " + this.df.format(data.tickTime) + "ms, Memory: " + this.df.format(data.memoryPercentage) + "%");
        }
    }

    public WorldStatistics gatherWorldStatistics(World world) {
        WorldStatistics stats = new WorldStatistics();
        if (world == null) {
            return stats;
        }
        stats.totalPlayers = world.getPlayers().size();
        if (this.plugin.getServer().isPrimaryThread()) {
            stats.totalLoadedEntities = world.getEntities().size();
            for (Chunk chunk : world.getLoadedChunks()) {
                int entitiesInChunk = chunk.getEntities().length;
                int tileEntitiesInChunk = chunk.getTileEntities().length;
                stats.totalEntities += entitiesInChunk;
                stats.totalTileEntities += tileEntitiesInChunk;
                if (entitiesInChunk <= 0 && tileEntitiesInChunk <= 0) continue;
                ++stats.activeChunks;
            }
        } else {
            stats.totalLoadedEntities = 0;
            stats.totalEntities = 0;
            stats.totalTileEntities = 0;
            stats.activeChunks = 0;
        }
        stats.totalTileEntitiesWorld = stats.totalTileEntities;
        return stats;
    }

    public Map<String, String> getPerformanceInfo() {
        PerformanceData data = this.getCurrentPerformance();
        HashMap<String, String> info = new HashMap<String, String>();
        info.put("tick_time", this.df.format(data.tickTime));
        info.put("memory_mb", this.df.format(data.memoryUsage));
        info.put("memory_percent", this.df.format(data.memoryPercentage));
        info.put("total_chunks", String.valueOf(data.totalLoadedChunks));
        info.put("status", data.status.name());
        info.put("online_players", String.valueOf(Bukkit.getOnlinePlayers().size()));
        return info;
    }

    public double getAverageTickTime() {
        return this.averageTickTime;
    }

    public MemoryUsage getHeapMemoryUsage() {
        return ManagementFactory.getMemoryMXBean().getHeapMemoryUsage();
    }

    public MemoryUsage getNonHeapMemoryUsage() {
        return ManagementFactory.getMemoryMXBean().getNonHeapMemoryUsage();
    }

    public int getTotalLoadedChunks() {
        return Bukkit.getWorlds().stream().mapToInt(world -> world.getLoadedChunks().length).sum();
    }

    public void runGarbageCollection(Consumer<Map<String, Double>> callback) {
        double beforeGC = (double)this.getHeapMemoryUsage().getUsed() / 1048576.0;
        System.gc();
        Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> {
            double afterGC = (double)this.getHeapMemoryUsage().getUsed() / 1048576.0;
            double freedMemory = Math.abs(beforeGC - afterGC);
            HashMap<String, Double> results = new HashMap<String, Double>();
            results.put("before", beforeGC);
            results.put("after", afterGC);
            results.put("freed", freedMemory);
            callback.accept(results);
        }, 20L);
    }

    public static enum PerformanceStatus {
        EXCELLENT,
        GOOD,
        MODERATE,
        POOR,
        CRITICAL;

    }

    public static class PerformanceData {
        public final double tickTime;
        public final double memoryUsage;
        public final double memoryPercentage;
        public final int totalLoadedChunks;
        public final PerformanceStatus status;

        public PerformanceData(double tickTime, double memoryUsage, double memoryPercentage, int totalLoadedChunks, PerformanceStatus status) {
            this.tickTime = tickTime;
            this.memoryUsage = memoryUsage;
            this.memoryPercentage = memoryPercentage;
            this.totalLoadedChunks = totalLoadedChunks;
            this.status = status;
        }
    }

    public static class WorldStatistics {
        public int activeChunks;
        public int totalEntities;
        public int totalTileEntities;
        public int totalPlayers;
        public int totalLoadedEntities;
        public int totalTileEntitiesWorld;

        public Map<String, String> toPlaceholderMap() {
            HashMap<String, String> placeholders = new HashMap<String, String>();
            placeholders.put("active_chunks", String.valueOf(this.activeChunks));
            placeholders.put("total_entities", String.valueOf(this.totalEntities));
            placeholders.put("total_tile_entities", String.valueOf(this.totalTileEntities));
            placeholders.put("total_players", String.valueOf(this.totalPlayers));
            placeholders.put("total_loaded_entities", String.valueOf(this.totalLoadedEntities));
            placeholders.put("total_tile_entities_world", String.valueOf(this.totalTileEntitiesWorld));
            return placeholders;
        }
    }
}

