/*
 * Decompiled with CFR 0.152.
 */
package cm.chunkManager.components;

import cm.chunkManager.components.ChunkProcessor;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.bukkit.Chunk;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.entity.IronGolem;
import org.bukkit.entity.Player;
import org.bukkit.entity.Villager;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.bukkit.scheduler.BukkitRunnable;

public class SmartChunkUnloader {
    private final JavaPlugin plugin;
    private final Map<ChunkIdentifier, ChunkHeatData> heatMap;
    private final Map<UUID, PlayerMovementData> playerMovement;
    private final ChunkProcessor chunkProcessor;
    private final double heatDecayRate = 0.95;
    private final double visitHeatIncrease = 10.0;
    private final double proximityHeatBonus = 5.0;
    private final double coldThreshold = 1.0;
    private final long minChunkAge = 30000L;

    public SmartChunkUnloader(JavaPlugin plugin, ChunkProcessor processor) {
        this.plugin = plugin;
        this.chunkProcessor = processor;
        this.heatMap = new ConcurrentHashMap<ChunkIdentifier, ChunkHeatData>();
        this.playerMovement = new ConcurrentHashMap<UUID, PlayerMovementData>();
        this.startHeatDecayTask();
        this.startMovementTracker();
    }

    private void startHeatDecayTask() {
        new BukkitRunnable(){

            public void run() {
                SmartChunkUnloader.this.decayHeatValues();
            }
        }.runTaskTimerAsynchronously((Plugin)this.plugin, 100L, 100L);
    }

    private void startMovementTracker() {
        new BukkitRunnable(){

            public void run() {
                for (Player player : SmartChunkUnloader.this.plugin.getServer().getOnlinePlayers()) {
                    SmartChunkUnloader.this.trackPlayerMovement(player);
                }
            }
        }.runTaskTimer((Plugin)this.plugin, 20L, 20L);
    }

    private void trackPlayerMovement(Player player) {
        PlayerMovementData data = this.playerMovement.computeIfAbsent(player.getUniqueId(), k -> new PlayerMovementData());
        data.addLocation(player.getLocation());
        this.updateNearbyChunkHeat(player.getLocation(), 5.0);
        Set<Chunk> predictedChunks = data.getPredictedChunks(2);
        for (Chunk chunk : predictedChunks) {
            this.increaseChunkHeat(chunk, 2.5);
        }
    }

    public void onChunkLoad(Chunk chunk) {
        ChunkIdentifier id = new ChunkIdentifier(chunk);
        ChunkHeatData data = this.heatMap.computeIfAbsent(id, k -> new ChunkHeatData());
        data.recordVisit();
        this.updateChunkEntityData(chunk, data);
    }

    public void onChunkAccess(Chunk chunk) {
        ChunkIdentifier id = new ChunkIdentifier(chunk);
        ChunkHeatData data = this.heatMap.computeIfAbsent(id, k -> new ChunkHeatData());
        data.recordVisit();
        this.increaseChunkHeat(chunk, 10.0);
    }

    private void updateChunkEntityData(Chunk chunk, ChunkHeatData data) {
        int entities = chunk.getEntities().length;
        int tileEntities = chunk.getTileEntities().length;
        data.entityCount = entities + tileEntities;
        data.hasImportantEntities = this.hasImportantEntities(chunk);
        if (data.hasImportantEntities) {
            data.updateHeat(data.getHeatValue() * 2.0);
        }
    }

    private boolean hasImportantEntities(Chunk chunk) {
        return Arrays.stream(chunk.getEntities()).anyMatch(entity -> entity instanceof Villager || entity instanceof IronGolem || entity.getCustomName() != null) || chunk.getTileEntities().length > 5;
    }

    private void increaseChunkHeat(Chunk chunk, double amount) {
        ChunkIdentifier id = new ChunkIdentifier(chunk);
        ChunkHeatData data = this.heatMap.computeIfAbsent(id, k -> new ChunkHeatData());
        double entityMultiplier = Math.max(1.0, Math.min(10.0, 1.0 + (double)data.entityCount * 0.1));
        double adjustedAmount = Math.max(0.0, amount * entityMultiplier);
        double currentHeat = data.getHeatValue();
        double newHeat = Math.max(0.0, Math.min(1000.0, currentHeat + adjustedAmount));
        data.updateHeat(newHeat);
    }

    private void updateNearbyChunkHeat(Location location, double baseHeat) {
        if (location == null) {
            return;
        }
        World world = location.getWorld();
        if (world == null) {
            return;
        }
        int centerX = location.getBlockX() >> 4;
        int centerZ = location.getBlockZ() >> 4;
        for (int dx = -2; dx <= 2; ++dx) {
            for (int dz = -2; dz <= 2; ++dz) {
                double distance = Math.sqrt(dx * dx + dz * dz);
                double heat = baseHeat / (1.0 + distance);
                Chunk nearbyChunk = world.getChunkAt(centerX + dx, centerZ + dz);
                if (!nearbyChunk.isLoaded()) continue;
                this.increaseChunkHeat(nearbyChunk, heat);
            }
        }
    }

    private void decayHeatValues() {
        this.heatMap.forEach((id, data) -> {
            double currentHeat = data.getHeatValue();
            if (currentHeat <= 0.0 || !Double.isFinite(currentHeat)) {
                data.updateHeat(0.0);
                return;
            }
            long timeSinceAccess = data.getTimeSinceLastAccess();
            if (timeSinceAccess < 0L) {
                return;
            }
            double timeFactor = Math.min(1.0, Math.max(0.0, (double)timeSinceAccess / 60000.0));
            double decayMultiplier = Math.pow(0.95, timeFactor);
            double newHeat = Math.max(0.0, Math.min(1000.0, currentHeat * decayMultiplier));
            if (data.hasImportantEntities) {
                newHeat = Math.max(newHeat, 5.0);
            }
            data.updateHeat(newHeat);
        });
    }

    public List<Chunk> identifyColdChunks(World world, int maxChunks) {
        List coldChunkSnapshot = this.heatMap.entrySet().stream().filter(entry -> ((ChunkIdentifier)entry.getKey()).worldName.equals(world.getName())).filter(entry -> {
            ChunkHeatData data = (ChunkHeatData)entry.getValue();
            return data.getHeatValue() < 1.0 && data.getAge() > 30000L && !data.hasImportantEntities;
        }).map(entry -> {
            double heat = ((ChunkHeatData)entry.getValue()).getHeatValue();
            if (Double.isNaN(heat) || Double.isInfinite(heat)) {
                heat = 0.0;
            }
            return new AbstractMap.SimpleEntry<ChunkIdentifier, Double>((ChunkIdentifier)entry.getKey(), heat);
        }).collect(Collectors.toList());
        coldChunkSnapshot.sort((a, b) -> {
            double heat1 = (Double)a.getValue();
            double heat2 = (Double)b.getValue();
            if (Double.isNaN(heat1)) {
                heat1 = 0.0;
            }
            if (Double.isNaN(heat2)) {
                heat2 = 0.0;
            }
            return Double.compare(heat1, heat2);
        });
        return coldChunkSnapshot.stream().limit(maxChunks).map(entry -> {
            ChunkIdentifier id = (ChunkIdentifier)entry.getKey();
            return world.getChunkAt(id.x, id.z);
        }).filter(Chunk::isLoaded).collect(Collectors.toList());
    }

    public int performSmartUnload(World world, int targetUnloadCount) {
        if (world == null || targetUnloadCount <= 0) {
            return 0;
        }
        List<Chunk> coldChunks = this.identifyColdChunks(world, targetUnloadCount);
        AtomicInteger unloadedCount = new AtomicInteger(0);
        for (Chunk chunk : coldChunks) {
            if (!this.isChunkSafeToUnload(chunk)) continue;
            this.chunkProcessor.submitUnloadTask(chunk).thenRun(() -> {
                this.heatMap.remove(new ChunkIdentifier(chunk));
                unloadedCount.incrementAndGet();
            });
        }
        return unloadedCount.get();
    }

    private boolean isChunkSafeToUnload(Chunk chunk) {
        for (Player player : chunk.getWorld().getPlayers()) {
            Location playerLoc = player.getLocation();
            int playerChunkX = playerLoc.getBlockX() >> 4;
            int playerChunkZ = playerLoc.getBlockZ() >> 4;
            int distance = Math.max(Math.abs(chunk.getX() - playerChunkX), Math.abs(chunk.getZ() - playerChunkZ));
            if (distance > 3) continue;
            return false;
        }
        return true;
    }

    public Map<String, Object> getHeatMapStats() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        double totalHeat = this.heatMap.values().stream().mapToDouble(ChunkHeatData::getHeatValue).sum();
        double averageHeat = this.heatMap.isEmpty() ? 0.0 : totalHeat / (double)this.heatMap.size();
        long coldChunks = this.heatMap.values().stream().filter(data -> data.getHeatValue() < 1.0).count();
        long hotChunks = this.heatMap.values().stream().filter(data -> data.getHeatValue() > 50.0).count();
        stats.put("totalChunks", this.heatMap.size());
        stats.put("averageHeat", averageHeat);
        stats.put("coldChunks", coldChunks);
        stats.put("hotChunks", hotChunks);
        stats.put("totalHeat", totalHeat);
        return stats;
    }

    public void onPlayerQuit(Player player) {
        this.playerMovement.remove(player.getUniqueId());
    }

    public void cleanup() {
        this.heatMap.clear();
        this.playerMovement.clear();
    }

    public static class PlayerMovementData {
        private final Queue<Location> recentLocations = new LinkedList<Location>();
        private final int maxHistory = 50;
        private Location predictedDestination;
        private double directionX;
        private double directionZ;

        public void addLocation(Location loc) {
            this.recentLocations.offer(loc);
            if (this.recentLocations.size() > 50) {
                this.recentLocations.poll();
            }
            this.calculateMovementPattern();
        }

        private void calculateMovementPattern() {
            if (this.recentLocations.size() < 3) {
                return;
            }
            ArrayList<Location> locs = new ArrayList<Location>(this.recentLocations);
            int size = locs.size();
            double avgDirX = 0.0;
            double avgDirZ = 0.0;
            for (int i = 1; i < size; ++i) {
                Location prev = (Location)locs.get(i - 1);
                Location curr = (Location)locs.get(i);
                double dx = curr.getX() - prev.getX();
                double dz = curr.getZ() - prev.getZ();
                avgDirX += dx;
                avgDirZ += dz;
            }
            this.directionX = avgDirX / (double)(size - 1);
            this.directionZ = avgDirZ / (double)(size - 1);
            Location last = (Location)locs.get(size - 1);
            double predictX = last.getX() + this.directionX * 10.0;
            double predictZ = last.getZ() + this.directionZ * 10.0;
            this.predictedDestination = new Location(last.getWorld(), predictX, last.getY(), predictZ);
        }

        public Set<Chunk> getPredictedChunks(int radius) {
            HashSet<Chunk> chunks = new HashSet<Chunk>();
            if (this.predictedDestination == null) {
                return chunks;
            }
            World world = this.predictedDestination.getWorld();
            int centerX = this.predictedDestination.getBlockX() >> 4;
            int centerZ = this.predictedDestination.getBlockZ() >> 4;
            for (int x = -radius; x <= radius; ++x) {
                for (int z = -radius; z <= radius; ++z) {
                    chunks.add(world.getChunkAt(centerX + x, centerZ + z));
                }
            }
            return chunks;
        }
    }

    public static class ChunkIdentifier {
        public final String worldName;
        public final int x;
        public final int z;

        public ChunkIdentifier(Chunk chunk) {
            this.worldName = chunk.getWorld().getName();
            this.x = chunk.getX();
            this.z = chunk.getZ();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ChunkIdentifier)) {
                return false;
            }
            ChunkIdentifier that = (ChunkIdentifier)o;
            return this.x == that.x && this.z == that.z && this.worldName.equals(that.worldName);
        }

        public int hashCode() {
            return Objects.hash(this.worldName, this.x, this.z);
        }
    }

    public static class ChunkHeatData {
        private final AtomicLong lastAccess = new AtomicLong(System.currentTimeMillis());
        private final AtomicLong loadTime = new AtomicLong(System.currentTimeMillis());
        private final AtomicInteger visitCount = new AtomicInteger(0);
        private volatile double heatValue = 5.0;
        private volatile double peakHeat = 5.0;
        private volatile boolean hasImportantEntities;
        private volatile int entityCount;

        public void recordVisit() {
            this.lastAccess.set(System.currentTimeMillis());
            this.visitCount.incrementAndGet();
        }

        public void updateHeat(double newHeat) {
            this.heatValue = newHeat;
            if (newHeat > this.peakHeat) {
                this.peakHeat = newHeat;
            }
        }

        public double getHeatValue() {
            return this.heatValue;
        }

        public long getAge() {
            return System.currentTimeMillis() - this.loadTime.get();
        }

        public long getTimeSinceLastAccess() {
            return System.currentTimeMillis() - this.lastAccess.get();
        }

        public int getVisitCount() {
            return this.visitCount.get();
        }
    }
}

