/*
 * Decompiled with CFR 0.152.
 */
package cm.chunkManager.metrics;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.bukkit.plugin.java.JavaPlugin;

public class MetricsCollector {
    private final JavaPlugin plugin;
    private final Map<String, MetricData> metrics;
    private final Map<String, TimeSeries> timeSeries;
    private final ScheduledExecutorService scheduler;
    private final Path metricsDirectory;
    private final SimpleDateFormat dateFormat;
    private boolean exportEnabled;
    private int exportInterval;

    public MetricsCollector(JavaPlugin plugin) {
        this.plugin = plugin;
        this.metrics = new ConcurrentHashMap<String, MetricData>();
        this.timeSeries = new ConcurrentHashMap<String, TimeSeries>();
        this.scheduler = Executors.newScheduledThreadPool(2);
        this.metricsDirectory = Paths.get(plugin.getDataFolder().getPath(), "metrics");
        this.dateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
        try {
            Files.createDirectories(this.metricsDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            plugin.getLogger().warning("Failed to create metrics directory: " + e.getMessage());
        }
        this.initializeMetrics();
    }

    private void initializeMetrics() {
        this.registerMetric("chunks_loaded", 1000);
        this.registerMetric("chunks_unloaded", 1000);
        this.registerMetric("cache_hits", 1000);
        this.registerMetric("cache_misses", 1000);
        this.registerMetric("tick_time", 1000);
        this.registerMetric("memory_usage", 1000);
        this.registerMetric("player_movements", 1000);
        this.registerMetric("optimization_runs", 100);
        this.registerMetric("gc_runs", 100);
        this.registerMetric("prefetch_success", 1000);
        this.registerMetric("prefetch_failure", 1000);
        this.registerTimeSeries("tps", 300, 1000L);
        this.registerTimeSeries("memory_mb", 300, 5000L);
        this.registerTimeSeries("loaded_chunks", 300, 5000L);
        this.registerTimeSeries("online_players", 300, 5000L);
    }

    public void registerMetric(String name, int maxSamples) {
        this.metrics.putIfAbsent(name, new MetricData(name, maxSamples));
    }

    public void registerTimeSeries(String name, int maxPoints, long intervalMs) {
        this.timeSeries.putIfAbsent(name, new TimeSeries(name, maxPoints, intervalMs));
    }

    public void recordMetric(String name, long value) {
        MetricData metric = this.metrics.get(name);
        if (metric != null) {
            metric.record(value);
        }
    }

    public void recordTimeSeriesPoint(String name, double value) {
        TimeSeries series = this.timeSeries.get(name);
        if (series != null) {
            series.addPoint(value);
        }
    }

    public void startExporting(boolean enabled, int intervalSeconds) {
        this.exportEnabled = enabled;
        this.exportInterval = intervalSeconds;
        if (this.exportEnabled) {
            this.scheduler.scheduleAtFixedRate(this::exportMetrics, intervalSeconds, intervalSeconds, TimeUnit.SECONDS);
        }
    }

    private void exportMetrics() {
        if (!this.exportEnabled) {
            return;
        }
        String filename = "metrics_" + this.dateFormat.format(new Date()) + ".json";
        Path filePath = this.metricsDirectory.resolve(filename);
        HashMap<String, Object> export = new HashMap<String, Object>();
        export.put("timestamp", System.currentTimeMillis());
        export.put("server", this.plugin.getServer().getName());
        export.put("version", this.plugin.getServer().getVersion());
        HashMap<String, Map<String, Object>> metricsData = new HashMap<String, Map<String, Object>>();
        for (Map.Entry<String, MetricData> entry : this.metrics.entrySet()) {
            metricsData.put(entry.getKey(), entry.getValue().toMap());
        }
        export.put("metrics", metricsData);
        HashMap timeSeriesData = new HashMap();
        for (Map.Entry<String, TimeSeries> entry : this.timeSeries.entrySet()) {
            TimeSeries series = entry.getValue();
            HashMap<String, Object> seriesData = new HashMap<String, Object>();
            seriesData.put("trend", series.getTrend());
            seriesData.put("recent", series.getRecentPoints(20));
            timeSeriesData.put(entry.getKey(), seriesData);
        }
        export.put("timeseries", timeSeriesData);
        try (BufferedWriter bufferedWriter = Files.newBufferedWriter(filePath, new OpenOption[0]);){
            bufferedWriter.write(this.toJson(export));
        }
        catch (IOException iOException) {
            this.plugin.getLogger().warning("Failed to export metrics: " + iOException.getMessage());
        }
        this.cleanOldMetrics();
    }

    private void cleanOldMetrics() {
        try {
            long cutoff = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(7L);
            Files.list(this.metricsDirectory).filter(path -> {
                try {
                    return Files.getLastModifiedTime(path, new LinkOption[0]).toMillis() < cutoff;
                }
                catch (IOException e) {
                    return false;
                }
            }).forEach(path -> {
                try {
                    Files.delete(path);
                }
                catch (IOException e) {
                    this.plugin.getLogger().warning("Failed to delete old metrics file: " + String.valueOf(path));
                }
            });
        }
        catch (IOException e) {
            this.plugin.getLogger().warning("Failed to clean old metrics: " + e.getMessage());
        }
    }

    private String toJson(Map<String, Object> map) {
        StringBuilder json = new StringBuilder();
        json.append("{");
        Iterator<Map.Entry<String, Object>> iter = map.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, Object> entry = iter.next();
            json.append("\"").append(entry.getKey()).append("\":");
            Object value = entry.getValue();
            if (value instanceof Map) {
                json.append(this.toJson((Map)value));
            } else if (value instanceof List) {
                json.append("[");
                List list = (List)value;
                for (int i = 0; i < list.size(); ++i) {
                    if (i > 0) {
                        json.append(",");
                    }
                    json.append(list.get(i));
                }
                json.append("]");
            } else if (value instanceof String) {
                json.append("\"").append(value).append("\"");
            } else {
                json.append(value);
            }
            if (!iter.hasNext()) continue;
            json.append(",");
        }
        json.append("}");
        return json.toString();
    }

    public Map<String, Object> getCurrentMetrics() {
        HashMap<String, Object> current = new HashMap<String, Object>();
        for (Map.Entry<String, MetricData> entry : this.metrics.entrySet()) {
            current.put(entry.getKey(), entry.getValue().toMap());
        }
        return current;
    }

    public void shutdown() {
        if (this.exportEnabled) {
            this.exportMetrics();
        }
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
        }
    }

    public static class MetricData {
        private final String name;
        private final AtomicLong count;
        private final AtomicLong sum;
        private final AtomicLong min;
        private final AtomicLong max;
        private final List<Long> samples;
        private final int maxSamples;

        public MetricData(String name, int maxSamples) {
            this.name = name;
            this.count = new AtomicLong(0L);
            this.sum = new AtomicLong(0L);
            this.min = new AtomicLong(Long.MAX_VALUE);
            this.max = new AtomicLong(Long.MIN_VALUE);
            this.samples = Collections.synchronizedList(new ArrayList());
            this.maxSamples = maxSamples;
        }

        public void record(long value) {
            this.count.incrementAndGet();
            this.sum.addAndGet(value);
            long currentMin = this.min.get();
            while (value < currentMin && !this.min.compareAndSet(currentMin, value)) {
                currentMin = this.min.get();
            }
            long currentMax = this.max.get();
            while (value > currentMax && !this.max.compareAndSet(currentMax, value)) {
                currentMax = this.max.get();
            }
            this.samples.add(value);
            if (this.samples.size() > this.maxSamples) {
                this.samples.remove(0);
            }
        }

        public double getAverage() {
            long c = this.count.get();
            return c > 0L ? (double)this.sum.get() / (double)c : 0.0;
        }

        public double getMedian() {
            if (this.samples.isEmpty()) {
                return 0.0;
            }
            ArrayList<Long> sorted = new ArrayList<Long>(this.samples);
            Collections.sort(sorted);
            int size = sorted.size();
            if (size % 2 == 0) {
                return (double)((Long)sorted.get(size / 2 - 1) + (Long)sorted.get(size / 2)) / 2.0;
            }
            return ((Long)sorted.get(size / 2)).longValue();
        }

        public double getPercentile(double percentile) {
            if (this.samples.isEmpty()) {
                return 0.0;
            }
            ArrayList<Long> sorted = new ArrayList<Long>(this.samples);
            Collections.sort(sorted);
            int index = (int)Math.ceil(percentile / 100.0 * (double)sorted.size()) - 1;
            return ((Long)sorted.get(Math.max(0, Math.min(index, sorted.size() - 1)))).longValue();
        }

        public Map<String, Object> toMap() {
            HashMap<String, Object> map = new HashMap<String, Object>();
            map.put("name", this.name);
            map.put("count", this.count.get());
            map.put("average", this.getAverage());
            map.put("median", this.getMedian());
            map.put("min", this.min.get() == Long.MAX_VALUE ? 0L : this.min.get());
            map.put("max", this.max.get() == Long.MIN_VALUE ? 0L : this.max.get());
            map.put("p95", this.getPercentile(95.0));
            map.put("p99", this.getPercentile(99.0));
            return map;
        }
    }

    public static class TimeSeries {
        private final String name;
        private final List<DataPoint> dataPoints;
        private final int maxPoints;
        private final long intervalMs;

        public TimeSeries(String name, int maxPoints, long intervalMs) {
            this.name = name;
            this.dataPoints = Collections.synchronizedList(new ArrayList());
            this.maxPoints = maxPoints;
            this.intervalMs = intervalMs;
        }

        public void addPoint(double value) {
            this.dataPoints.add(new DataPoint(System.currentTimeMillis(), value));
            if (this.dataPoints.size() > this.maxPoints) {
                this.dataPoints.remove(0);
            }
        }

        public List<DataPoint> getRecentPoints(int count) {
            int start = Math.max(0, this.dataPoints.size() - count);
            return new ArrayList<DataPoint>(this.dataPoints.subList(start, this.dataPoints.size()));
        }

        public double getTrend() {
            if (this.dataPoints.size() < 2) {
                return 0.0;
            }
            int n = Math.min(10, this.dataPoints.size());
            List<DataPoint> recent = this.getRecentPoints(n);
            double sumX = 0.0;
            double sumY = 0.0;
            double sumXY = 0.0;
            double sumX2 = 0.0;
            for (int i = 0; i < recent.size(); ++i) {
                sumX += (double)i;
                sumY += recent.get((int)i).value;
                sumXY += (double)i * recent.get((int)i).value;
                sumX2 += (double)(i * i);
            }
            double slope = ((double)n * sumXY - sumX * sumY) / ((double)n * sumX2 - sumX * sumX);
            return slope;
        }

        public static class DataPoint {
            public final long timestamp;
            public final double value;

            public DataPoint(long timestamp, double value) {
                this.timestamp = timestamp;
                this.value = value;
            }
        }
    }
}

