/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.core;

import NC.noChance.core.ViolationType;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

public class CrossCheckCorrelation {
    private final Map<UUID, CorrelationTracker> playerTrackers = new ConcurrentHashMap<UUID, CorrelationTracker>();
    private static final long CORRELATION_WINDOW_MS = 5000L;
    private static final long EXTENDED_WINDOW_MS = 10000L;
    private static final double COMBAT_CLUSTER_WEIGHT = 1.5;
    private static final double MOVEMENT_CLUSTER_WEIGHT = 1.4;
    private static final double BLOCK_CLUSTER_WEIGHT = 1.3;
    private static final double EVASION_CLUSTER_WEIGHT = 1.6;

    public void recordViolation(UUID playerId, ViolationType type, double severity) {
        CorrelationTracker tracker = this.playerTrackers.computeIfAbsent(playerId, k -> new CorrelationTracker());
        tracker.recordViolation(type, severity);
    }

    public CorrelationResult analyzeCorrelation(UUID playerId, ViolationType currentType) {
        CorrelationTracker tracker = this.playerTrackers.get(playerId);
        if (tracker == null) {
            return new CorrelationResult(0.0, CorrelationType.NONE, new ArrayList<ViolationType>(), 1.0);
        }
        long now = System.currentTimeMillis();
        List<ViolationEntry> recentViolations = tracker.getRecentViolations(5000L);
        if (recentViolations.isEmpty()) {
            return new CorrelationResult(0.0, CorrelationType.NONE, new ArrayList<ViolationType>(), 1.0);
        }
        CorrelationType clusterType = this.determineClusterType(currentType, recentViolations);
        List<ViolationType> correlatedTypes = this.extractCorrelatedTypes(recentViolations);
        double baseScore = this.calculateCorrelationScore(currentType, recentViolations, clusterType);
        double timeDecayFactor = this.calculateTimeDecay(recentViolations, now);
        double severityBonus = this.calculateSeverityBonus(recentViolations);
        double finalScore = baseScore * timeDecayFactor + severityBonus;
        double punishmentMultiplier = this.calculatePunishmentMultiplier(clusterType, correlatedTypes.size(), finalScore);
        return new CorrelationResult(finalScore, clusterType, correlatedTypes, punishmentMultiplier);
    }

    private CorrelationType determineClusterType(ViolationType current, List<ViolationEntry> recent) {
        HashSet<ViolationType> allTypes = new HashSet<ViolationType>();
        allTypes.add(current);
        for (ViolationEntry entry : recent) {
            allTypes.add(entry.type);
        }
        int combatCount = 0;
        int movementCount = 0;
        int blockCount = 0;
        int evasionCount = 0;
        for (ViolationType type : allTypes) {
            if (this.isCombatType(type)) {
                ++combatCount;
            }
            if (this.isMovementType(type)) {
                ++movementCount;
            }
            if (this.isBlockType(type)) {
                ++blockCount;
            }
            if (!this.isEvasionType(type)) continue;
            ++evasionCount;
        }
        if (combatCount >= 2) {
            return CorrelationType.COMBAT_CLUSTER;
        }
        if (movementCount >= 2) {
            return CorrelationType.MOVEMENT_CLUSTER;
        }
        if (blockCount >= 2) {
            return CorrelationType.BLOCK_CLUSTER;
        }
        if (evasionCount >= 2) {
            return CorrelationType.EVASION_CLUSTER;
        }
        if (combatCount >= 1 && movementCount >= 1) {
            return CorrelationType.HYBRID_COMBAT_MOVEMENT;
        }
        if (combatCount >= 1 && evasionCount >= 1) {
            return CorrelationType.HYBRID_COMBAT_EVASION;
        }
        return CorrelationType.MIXED;
    }

    private boolean isCombatType(ViolationType type) {
        return type == ViolationType.KILLAURA || type == ViolationType.KILLAURA_MULTI || type == ViolationType.KILLAURA_ANGLE || type == ViolationType.KILLAURA_ROTATION || type == ViolationType.KILLAURA_PATTERN || type == ViolationType.REACH || type == ViolationType.AUTOCLICKER || type == ViolationType.CRITICALS;
    }

    private boolean isMovementType(ViolationType type) {
        return type == ViolationType.FLY || type == ViolationType.SPEED || type == ViolationType.NOCLIP || type == ViolationType.JESUS || type == ViolationType.STEP || type == ViolationType.TIMER || type == ViolationType.GROUNDSPOOF || type == ViolationType.ELYTRAFLY || type == ViolationType.STRIDER || type == ViolationType.BOATFLY;
    }

    private boolean isBlockType(ViolationType type) {
        return type == ViolationType.SCAFFOLD || type == ViolationType.FASTPLACE || type == ViolationType.FASTBREAK || type == ViolationType.NUKER;
    }

    private boolean isEvasionType(ViolationType type) {
        return type == ViolationType.BLINK || type == ViolationType.NOFALL || type == ViolationType.PHASE || type == ViolationType.VELOCITY;
    }

    private List<ViolationType> extractCorrelatedTypes(List<ViolationEntry> violations) {
        ArrayList<ViolationType> types = new ArrayList<ViolationType>();
        for (ViolationEntry entry : violations) {
            types.add(entry.type);
        }
        return types;
    }

    private double calculateCorrelationScore(ViolationType current, List<ViolationEntry> recent, CorrelationType cluster) {
        if (recent.isEmpty()) {
            return 0.0;
        }
        int matchingClusterCount = 0;
        for (ViolationEntry entry : recent) {
            if (!this.isInCluster(entry.type, cluster)) continue;
            ++matchingClusterCount;
        }
        double clusterRatio = (double)matchingClusterCount / (double)Math.max(1, recent.size());
        double baseScore = clusterRatio * 0.6;
        if (recent.size() >= 5) {
            baseScore += 0.25;
        } else if (recent.size() >= 3) {
            baseScore += 0.15;
        }
        HashSet<ViolationType> uniqueTypes = new HashSet<ViolationType>();
        for (ViolationEntry entry : recent) {
            uniqueTypes.add(entry.type);
        }
        if (uniqueTypes.size() >= 4) {
            baseScore += 0.2;
        } else if (uniqueTypes.size() >= 3) {
            baseScore += 0.12;
        } else if (uniqueTypes.size() >= 2) {
            baseScore += 0.06;
        }
        return Math.min(1.0, baseScore);
    }

    private boolean isInCluster(ViolationType type, CorrelationType cluster) {
        switch (cluster.ordinal()) {
            case 0: {
                return this.isCombatType(type);
            }
            case 1: {
                return this.isMovementType(type);
            }
            case 2: {
                return this.isBlockType(type);
            }
            case 3: {
                return this.isEvasionType(type);
            }
            case 4: {
                return this.isCombatType(type) || this.isMovementType(type);
            }
            case 5: {
                return this.isCombatType(type) || this.isEvasionType(type);
            }
        }
        return false;
    }

    private double calculateTimeDecay(List<ViolationEntry> violations, long now) {
        if (violations.isEmpty()) {
            return 0.0;
        }
        double totalDecay = 0.0;
        for (ViolationEntry entry : violations) {
            long age = now - entry.timestamp;
            double decay = 1.0 - (double)age / 5000.0;
            totalDecay += Math.max(0.0, decay);
        }
        return Math.min(1.0, totalDecay / (double)violations.size());
    }

    private double calculateSeverityBonus(List<ViolationEntry> violations) {
        if (violations.isEmpty()) {
            return 0.0;
        }
        double avgSeverity = violations.stream().mapToDouble(v -> v.severity).average().orElse(0.0);
        if (avgSeverity > 0.85) {
            return 0.15;
        }
        if (avgSeverity > 0.7) {
            return 0.1;
        }
        if (avgSeverity > 0.55) {
            return 0.05;
        }
        return 0.0;
    }

    private double calculatePunishmentMultiplier(CorrelationType cluster, int correlatedCount, double score) {
        double baseMultiplier = 1.0;
        switch (cluster.ordinal()) {
            case 0: {
                baseMultiplier = 1.5;
                break;
            }
            case 1: {
                baseMultiplier = 1.4;
                break;
            }
            case 2: {
                baseMultiplier = 1.3;
                break;
            }
            case 3: {
                baseMultiplier = 1.6;
                break;
            }
            case 4: {
                baseMultiplier = 1.7;
                break;
            }
            case 5: {
                baseMultiplier = 1.8;
                break;
            }
            case 6: {
                baseMultiplier = 1.2;
            }
        }
        if (correlatedCount >= 5) {
            baseMultiplier += 0.4;
        } else if (correlatedCount >= 3) {
            baseMultiplier += 0.2;
        }
        if (score > 0.8) {
            baseMultiplier += 0.3;
        } else if (score > 0.65) {
            baseMultiplier += 0.15;
        }
        return Math.min(3.0, baseMultiplier);
    }

    public void cleanup(UUID playerId) {
        this.playerTrackers.remove(playerId);
    }

    public void cleanupOldEntries() {
        long cutoff = System.currentTimeMillis() - 10000L;
        for (CorrelationTracker tracker : this.playerTrackers.values()) {
            tracker.removeOldEntries(cutoff);
        }
    }

    private static class CorrelationTracker {
        private final List<ViolationEntry> violations = new ArrayList<ViolationEntry>();

        private CorrelationTracker() {
        }

        private synchronized void recordViolation(ViolationType type, double severity) {
            this.violations.add(new ViolationEntry(type, severity, System.currentTimeMillis()));
        }

        private synchronized List<ViolationEntry> getRecentViolations(long windowMs) {
            long cutoff = System.currentTimeMillis() - windowMs;
            ArrayList<ViolationEntry> recent = new ArrayList<ViolationEntry>();
            for (ViolationEntry entry : this.violations) {
                if (entry.timestamp <= cutoff) continue;
                recent.add(entry);
            }
            return recent;
        }

        private synchronized void removeOldEntries(long cutoff) {
            this.violations.removeIf(entry -> entry.timestamp < cutoff);
        }
    }

    public static class CorrelationResult {
        public final double score;
        public final CorrelationType clusterType;
        public final List<ViolationType> correlatedTypes;
        public final double punishmentMultiplier;

        private CorrelationResult(double score, CorrelationType clusterType, List<ViolationType> correlatedTypes, double punishmentMultiplier) {
            this.score = score;
            this.clusterType = clusterType;
            this.correlatedTypes = correlatedTypes;
            this.punishmentMultiplier = punishmentMultiplier;
        }
    }

    public static enum CorrelationType {
        COMBAT_CLUSTER,
        MOVEMENT_CLUSTER,
        BLOCK_CLUSTER,
        EVASION_CLUSTER,
        HYBRID_COMBAT_MOVEMENT,
        HYBRID_COMBAT_EVASION,
        MIXED,
        NONE;

    }

    private static class ViolationEntry {
        private final ViolationType type;
        private final double severity;
        private final long timestamp;

        private ViolationEntry(ViolationType type, double severity, long timestamp) {
            this.type = type;
            this.severity = severity;
            this.timestamp = timestamp;
        }
    }
}

