/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.core;

import NC.noChance.core.ACConfig;
import NC.noChance.core.CheckResult;
import NC.noChance.core.CrossCheckCorrelation;
import NC.noChance.core.PlayerData;
import NC.noChance.core.ThresholdAdapter;
import NC.noChance.core.VariantCheck;
import NC.noChance.core.ViolationQueue;
import NC.noChance.core.ViolationType;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.entity.Player;

public class DetectionEngine {
    private final Map<UUID, BehavioralProfile> profiles;
    private final Map<UUID, SuspicionLevel> suspicionLevels;
    private final ACConfig config;
    private final ViolationQueue violationBuffer;
    private final ThresholdAdapter thresholdManager;
    private final VariantCheck variantCheck;
    private final CrossCheckCorrelation correlationSystem;
    private static final int ENTROPY_WINDOW = 50;
    private static final double HUMAN_ENTROPY_MIN = 0.35;
    private static final double BOT_ENTROPY_MAX = 0.12;

    public DetectionEngine(ACConfig config) {
        this.config = config;
        this.profiles = new ConcurrentHashMap<UUID, BehavioralProfile>();
        this.suspicionLevels = new ConcurrentHashMap<UUID, SuspicionLevel>();
        this.violationBuffer = new ViolationQueue();
        this.thresholdManager = new ThresholdAdapter();
        this.variantCheck = new VariantCheck();
        this.correlationSystem = new CrossCheckCorrelation();
    }

    public AnalysisResult analyze(Player player, PlayerData data, CheckResult checkResult, ViolationType type) {
        boolean bufferDecision;
        UUID playerId = player.getUniqueId();
        BehavioralProfile profile = this.profiles.computeIfAbsent(playerId, k -> new BehavioralProfile());
        SuspicionLevel suspicion = this.suspicionLevels.computeIfAbsent(playerId, k -> SuspicionLevel.CLEAN);
        this.thresholdManager.updateTPS();
        String variant = this.variantCheck.getPlayerVariant(playerId, type);
        VariantCheck.ThresholdSet thresholds = this.variantCheck.getThresholds(playerId, type);
        profile.recordCheck(checkResult, type);
        if (checkResult.isFailed()) {
            this.correlationSystem.recordViolation(playerId, type, checkResult.getSeverity());
        }
        CrossCheckCorrelation.CorrelationResult correlation = this.correlationSystem.analyzeCorrelation(playerId, type);
        double entropyScore = this.calculateMovementEntropy(data, type);
        double consistencyScore = this.calculateConsistencyScore(profile, type);
        double correlationScore = this.calculateCrossMetricCorrelation(data, profile, type);
        double anomalyScore = this.calculateStatisticalAnomaly(data, profile, type);
        double fingerprintScore = this.calculateBehavioralFingerprint(data, profile);
        correlationScore = Math.max(correlationScore, correlation.score);
        EnsembleVote vote = this.performEnsembleVoting(entropyScore, consistencyScore, correlationScore, anomalyScore, fingerprintScore, suspicion, type);
        double finalConfidence = vote.confidence;
        boolean shouldFlag = vote.shouldFlag;
        double adjustedThreshold = Math.max(this.thresholdManager.getAdjustedThreshold(type), thresholds.minConfidence);
        double toleranceMultiplier = this.thresholdManager.getToleranceMultiplier(type);
        if (finalConfidence < adjustedThreshold && !this.thresholdManager.isServerLagging()) {
            shouldFlag = false;
        }
        if (checkResult.isFailed() && !(bufferDecision = this.violationBuffer.shouldFlag(playerId, type, checkResult.getSeverity() * correlation.punishmentMultiplier, checkResult.getDetails()))) {
            shouldFlag = false;
        }
        if (shouldFlag && finalConfidence >= thresholds.highConfidence) {
            this.suspicionLevels.put(playerId, SuspicionLevel.CONFIRMED);
            profile.incrementConfirmedViolations();
            this.thresholdManager.recordTruePositive(type);
            this.variantCheck.recordDetection(playerId, type, true);
        } else if (finalConfidence >= 0.78) {
            this.suspicionLevels.put(playerId, SuspicionLevel.HIGH);
            this.variantCheck.recordDetection(playerId, type, true);
        } else if (finalConfidence >= 0.68) {
            this.suspicionLevels.put(playerId, SuspicionLevel.MEDIUM);
        } else if (finalConfidence >= 0.52) {
            this.suspicionLevels.put(playerId, SuspicionLevel.LOW);
        } else if (finalConfidence < 0.42) {
            this.suspicionLevels.put(playerId, SuspicionLevel.CLEAN);
            if (checkResult.isFailed() && finalConfidence < 0.3) {
                this.thresholdManager.recordFalsePositive(type);
                this.variantCheck.recordFalsePositive(playerId, type);
            }
        }
        return new AnalysisResult(shouldFlag, finalConfidence, vote.detectionMethod, suspicion, entropyScore, consistencyScore, correlationScore, anomalyScore, fingerprintScore, variant, correlation.clusterType, correlation.punishmentMultiplier);
    }

    private double calculateMovementEntropy(PlayerData data, ViolationType type) {
        if (type == ViolationType.FLY || type == ViolationType.SPEED || type == ViolationType.NOCLIP || type == ViolationType.JESUS || type == ViolationType.NOSLOW) {
            Deque<PlayerData.VelocityData> velocities = data.getVelocityHistory();
            if (velocities.size() < 10) {
                return 0.5;
            }
            double[] magnitudes = new double[velocities.size()];
            int i = 0;
            for (PlayerData.VelocityData v : velocities) {
                magnitudes[i++] = Math.sqrt(v.x * v.x + v.y * v.y + v.z * v.z);
            }
            double entropy = this.calculateShannonEntropy(magnitudes);
            if (entropy < 0.12) {
                return 0.95;
            }
            if (entropy > 0.35) {
                return 0.1;
            }
            return Math.max(0.0, (0.35 - entropy) / 0.35);
        }
        if (type == ViolationType.KILLAURA || type == ViolationType.KILLAURA_ROTATION || type == ViolationType.KILLAURA_ANGLE || type == ViolationType.KILLAURA_PATTERN) {
            double cv;
            Deque<Long> clicks = data.getClickHistory();
            if (clicks.size() < 20) {
                return 0.5;
            }
            double sum = 0.0;
            double sumSquared = 0.0;
            int count = 0;
            Long prev = null;
            for (Long click : clicks) {
                if (prev != null) {
                    long interval = click - prev;
                    sum += (double)interval;
                    sumSquared += (double)(interval * interval);
                    ++count;
                }
                prev = click;
            }
            if (count == 0) {
                return 0.5;
            }
            double mean = sum / (double)count;
            double variance = sumSquared / (double)count - mean * mean;
            double d = cv = mean > 0.01 ? Math.sqrt(variance) / mean : 0.0;
            if (cv < 0.08) {
                return 0.96;
            }
            if (cv > 0.35) {
                return 0.08;
            }
            return Math.max(0.0, (0.35 - cv) / 0.35);
        }
        return 0.5;
    }

    private double calculateConsistencyScore(BehavioralProfile profile, ViolationType type) {
        List<Double> recentSeverities = profile.getRecentSeverities(type, 30);
        if (recentSeverities.size() < 5) {
            return 0.3;
        }
        double mean = recentSeverities.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
        double variance = this.calculateVariance(recentSeverities.stream().mapToDouble(Double::doubleValue).toArray());
        double stdDev = Math.sqrt(variance);
        if (stdDev < 0.05 && mean > 0.7) {
            return 0.95;
        }
        if (stdDev > 0.3) {
            return 0.2;
        }
        double consistencyRatio = mean > 0.01 ? 1.0 - Math.min(1.0, stdDev / mean) : 0.5;
        return mean * 0.7 + consistencyRatio * 0.3;
    }

    private double calculateCrossMetricCorrelation(PlayerData data, BehavioralProfile profile, ViolationType type) {
        int correlatedViolations = 0;
        int totalChecks = 0;
        Map<ViolationType, Integer> violationCounts = profile.getViolationCounts();
        if (type == ViolationType.KILLAURA || type.name().startsWith("KILLAURA")) {
            if (violationCounts.getOrDefault((Object)ViolationType.AUTOCLICKER, 0) > 5) {
                ++correlatedViolations;
            }
            if (violationCounts.getOrDefault((Object)ViolationType.REACH, 0) > 3) {
                ++correlatedViolations;
            }
            if (data.getAverageCPS() > 20.0) {
                ++correlatedViolations;
            }
            if (data.getAverageRotationSpeed() > 500.0) {
                ++correlatedViolations;
            }
            totalChecks = 4;
        } else if (type == ViolationType.FLY) {
            if (violationCounts.getOrDefault((Object)ViolationType.SPEED, 0) > 5) {
                ++correlatedViolations;
            }
            if (violationCounts.getOrDefault((Object)ViolationType.NOCLIP, 0) > 3) {
                ++correlatedViolations;
            }
            if (data.getAirTicks() > 100) {
                ++correlatedViolations;
            }
            totalChecks = 3;
        } else if (type == ViolationType.SCAFFOLD) {
            if (violationCounts.getOrDefault((Object)ViolationType.FASTPLACE, 0) > 5) {
                ++correlatedViolations;
            }
            if (violationCounts.getOrDefault((Object)ViolationType.KILLAURA, 0) > 3) {
                ++correlatedViolations;
            }
            totalChecks = 2;
        }
        if (totalChecks == 0) {
            return 0.5;
        }
        return (double)correlatedViolations / (double)totalChecks;
    }

    private double calculateStatisticalAnomaly(PlayerData data, BehavioralProfile profile, ViolationType type) {
        double zScore;
        List<Double> recentSeverities = profile.getRecentSeverities(type, 50);
        if (recentSeverities.isEmpty()) {
            return 0.0;
        }
        double mean = recentSeverities.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
        double variance = this.calculateVariance(recentSeverities.stream().mapToDouble(Double::doubleValue).toArray());
        double stdDev = Math.sqrt(variance);
        double currentSeverity = recentSeverities.get(recentSeverities.size() - 1);
        double d = zScore = stdDev > 0.0 ? Math.abs((currentSeverity - mean) / stdDev) : 0.0;
        if (zScore > 3.5) {
            return 0.96;
        }
        if (zScore > 2.5) {
            return 0.78;
        }
        if (zScore > 1.8) {
            return 0.52;
        }
        return 0.18;
    }

    private double calculateBehavioralFingerprint(PlayerData data, BehavioralProfile profile) {
        double violationRatio;
        double fingerprintScore = 0.0;
        int factors = 0;
        if (data.getClickHistory().size() >= 20) {
            double intervalDiversity;
            Deque<Long> clicks = data.getClickHistory();
            HashSet<Long> uniqueIntervals = new HashSet<Long>();
            int intervalCount = 0;
            Long prev = null;
            for (Long click : clicks) {
                if (prev != null) {
                    uniqueIntervals.add(click - prev);
                    ++intervalCount;
                }
                prev = click;
            }
            double d = intervalDiversity = intervalCount > 0 ? (double)uniqueIntervals.size() / (double)intervalCount : 0.0;
            if (intervalDiversity < 0.1) {
                fingerprintScore += 1.0;
            } else if (intervalDiversity > 0.6) {
                fingerprintScore -= 0.5;
            }
            ++factors;
        }
        if (data.getRotationHistory().size() >= 10) {
            double yawVariance;
            Deque<PlayerData.RotationData> rotations = data.getRotationHistory();
            double sum = 0.0;
            double sumSquared = 0.0;
            int count = 0;
            PlayerData.RotationData prev = null;
            for (PlayerData.RotationData rot : rotations) {
                if (prev != null) {
                    float change = Math.abs(rot.yaw - prev.yaw);
                    sum += (double)change;
                    sumSquared += (double)(change * change);
                    ++count;
                }
                prev = rot;
            }
            double mean = count > 0 ? sum / (double)count : 0.0;
            double d = yawVariance = count > 0 ? sumSquared / (double)count - mean * mean : 0.0;
            if (yawVariance < 1.0) {
                fingerprintScore += 0.8;
            } else if (yawVariance > 100.0) {
                fingerprintScore -= 0.3;
            }
            ++factors;
        }
        if ((violationRatio = data.getViolationRatio()) > 0.5) {
            fingerprintScore += 1.0;
            ++factors;
        } else if (violationRatio < 0.05) {
            fingerprintScore -= 0.5;
            ++factors;
        }
        if (profile.getConfirmedViolations() > 10) {
            fingerprintScore += 1.5;
            ++factors;
        }
        return factors > 0 ? Math.max(0.0, Math.min(1.0, (fingerprintScore + (double)factors * 0.5) / (double)(factors * 2))) : 0.5;
    }

    private EnsembleVote performEnsembleVoting(double entropy, double consistency, double correlation, double anomaly, double fingerprint, SuspicionLevel suspicion, ViolationType type) {
        double[] scores = new double[]{entropy, consistency, correlation, anomaly, fingerprint};
        double[] weights = new double[]{0.24, 0.26, 0.24, 0.18, 0.08};
        if (suspicion == SuspicionLevel.CONFIRMED) {
            weights = new double[]{0.22, 0.28, 0.28, 0.16, 0.06};
        } else if (suspicion == SuspicionLevel.HIGH) {
            weights = new double[]{0.24, 0.26, 0.24, 0.18, 0.08};
        } else if (suspicion == SuspicionLevel.CLEAN) {
            weights = new double[]{0.26, 0.22, 0.2, 0.22, 0.1};
        }
        double weightedScore = 0.0;
        for (int i = 0; i < scores.length; ++i) {
            weightedScore += scores[i] * weights[i];
        }
        int votesForCheat = 0;
        int totalVotes = 0;
        if (entropy > 0.62) {
            ++votesForCheat;
            ++totalVotes;
        }
        if (consistency > 0.62) {
            ++votesForCheat;
            ++totalVotes;
        }
        if (correlation > 0.58) {
            ++votesForCheat;
            ++totalVotes;
        }
        if (anomaly > 0.62) {
            ++votesForCheat;
            ++totalVotes;
        }
        if (fingerprint > 0.62) {
            ++votesForCheat;
            ++totalVotes;
        }
        double voteRatio = totalVotes > 0 ? (double)votesForCheat / (double)totalVotes : 0.0;
        double finalConfidence = weightedScore * 0.58 + voteRatio * 0.42;
        boolean shouldFlag = votesForCheat >= 2 && finalConfidence >= 0.66;
        String detectionMethod = this.getDominantDetectionMethod(entropy, consistency, correlation, anomaly, fingerprint);
        return new EnsembleVote(shouldFlag, finalConfidence, detectionMethod, votesForCheat, totalVotes);
    }

    private String getDominantDetectionMethod(double entropy, double consistency, double correlation, double anomaly, double fingerprint) {
        double max = Math.max(Math.max(Math.max(Math.max(entropy, consistency), correlation), anomaly), fingerprint);
        if (max == entropy) {
            return "Entropy";
        }
        if (max == consistency) {
            return "Consistency";
        }
        if (max == correlation) {
            return "Correlation";
        }
        if (max == anomaly) {
            return "Anomaly";
        }
        return "Fingerprint";
    }

    private double calculateShannonEntropy(double[] values) {
        if (values.length == 0) {
            return 0.0;
        }
        int bins = 10;
        double min = Double.MAX_VALUE;
        double max = Double.MIN_VALUE;
        for (double v : values) {
            if (v < min) {
                min = v;
            }
            if (!(v > max)) continue;
            max = v;
        }
        double range = max - min;
        if (range == 0.0) {
            return 0.0;
        }
        int[] histogram = new int[bins];
        for (double value : values) {
            int bin;
            int n = bin = Math.min(bins - 1, (int)((value - min) / range * (double)bins));
            histogram[n] = histogram[n] + 1;
        }
        double entropy = 0.0;
        int total = values.length;
        for (int count : histogram) {
            if (count <= 0) continue;
            double p = (double)count / (double)total;
            entropy -= p * (Math.log(p) / Math.log(2.0));
        }
        return entropy / Math.log(bins) / Math.log(2.0);
    }

    private double calculateVariance(double[] values) {
        if (values.length == 0) {
            return 0.0;
        }
        double mean = 0.0;
        for (double v : values) {
            mean += v;
        }
        mean /= (double)values.length;
        double variance = 0.0;
        for (double v : values) {
            variance += Math.pow(v - mean, 2.0);
        }
        return variance / (double)values.length;
    }

    public void cleanup(UUID playerId) {
        this.profiles.remove(playerId);
        this.suspicionLevels.remove(playerId);
        this.violationBuffer.cleanup(playerId);
        this.variantCheck.cleanup(playerId);
        this.correlationSystem.cleanup(playerId);
    }

    public void cleanupOldData() {
        this.correlationSystem.cleanupOldEntries();
        long now = System.currentTimeMillis();
        this.profiles.forEach((uuid, profile) -> profile.cleanupOldSeverities(now, 120000L));
    }

    public SuspicionLevel getSuspicionLevel(UUID playerId) {
        return this.suspicionLevels.getOrDefault(playerId, SuspicionLevel.CLEAN);
    }

    private static class BehavioralProfile {
        private final Map<ViolationType, List<SeverityRecord>> severityHistory = new HashMap<ViolationType, List<SeverityRecord>>();
        private final Map<ViolationType, Integer> violationCounts = new HashMap<ViolationType, Integer>();
        private int confirmedViolations = 0;

        public void recordCheck(CheckResult result, ViolationType type) {
            if (result.isFailed()) {
                this.severityHistory.computeIfAbsent(type, k -> new ArrayList()).add(new SeverityRecord(result.getSeverity(), System.currentTimeMillis()));
                this.violationCounts.put(type, this.violationCounts.getOrDefault((Object)type, 0) + 1);
            }
        }

        public List<Double> getRecentSeverities(ViolationType type, int count) {
            int start;
            List records = this.severityHistory.getOrDefault((Object)type, new ArrayList());
            ArrayList<Double> result = new ArrayList<Double>();
            for (int i = start = Math.max(0, records.size() - count); i < records.size(); ++i) {
                result.add(((SeverityRecord)records.get((int)i)).severity);
            }
            return result;
        }

        public Map<ViolationType, Integer> getViolationCounts() {
            return this.violationCounts;
        }

        public int getConfirmedViolations() {
            return this.confirmedViolations;
        }

        public void incrementConfirmedViolations() {
            ++this.confirmedViolations;
        }

        public void cleanupOldSeverities(long now, long maxAge) {
            long cutoff = now - maxAge;
            for (List<SeverityRecord> records : this.severityHistory.values()) {
                records.removeIf(r -> r.timestamp < cutoff);
            }
        }

        private static final class SeverityRecord {
            private final double severity;
            private final long timestamp;

            private SeverityRecord(double severity, long timestamp) {
                this.severity = severity;
                this.timestamp = timestamp;
            }
        }
    }

    public static enum SuspicionLevel {
        CLEAN,
        LOW,
        MEDIUM,
        HIGH,
        CONFIRMED;

    }

    private static final class EnsembleVote {
        private final boolean shouldFlag;
        private final double confidence;
        private final String detectionMethod;
        private final int votesForCheat;
        private final int totalVotes;

        private EnsembleVote(boolean shouldFlag, double confidence, String detectionMethod, int votesForCheat, int totalVotes) {
            this.shouldFlag = shouldFlag;
            this.confidence = confidence;
            this.detectionMethod = detectionMethod;
            this.votesForCheat = votesForCheat;
            this.totalVotes = totalVotes;
        }
    }

    public static final class AnalysisResult {
        public final boolean shouldFlag;
        public final double confidence;
        public final String detectionMethod;
        public final SuspicionLevel suspicionLevel;
        public final double entropyScore;
        public final double consistencyScore;
        public final double correlationScore;
        public final double anomalyScore;
        public final double fingerprintScore;
        public final String variant;
        public final CrossCheckCorrelation.CorrelationType clusterType;
        public final double punishmentMultiplier;

        AnalysisResult(boolean shouldFlag, double confidence, String detectionMethod, SuspicionLevel suspicionLevel, double entropyScore, double consistencyScore, double correlationScore, double anomalyScore, double fingerprintScore, String variant, CrossCheckCorrelation.CorrelationType clusterType, double punishmentMultiplier) {
            this.shouldFlag = shouldFlag;
            this.confidence = confidence;
            this.detectionMethod = detectionMethod;
            this.suspicionLevel = suspicionLevel;
            this.entropyScore = entropyScore;
            this.consistencyScore = consistencyScore;
            this.correlationScore = correlationScore;
            this.anomalyScore = anomalyScore;
            this.fingerprintScore = fingerprintScore;
            this.variant = variant;
            this.clusterType = clusterType;
            this.punishmentMultiplier = punishmentMultiplier;
        }
    }
}

