/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.core;

import NC.noChance.core.ViolationType;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.entity.Player;

public class FalsePositiveFilter {
    private final Map<UUID, PlayerProfile> profiles = new ConcurrentHashMap<UUID, PlayerProfile>();
    private final Map<ViolationType, CheckStatistics> globalStats = new ConcurrentHashMap<ViolationType, CheckStatistics>();
    private static final int PROFILE_WINDOW = 100;
    private static final double BAYESIAN_PRIOR = 0.05;
    private static final double Z_SCORE_THRESHOLD = 2.5;

    public FalsePositiveFilter() {
        for (ViolationType type : ViolationType.values()) {
            this.globalStats.put(type, new CheckStatistics());
        }
    }

    public FilterResult evaluate(Player player, ViolationType type, double rawSeverity, String reason) {
        UUID playerId = player.getUniqueId();
        PlayerProfile profile = this.profiles.computeIfAbsent(playerId, k -> new PlayerProfile());
        CheckStatistics stats = this.globalStats.get((Object)type);
        stats.record(rawSeverity, true);
        profile.recordCheck(type, true);
        double bayesianConfidence = this.calculateBayesianProbability(profile, type, rawSeverity);
        double zScore = stats.getZScore(rawSeverity);
        double trustMultiplier = this.calculateTrustMultiplier(profile, type);
        double varianceScore = this.calculateVarianceScore(profile, type);
        double temporalScore = this.calculateTemporalScore(profile, type);
        double aggregatedConfidence = this.aggregateConfidence(bayesianConfidence, rawSeverity, trustMultiplier, varianceScore, temporalScore, zScore);
        if (this.shouldFilterOut(profile, type, aggregatedConfidence, zScore, trustMultiplier)) {
            return FilterResult.pass();
        }
        profile.recordViolation(type, rawSeverity, reason);
        return FilterResult.allow(aggregatedConfidence, trustMultiplier);
    }

    private double calculateBayesianProbability(PlayerProfile profile, ViolationType type, double severity) {
        double likelihood;
        double evidence;
        double prior = 0.05;
        if (profile.hasEstablishedBaseline()) {
            double violationRate = profile.getViolationRate(type);
            prior = Math.max(0.01, Math.min(0.5, violationRate));
        }
        if ((evidence = (likelihood = severity) * prior + (1.0 - likelihood) * (1.0 - prior)) == 0.0) {
            return prior;
        }
        return likelihood * prior / evidence;
    }

    private double calculateTrustMultiplier(PlayerProfile profile, ViolationType type) {
        double trustScore = profile.getTrustScore();
        double playTime = Math.min(1.0, (double)profile.getPlayTime() / 3600000.0);
        double checkRatio = profile.totalChecks > 0 ? (double)profile.confirmedLegitActions / (double)profile.totalChecks : 0.5;
        double trustMultiplier = trustScore * 0.5 + playTime * 0.25 + checkRatio * 0.25;
        return Math.max(0.1, Math.min(2.0, trustMultiplier));
    }

    private double calculateVarianceScore(PlayerProfile profile, ViolationType type) {
        if (!profile.hasEstablishedBaseline()) {
            return 1.0;
        }
        double variance = 0.0;
        switch (type) {
            case FASTBREAK: {
                variance = profile.getBreakTimingVariance();
                if (variance < 100.0) {
                    return 0.3;
                }
                if (variance < 500.0) {
                    return 0.6;
                }
                return 1.0;
            }
            case KILLAURA: 
            case KILLAURA_MULTI: 
            case KILLAURA_ANGLE: 
            case KILLAURA_ROTATION: 
            case KILLAURA_PATTERN: {
                variance = profile.getRotationVariance();
                if (variance < 50.0) {
                    return 0.3;
                }
                if (variance < 200.0) {
                    return 0.6;
                }
                return 1.0;
            }
        }
        return 1.0;
    }

    private double calculateTemporalScore(PlayerProfile profile, ViolationType type) {
        double cv;
        Deque<ViolationRecord> history = profile.violationHistory.get((Object)type);
        if (history.size() < 3) {
            return 1.0;
        }
        List<ViolationRecord> recent = new ArrayList<ViolationRecord>(history);
        if (recent.size() > 10) {
            recent = recent.subList(recent.size() - 10, recent.size());
        }
        ArrayList<Long> intervals = new ArrayList<Long>();
        for (int i = 1; i < recent.size(); ++i) {
            intervals.add(recent.get((int)i).timestamp - recent.get((int)(i - 1)).timestamp);
        }
        if (intervals.isEmpty()) {
            return 1.0;
        }
        double mean = intervals.stream().mapToLong(Long::longValue).average().orElse(0.0);
        double variance = intervals.stream().mapToDouble(v -> Math.pow((double)v.longValue() - mean, 2.0)).average().orElse(0.0);
        double d = cv = mean > 0.0 ? Math.sqrt(variance) / mean : 0.0;
        if (cv < 0.15) {
            return 0.4;
        }
        if (cv < 0.3) {
            return 0.7;
        }
        return 1.0;
    }

    private double aggregateConfidence(double bayesian, double rawSeverity, double trust, double variance, double temporal, double zScore) {
        double zScoreWeight = Math.min(1.0, Math.abs(zScore) / 2.5);
        double weightedConfidence = bayesian * 0.3 + rawSeverity * 0.25 + variance * 0.2 + temporal * 0.15 + zScoreWeight * 0.1;
        return Math.max(0.0, Math.min(1.0, weightedConfidence *= trust));
    }

    private boolean shouldFilterOut(PlayerProfile profile, ViolationType type, double confidence, double zScore, double trustMultiplier) {
        if (profile.getTrustScore() > 0.8 && confidence < 0.85) {
            return true;
        }
        if (profile.hasEstablishedBaseline() && Math.abs(zScore) < 2.0) {
            return true;
        }
        if (trustMultiplier > 1.3 && confidence < 0.8) {
            return true;
        }
        if (profile.confirmedLegitActions > 500 && profile.totalViolations < 5 && confidence < 0.9) {
            return true;
        }
        if (profile.confirmedLegitActions > 200 && confidence < 0.85) {
            return true;
        }
        Deque<ViolationRecord> history = profile.violationHistory.get((Object)type);
        if (history.size() >= 3) {
            long now = System.currentTimeMillis();
            long recentCount = history.stream().filter(r -> now - r.timestamp < 5000L).count();
            if (recentCount >= 3L && confidence < 0.88) {
                return true;
            }
        }
        return confidence < 0.7;
    }

    public void recordLegitAction(Player player, ViolationType type) {
        UUID playerId = player.getUniqueId();
        PlayerProfile profile = this.profiles.computeIfAbsent(playerId, k -> new PlayerProfile());
        profile.recordCheck(type, false);
        profile.adjustTrustScore(0.001);
        CheckStatistics stats = this.globalStats.get((Object)type);
        stats.record(0.0, false);
    }

    public void recordBreakTiming(Player player, long timing) {
        UUID playerId = player.getUniqueId();
        PlayerProfile profile = this.profiles.computeIfAbsent(playerId, k -> new PlayerProfile());
        profile.recordBreakTiming(timing);
    }

    public void recordRotation(Player player, double speed) {
        UUID playerId = player.getUniqueId();
        PlayerProfile profile = this.profiles.computeIfAbsent(playerId, k -> new PlayerProfile());
        profile.recordRotation(speed);
    }

    public void recordMovement(Player player, double speed) {
        UUID playerId = player.getUniqueId();
        PlayerProfile profile = this.profiles.computeIfAbsent(playerId, k -> new PlayerProfile());
        profile.recordMovement(speed);
    }

    public double getPlayerTrustScore(UUID playerId) {
        PlayerProfile profile = this.profiles.get(playerId);
        return profile != null ? profile.getTrustScore() : 0.5;
    }

    public void cleanup(UUID playerId) {
        this.profiles.remove(playerId);
    }

    public static class CheckStatistics {
        private final Deque<Double> severities = new ArrayDeque<Double>(1000);
        private double meanSeverity = 0.0;
        private double stdDevSeverity = 0.0;
        private int totalChecks = 0;
        private int totalViolations = 0;

        public void record(double severity, boolean violated) {
            ++this.totalChecks;
            if (violated) {
                ++this.totalViolations;
                this.severities.addLast(severity);
                if (this.severities.size() > 1000) {
                    this.severities.pollFirst();
                }
                this.updateStatistics();
            }
        }

        private void updateStatistics() {
            if (this.severities.isEmpty()) {
                return;
            }
            this.meanSeverity = this.severities.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
            this.stdDevSeverity = Math.sqrt(this.severities.stream().mapToDouble(v -> Math.pow(v - this.meanSeverity, 2.0)).average().orElse(0.0));
        }

        public double getZScore(double severity) {
            if (this.stdDevSeverity == 0.0) {
                return 0.0;
            }
            return (severity - this.meanSeverity) / this.stdDevSeverity;
        }

        public double getViolationRate() {
            if (this.totalChecks == 0) {
                return 0.0;
            }
            return (double)this.totalViolations / (double)this.totalChecks;
        }
    }

    public static class PlayerProfile {
        private final Map<ViolationType, Deque<ViolationRecord>> violationHistory = new HashMap<ViolationType, Deque<ViolationRecord>>();
        private final Map<ViolationType, Double> baselineScores = new HashMap<ViolationType, Double>();
        private final Deque<Long> breakTimings = new ArrayDeque<Long>(100);
        private final Deque<Double> rotationSpeeds = new ArrayDeque<Double>(100);
        private final Deque<Double> movementSpeeds = new ArrayDeque<Double>(100);
        private double trustScore = 0.5;
        private long firstSeen = System.currentTimeMillis();
        private int totalChecks = 0;
        private int totalViolations = 0;
        private int confirmedLegitActions = 0;

        public PlayerProfile() {
            for (ViolationType type : ViolationType.values()) {
                this.violationHistory.put(type, new ArrayDeque(50));
                this.baselineScores.put(type, 0.0);
            }
        }

        public void recordCheck(ViolationType type, boolean violated) {
            ++this.totalChecks;
            if (!violated) {
                ++this.confirmedLegitActions;
                this.adjustTrustScore(8.0E-4);
            }
        }

        public void recordViolation(ViolationType type, double severity, String reason) {
            ++this.totalViolations;
            Deque<ViolationRecord> history = this.violationHistory.get((Object)type);
            history.addLast(new ViolationRecord(severity, reason, System.currentTimeMillis()));
            if (history.size() > 50) {
                history.pollFirst();
            }
            long now = System.currentTimeMillis();
            long recentViolations = history.stream().filter(v -> now - v.timestamp < 5000L).count();
            double trustPenalty = -0.015 * severity;
            if (recentViolations >= 3L) {
                trustPenalty *= 1.5;
            }
            this.adjustTrustScore(trustPenalty);
        }

        public void recordBreakTiming(long timing) {
            this.breakTimings.addLast(timing);
            if (this.breakTimings.size() > 100) {
                this.breakTimings.pollFirst();
            }
        }

        public void recordRotation(double speed) {
            this.rotationSpeeds.addLast(speed);
            if (this.rotationSpeeds.size() > 100) {
                this.rotationSpeeds.pollFirst();
            }
        }

        public void recordMovement(double speed) {
            this.movementSpeeds.addLast(speed);
            if (this.movementSpeeds.size() > 100) {
                this.movementSpeeds.pollFirst();
            }
        }

        private void adjustTrustScore(double delta) {
            this.trustScore = Math.max(0.0, Math.min(1.0, this.trustScore + delta));
        }

        public double getTrustScore() {
            return this.trustScore;
        }

        public double getViolationRate(ViolationType type) {
            Deque<ViolationRecord> history = this.violationHistory.get((Object)type);
            if (history.isEmpty() || this.totalChecks == 0) {
                return 0.0;
            }
            return (double)history.size() / (double)this.totalChecks;
        }

        public boolean hasEstablishedBaseline() {
            return this.totalChecks >= 50 && System.currentTimeMillis() - this.firstSeen > 60000L;
        }

        public double getBreakTimingVariance() {
            if (this.breakTimings.size() < 10) {
                return Double.MAX_VALUE;
            }
            double mean = this.breakTimings.stream().mapToLong(Long::longValue).average().orElse(0.0);
            double variance = this.breakTimings.stream().mapToDouble(v -> Math.pow((double)v.longValue() - mean, 2.0)).average().orElse(0.0);
            return variance;
        }

        public double getRotationVariance() {
            if (this.rotationSpeeds.size() < 10) {
                return Double.MAX_VALUE;
            }
            double mean = this.rotationSpeeds.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
            double variance = this.rotationSpeeds.stream().mapToDouble(v -> Math.pow(v - mean, 2.0)).average().orElse(0.0);
            return variance;
        }

        public long getPlayTime() {
            return System.currentTimeMillis() - this.firstSeen;
        }
    }

    public static class FilterResult {
        public final boolean shouldFilter;
        public final double adjustedConfidence;
        public final String filterReason;
        public final double trustMultiplier;

        public FilterResult(boolean shouldFilter, double adjustedConfidence, String filterReason, double trustMultiplier) {
            this.shouldFilter = shouldFilter;
            this.adjustedConfidence = adjustedConfidence;
            this.filterReason = filterReason;
            this.trustMultiplier = trustMultiplier;
        }

        public static FilterResult pass() {
            return new FilterResult(true, 0.0, "Filtered", 1.0);
        }

        public static FilterResult allow(double confidence, double trustMultiplier) {
            return new FilterResult(false, confidence, "Allowed", trustMultiplier);
        }
    }

    public static class ViolationRecord {
        public final double severity;
        public final String reason;
        public final long timestamp;

        public ViolationRecord(double severity, String reason, long timestamp) {
            this.severity = severity;
            this.reason = reason;
            this.timestamp = timestamp;
        }
    }
}

