/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.database;

import NC.noChance.core.ACConfig;
import NC.noChance.core.ViolationType;
import NC.noChance.hikari.HikariConfig;
import NC.noChance.hikari.HikariDataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.bukkit.plugin.Plugin;

public class DatabaseManager {
    private final Plugin plugin;
    private final ACConfig config;
    private HikariDataSource dataSource;
    private final DatabaseType type;
    private final String tablePrefix;
    private final Queue<BatchOperation> violationBatchQueue;
    private final Queue<BatchOperation> punishmentBatchQueue;
    private final Queue<BatchOperation> playerDataBatchQueue;
    private final ScheduledExecutorService batchExecutor;

    public DatabaseManager(Plugin plugin, ACConfig config, DatabaseType type, String host, int port, String database, String username, String password, String tablePrefix) {
        this.plugin = plugin;
        this.config = config;
        this.type = type;
        this.tablePrefix = tablePrefix;
        this.violationBatchQueue = new ConcurrentLinkedQueue<BatchOperation>();
        this.punishmentBatchQueue = new ConcurrentLinkedQueue<BatchOperation>();
        this.playerDataBatchQueue = new ConcurrentLinkedQueue<BatchOperation>();
        this.batchExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "NoChance-BatchDB");
            t.setDaemon(true);
            return t;
        });
        this.initializeDataSource(host, port, database, username, password);
        this.createTables();
        this.startBatchProcessor();
    }

    private void initializeDataSource(String host, int port, String database, String username, String password) {
        HikariConfig hikariConfig = new HikariConfig();
        if (this.type == DatabaseType.SQLITE) {
            hikariConfig.setJdbcUrl("jdbc:sqlite:" + String.valueOf(this.plugin.getDataFolder()) + "/data.db");
            hikariConfig.setDriverClassName("org.sqlite.JDBC");
        } else {
            hikariConfig.setJdbcUrl("jdbc:mysql://" + host + ":" + port + "/" + database + "?useSSL=false&autoReconnect=true&cachePrepStmts=true&prepStmtCacheSize=250&prepStmtCacheSqlLimit=2048&useServerPrepStmts=true");
            hikariConfig.setUsername(username);
            hikariConfig.setPassword(password);
            hikariConfig.setDriverClassName("com.mysql.cj.jdbc.Driver");
        }
        hikariConfig.setMaximumPoolSize(this.config.getDatabaseMaxPoolSize());
        hikariConfig.setMinimumIdle(this.config.getDatabaseMinIdle());
        hikariConfig.setConnectionTimeout(this.config.getDatabaseConnectionTimeout());
        hikariConfig.setIdleTimeout(this.config.getDatabaseIdleTimeout());
        hikariConfig.setMaxLifetime(this.config.getDatabaseMaxLifetime());
        this.dataSource = new HikariDataSource(hikariConfig);
    }

    private void createTables() {
        String autoIncrement = this.type == DatabaseType.SQLITE ? "AUTOINCREMENT" : "AUTO_INCREMENT";
        String violationsTable = "CREATE TABLE IF NOT EXISTS " + this.tablePrefix + "violations (id INTEGER PRIMARY KEY " + autoIncrement + ", uuid VARCHAR(36) NOT NULL, player_name VARCHAR(16) NOT NULL, violation_type VARCHAR(32) NOT NULL, severity DOUBLE NOT NULL, details TEXT, confidence_level VARCHAR(16), detection_method VARCHAR(32), timestamp BIGINT NOT NULL)";
        String punishmentsTable = "CREATE TABLE IF NOT EXISTS " + this.tablePrefix + "punishments (id INTEGER PRIMARY KEY " + autoIncrement + ", uuid VARCHAR(36) NOT NULL, player_name VARCHAR(16) NOT NULL, punishment_type VARCHAR(16) NOT NULL, reason TEXT, duration BIGINT, issued_by VARCHAR(32), timestamp BIGINT NOT NULL, active BOOLEAN DEFAULT 1)";
        String playerDataTable = "CREATE TABLE IF NOT EXISTS " + this.tablePrefix + "player_data (uuid VARCHAR(36) PRIMARY KEY, player_name VARCHAR(16) NOT NULL, total_violations INTEGER DEFAULT 0, total_checks INTEGER DEFAULT 0, trust_score DOUBLE DEFAULT 1.0, skill_level VARCHAR(16), first_seen BIGINT NOT NULL, last_seen BIGINT NOT NULL, banned BOOLEAN DEFAULT 0)";
        this.executeUpdate(violationsTable, new String[0]);
        this.executeUpdate(punishmentsTable, new String[0]);
        this.executeUpdate(playerDataTable, new String[0]);
        this.createIndexes();
    }

    private void createIndexes() {
        this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_violations_uuid ON " + this.tablePrefix + "violations(uuid)", new String[0]);
        this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_violations_type ON " + this.tablePrefix + "violations(violation_type)", new String[0]);
        this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_violations_timestamp ON " + this.tablePrefix + "violations(timestamp)", new String[0]);
        this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_punishments_uuid ON " + this.tablePrefix + "punishments(uuid)", new String[0]);
        this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_punishments_active ON " + this.tablePrefix + "punishments(active)", new String[0]);
    }

    private void startBatchProcessor() {
        long interval = this.config.getDatabaseBatchInterval();
        this.batchExecutor.scheduleAtFixedRate(this::processBatches, interval, interval, TimeUnit.MILLISECONDS);
    }

    private void processBatches() {
        this.processViolationBatch();
        this.processPunishmentBatch();
        this.processPlayerDataBatch();
    }

    private void processViolationBatch() {
        if (this.violationBatchQueue.isEmpty()) {
            return;
        }
        ArrayList<BatchOperation> batch = new ArrayList<BatchOperation>();
        int batchSize = this.config.getDatabaseBatchSize();
        for (int i = 0; i < batchSize && !this.violationBatchQueue.isEmpty(); ++i) {
            batch.add(this.violationBatchQueue.poll());
        }
        if (batch.isEmpty()) {
            return;
        }
        String sql = "INSERT INTO " + this.tablePrefix + "violations (uuid, player_name, violation_type, severity, details, confidence_level, detection_method, timestamp) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        try (Connection conn = this.dataSource.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql);){
            for (BatchOperation op : batch) {
                Object[] params = op.parameters;
                stmt.setString(1, (String)params[0]);
                stmt.setString(2, (String)params[1]);
                stmt.setString(3, (String)params[2]);
                stmt.setDouble(4, (Double)params[3]);
                stmt.setString(5, (String)params[4]);
                stmt.setString(6, (String)params[5]);
                stmt.setString(7, (String)params[6]);
                stmt.setLong(8, (Long)params[7]);
                stmt.addBatch();
            }
            stmt.executeBatch();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to process violation batch: " + e.getMessage());
        }
    }

    private void processPunishmentBatch() {
        if (this.punishmentBatchQueue.isEmpty()) {
            return;
        }
        ArrayList<BatchOperation> batch = new ArrayList<BatchOperation>();
        int batchSize = this.config.getDatabaseBatchSize();
        for (int i = 0; i < batchSize && !this.punishmentBatchQueue.isEmpty(); ++i) {
            batch.add(this.punishmentBatchQueue.poll());
        }
        if (batch.isEmpty()) {
            return;
        }
        String sql = "INSERT INTO " + this.tablePrefix + "punishments (uuid, player_name, punishment_type, reason, duration, issued_by, timestamp, active) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        try (Connection conn = this.dataSource.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql);){
            for (BatchOperation op : batch) {
                Object[] params = op.parameters;
                stmt.setString(1, (String)params[0]);
                stmt.setString(2, (String)params[1]);
                stmt.setString(3, (String)params[2]);
                stmt.setString(4, (String)params[3]);
                stmt.setLong(5, (Long)params[4]);
                stmt.setString(6, (String)params[5]);
                stmt.setLong(7, (Long)params[6]);
                stmt.setBoolean(8, (Boolean)params[7]);
                stmt.addBatch();
            }
            stmt.executeBatch();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to process punishment batch: " + e.getMessage());
        }
    }

    private void processPlayerDataBatch() {
        if (this.playerDataBatchQueue.isEmpty()) {
            return;
        }
        ArrayList<BatchOperation> batch = new ArrayList<BatchOperation>();
        int batchSize = this.config.getDatabaseBatchSize();
        for (int i = 0; i < batchSize && !this.playerDataBatchQueue.isEmpty(); ++i) {
            batch.add(this.playerDataBatchQueue.poll());
        }
        if (batch.isEmpty()) {
            return;
        }
        try (Connection conn = this.dataSource.getConnection();){
            for (BatchOperation op : batch) {
                Object[] params = op.parameters;
                String sql = this.type == DatabaseType.MYSQL ? "INSERT INTO " + this.tablePrefix + "player_data (uuid, player_name, total_violations, total_checks, trust_score, skill_level, first_seen, last_seen, banned) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE player_name = ?, total_violations = ?, total_checks = ?, trust_score = ?, skill_level = ?, last_seen = ?, banned = ?" : "INSERT OR REPLACE INTO " + this.tablePrefix + "player_data (uuid, player_name, total_violations, total_checks, trust_score, skill_level, first_seen, last_seen, banned) VALUES (?, ?, ?, ?, ?, ?, COALESCE((SELECT first_seen FROM " + this.tablePrefix + "player_data WHERE uuid = ?), ?), ?, ?)";
                PreparedStatement stmt = conn.prepareStatement(sql);
                try {
                    if (this.type == DatabaseType.MYSQL) {
                        stmt.setString(1, (String)params[0]);
                        stmt.setString(2, (String)params[1]);
                        stmt.setInt(3, (Integer)params[2]);
                        stmt.setInt(4, (Integer)params[3]);
                        stmt.setDouble(5, (Double)params[4]);
                        stmt.setString(6, (String)params[5]);
                        stmt.setLong(7, (Long)params[6]);
                        stmt.setLong(8, (Long)params[6]);
                        stmt.setBoolean(9, (Boolean)params[7]);
                        stmt.setString(10, (String)params[1]);
                        stmt.setInt(11, (Integer)params[2]);
                        stmt.setInt(12, (Integer)params[3]);
                        stmt.setDouble(13, (Double)params[4]);
                        stmt.setString(14, (String)params[5]);
                        stmt.setLong(15, (Long)params[6]);
                        stmt.setBoolean(16, (Boolean)params[7]);
                    } else {
                        stmt.setString(1, (String)params[0]);
                        stmt.setString(2, (String)params[1]);
                        stmt.setInt(3, (Integer)params[2]);
                        stmt.setInt(4, (Integer)params[3]);
                        stmt.setDouble(5, (Double)params[4]);
                        stmt.setString(6, (String)params[5]);
                        stmt.setString(7, (String)params[0]);
                        stmt.setLong(8, (Long)params[6]);
                        stmt.setLong(9, (Long)params[6]);
                        stmt.setBoolean(10, (Boolean)params[7]);
                    }
                    stmt.executeUpdate();
                }
                finally {
                    if (stmt == null) continue;
                    stmt.close();
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to process player data batch: " + e.getMessage());
        }
    }

    public CompletableFuture<Void> logViolation(UUID uuid, String playerName, ViolationType type, double severity, String details, String confidenceLevel, String detectionMethod) {
        this.violationBatchQueue.offer(new BatchOperation(new Object[]{uuid.toString(), playerName, type.name(), severity, details, confidenceLevel, detectionMethod, System.currentTimeMillis()}));
        return CompletableFuture.completedFuture(null);
    }

    public CompletableFuture<Void> logPunishment(UUID uuid, String playerName, PunishmentType type, String reason, long duration, String issuedBy) {
        this.punishmentBatchQueue.offer(new BatchOperation(new Object[]{uuid.toString(), playerName, type.name(), reason, duration, issuedBy, System.currentTimeMillis(), true}));
        return CompletableFuture.completedFuture(null);
    }

    public CompletableFuture<Integer> getViolationCount(UUID uuid, ViolationType type, long timeWindow) {
        return CompletableFuture.supplyAsync(() -> {
            String sql = "SELECT COUNT(*) FROM " + this.tablePrefix + "violations WHERE uuid = ? AND violation_type = ? AND timestamp > ?";
            try (Connection conn = this.dataSource.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql);){
                stmt.setString(1, uuid.toString());
                stmt.setString(2, type.name());
                stmt.setLong(3, System.currentTimeMillis() - timeWindow);
                try (ResultSet rs = stmt.executeQuery();){
                    if (!rs.next()) return 0;
                    Integer n = rs.getInt(1);
                    return n;
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Failed to get violation count: " + e.getMessage());
            }
            return 0;
        });
    }

    public CompletableFuture<List<ViolationRecord>> getViolations(UUID uuid, long timeWindow) {
        return CompletableFuture.supplyAsync(() -> {
            ArrayList<ViolationRecord> violations = new ArrayList<ViolationRecord>();
            String sql = "SELECT * FROM " + this.tablePrefix + "violations WHERE uuid = ? AND timestamp > ? ORDER BY timestamp DESC LIMIT 100";
            try (Connection conn = this.dataSource.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql);){
                stmt.setString(1, uuid.toString());
                stmt.setLong(2, System.currentTimeMillis() - timeWindow);
                try (ResultSet rs = stmt.executeQuery();){
                    while (rs.next()) {
                        violations.add(new ViolationRecord(rs.getString("player_name"), ViolationType.valueOf(rs.getString("violation_type")), rs.getDouble("severity"), rs.getString("details"), rs.getString("confidence_level"), rs.getString("detection_method"), rs.getLong("timestamp")));
                    }
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Failed to get violations: " + e.getMessage());
            }
            return violations;
        });
    }

    public CompletableFuture<ActivePunishment> getActivePunishment(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> {
            String sql = "SELECT * FROM " + this.tablePrefix + "punishments WHERE uuid = ? AND active = 1 ORDER BY timestamp DESC LIMIT 1";
            try (Connection conn = this.dataSource.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql);){
                stmt.setString(1, uuid.toString());
                try (ResultSet rs = stmt.executeQuery();){
                    if (!rs.next()) return null;
                    long timestamp = rs.getLong("timestamp");
                    long duration = rs.getLong("duration");
                    if (duration > 0L && System.currentTimeMillis() > timestamp + duration) {
                        CompletableFuture.runAsync(() -> this.deactivatePunishment(uuid));
                        ActivePunishment activePunishment = null;
                        return activePunishment;
                    }
                    ActivePunishment activePunishment = new ActivePunishment(PunishmentType.valueOf(rs.getString("punishment_type")), rs.getString("reason"), timestamp, duration);
                    return activePunishment;
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Failed to get active punishment: " + e.getMessage());
            }
            return null;
        });
    }

    private void deactivatePunishment(UUID uuid) {
        try (Connection conn = this.dataSource.getConnection();
             PreparedStatement stmt = conn.prepareStatement("UPDATE " + this.tablePrefix + "punishments SET active = 0 WHERE uuid = ? AND active = 1");){
            stmt.setString(1, uuid.toString());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to deactivate punishment: " + e.getMessage());
        }
    }

    public CompletableFuture<Void> updatePlayerData(UUID uuid, String playerName, int totalViolations, int totalChecks, double trustScore, String skillLevel, boolean banned) {
        long now = System.currentTimeMillis();
        this.playerDataBatchQueue.offer(new BatchOperation(new Object[]{uuid.toString(), playerName, totalViolations, totalChecks, trustScore, skillLevel, now, banned}));
        return CompletableFuture.completedFuture(null);
    }

    public CompletableFuture<PlayerHistory> getPlayerHistory(UUID uuid, long timeWindow) {
        return CompletableFuture.supplyAsync(() -> {
            String violationSql = "SELECT COUNT(*) FROM " + this.tablePrefix + "violations WHERE uuid = ? AND timestamp > ?";
            String kickSql = "SELECT MAX(timestamp) FROM " + this.tablePrefix + "punishments WHERE uuid = ? AND punishment_type = 'KICK' AND timestamp > ?";
            int violationCount = 0;
            long lastKickTime = 0L;
            try (Connection conn = this.dataSource.getConnection();){
                ResultSet rs;
                try (PreparedStatement stmt = conn.prepareStatement(violationSql);){
                    stmt.setString(1, uuid.toString());
                    stmt.setLong(2, System.currentTimeMillis() - timeWindow);
                    rs = stmt.executeQuery();
                    try {
                        if (rs.next()) {
                            violationCount = rs.getInt(1);
                        }
                    }
                    finally {
                        if (rs != null) {
                            rs.close();
                        }
                    }
                }
                stmt = conn.prepareStatement(kickSql);
                try {
                    stmt.setString(1, uuid.toString());
                    stmt.setLong(2, System.currentTimeMillis() - 86400000L);
                    rs = stmt.executeQuery();
                    try {
                        if (rs.next()) {
                            lastKickTime = rs.getLong(1);
                        }
                    }
                    finally {
                        if (rs != null) {
                            rs.close();
                        }
                    }
                }
                finally {
                    if (stmt != null) {
                        stmt.close();
                    }
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Failed to get player history: " + e.getMessage());
            }
            return new PlayerHistory(violationCount, lastKickTime);
        });
    }

    private void executeUpdate(String sql, String ... params) {
        try (Connection conn = this.dataSource.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql);){
            for (int i = 0; i < params.length; ++i) {
                stmt.setString(i + 1, params[i]);
            }
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to execute update: " + e.getMessage());
        }
    }

    public void close() {
        if (this.batchExecutor != null && !this.batchExecutor.isShutdown()) {
            this.processBatches();
            this.batchExecutor.shutdown();
            try {
                if (!this.batchExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.batchExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.batchExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        if (this.dataSource != null && !this.dataSource.isClosed()) {
            this.dataSource.close();
        }
    }

    public static enum DatabaseType {
        MYSQL,
        SQLITE;

    }

    private static final class BatchOperation {
        private final Object[] parameters;

        private BatchOperation(Object[] parameters) {
            this.parameters = parameters;
        }
    }

    public static enum PunishmentType {
        WARN,
        KICK,
        TEMPBAN,
        BAN;

    }

    public static final class PlayerHistory {
        public final int violationCount;
        public final long lastKickTime;

        PlayerHistory(int violationCount, long lastKickTime) {
            this.violationCount = violationCount;
            this.lastKickTime = lastKickTime;
        }

        public boolean wasRecentlyKicked(long threshold) {
            return this.lastKickTime > 0L && System.currentTimeMillis() - this.lastKickTime < threshold;
        }
    }

    public static final class ActivePunishment {
        public final PunishmentType type;
        public final String reason;
        public final long timestamp;
        public final long duration;

        ActivePunishment(PunishmentType type, String reason, long timestamp, long duration) {
            this.type = type;
            this.reason = reason;
            this.timestamp = timestamp;
            this.duration = duration;
        }

        public boolean isExpired() {
            return this.duration > 0L && System.currentTimeMillis() > this.timestamp + this.duration;
        }

        public long getTimeRemaining() {
            if (this.duration == 0L) {
                return -1L;
            }
            return Math.max(0L, this.timestamp + this.duration - System.currentTimeMillis());
        }
    }

    public static final class ViolationRecord {
        public final String playerName;
        public final ViolationType type;
        public final double severity;
        public final String details;
        public final String confidenceLevel;
        public final String detectionMethod;
        public final long timestamp;

        ViolationRecord(String playerName, ViolationType type, double severity, String details, String confidenceLevel, String detectionMethod, long timestamp) {
            this.playerName = playerName;
            this.type = type;
            this.severity = severity;
            this.details = details;
            this.confidenceLevel = confidenceLevel;
            this.detectionMethod = detectionMethod;
            this.timestamp = timestamp;
        }
    }
}

