/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.detection.damage;

import NC.noChance.core.ACConfig;
import NC.noChance.core.CheckResult;
import NC.noChance.core.EnhancementTracker;
import NC.noChance.core.FalsePositiveFilter;
import NC.noChance.core.LayerFiltering;
import NC.noChance.core.PlayerData;
import NC.noChance.core.ViolationType;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.block.Block;
import org.bukkit.enchantments.Enchantment;
import org.bukkit.entity.Player;
import org.bukkit.event.entity.EntityDamageEvent;
import org.bukkit.inventory.ItemStack;
import org.bukkit.potion.PotionEffectType;

public class NoFallCheck {
    private final ACConfig config;
    private final Map<UUID, PlayerData> playerDataMap;
    private final LayerFiltering filtering;
    private final FalsePositiveFilter falsePositiveFilter;
    private final Map<UUID, NoFallTracker> trackers;

    public NoFallCheck(ACConfig config, Map<UUID, PlayerData> playerDataMap, LayerFiltering filtering) {
        this.config = config;
        this.playerDataMap = playerDataMap;
        this.filtering = filtering;
        this.falsePositiveFilter = new FalsePositiveFilter();
        this.trackers = new ConcurrentHashMap<UUID, NoFallTracker>();
    }

    public void updateFallDistance(Player player) {
        PlayerData data = this.playerDataMap.get(player.getUniqueId());
        if (data == null) {
            return;
        }
        UUID uuid = player.getUniqueId();
        NoFallTracker tracker = this.trackers.computeIfAbsent(uuid, k -> new NoFallTracker());
        if (player.isGliding() || player.isSwimming() || player.isRiptiding() || player.isFlying() || player.isClimbing()) {
            data.resetFallDistance();
            tracker.reset();
            return;
        }
        if (this.isInSafeLandingBlock(player)) {
            data.resetFallDistance();
            tracker.reset();
            return;
        }
        if (player.isOnGround()) {
            if (!data.wasOnGround()) {
                double trustScore;
                data.setLastGroundTime(System.currentTimeMillis());
                tracker.landed();
                double serverFallDistance = data.getTotalFallDistance();
                double clientFallDistance = player.getFallDistance();
                if (serverFallDistance > 3.0 && clientFallDistance < serverFallDistance * 0.5 && (trustScore = this.falsePositiveFilter.getPlayerTrustScore(uuid)) < 0.7) {
                    tracker.recordSuspiciousFall(serverFallDistance, clientFallDistance);
                }
            }
            data.setWasOnGround(true);
        } else {
            data.setWasOnGround(false);
            if (data.getLastLocation() != null) {
                double deltaY = data.getLastLocation().getY() - player.getLocation().getY();
                if (deltaY > 0.0) {
                    data.addFallDistance(deltaY);
                    tracker.recordFall(deltaY);
                } else if (deltaY < -0.1 && !player.hasPotionEffect(PotionEffectType.LEVITATION)) {
                    tracker.recordAscent();
                }
            }
        }
    }

    public CheckResult checkOnLanding(Player player, EntityDamageEvent event) {
        if (!this.config.isCheckEnabled("nofall")) {
            return CheckResult.passed();
        }
        PlayerData data = this.playerDataMap.get(player.getUniqueId());
        if (data == null) {
            return CheckResult.passed();
        }
        if (data.isInGracePeriod(this.config.getGracePeriod())) {
            return CheckResult.passed();
        }
        UUID uuid = player.getUniqueId();
        NoFallTracker tracker = this.trackers.computeIfAbsent(uuid, k -> new NoFallTracker());
        EnhancementTracker.MovementEnhancements enhancements = EnhancementTracker.calculateMovementEnhancements(player);
        if (player.hasPotionEffect(PotionEffectType.SLOW_FALLING)) {
            this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
            return CheckResult.passed();
        }
        if (player.hasPotionEffect(PotionEffectType.LEVITATION)) {
            this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
            return CheckResult.passed();
        }
        if (enhancements.hasLegitFlight || player.isGliding() || player.isRiptiding()) {
            this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
            return CheckResult.passed();
        }
        if (this.isInSafeLandingBlock(player)) {
            this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
            return CheckResult.passed();
        }
        if (tracker.getTimeSinceLanded() < 500L) {
            this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
            return CheckResult.passed();
        }
        if (tracker.getAscentCount() >= 3) {
            CheckResult prelimResult = CheckResult.failed(ViolationType.NOFALL, 0.9, String.format("Multiple ascents during fall (count: %d)", tracker.getAscentCount()));
            if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.NOFALL, prelimResult)) {
                return CheckResult.passed();
            }
            return prelimResult;
        }
        double fallDistance = Math.max((double)player.getFallDistance(), data.getTotalFallDistance());
        if (fallDistance < 3.5) {
            this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
            data.resetFallDistance();
            return CheckResult.passed();
        }
        double expectedDamage = this.calculateExpectedFallDamage(fallDistance);
        expectedDamage = this.applyDamageReduction(player, expectedDamage);
        expectedDamage = this.applyArmorReduction(player, expectedDamage);
        double actualDamage = 0.0;
        if (event != null && event.getCause() == EntityDamageEvent.DamageCause.FALL) {
            actualDamage = event.getDamage();
        }
        data.resetFallDistance();
        tracker.reset();
        if (expectedDamage < 1.0) {
            this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
            return CheckResult.passed();
        }
        double trustScore = this.falsePositiveFilter.getPlayerTrustScore(uuid);
        double tolerance = 0.5;
        if (trustScore > 0.75) {
            tolerance = 0.3;
        } else if (trustScore > 0.6) {
            tolerance = 0.4;
        }
        if (expectedDamage > 2.0 && actualDamage == 0.0) {
            int requiredViolations;
            tracker.recordViolation();
            int n = requiredViolations = trustScore > 0.75 ? 4 : 3;
            if (tracker.getViolationCount() < requiredViolations) {
                this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
                return CheckResult.passed();
            }
            if (tracker.getViolationCount() >= requiredViolations && fallDistance < 5.0) {
                tracker.decay();
                this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
                return CheckResult.passed();
            }
            double severity = Math.min(1.0, expectedDamage / 15.0);
            CheckResult prelimResult = CheckResult.failed(ViolationType.NOFALL, severity, String.format("No damage from %.1f block fall (expected: %.1f damage, violations: %d, trust: %.2f)", fallDistance, expectedDamage, tracker.getViolationCount(), trustScore));
            if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.NOFALL, prelimResult)) {
                tracker.decay();
                return CheckResult.passed();
            }
            return prelimResult;
        }
        if (expectedDamage > 3.0 && actualDamage < expectedDamage * tolerance) {
            int requiredViolations;
            tracker.recordViolation();
            int n = requiredViolations = trustScore > 0.75 ? 5 : 4;
            if (tracker.getViolationCount() < requiredViolations) {
                this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
                return CheckResult.passed();
            }
            double damageReduction = 1.0 - actualDamage / expectedDamage;
            if (damageReduction < 0.3) {
                tracker.decay();
                this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
                return CheckResult.passed();
            }
            double severity = Math.min(1.0, damageReduction * 1.2);
            CheckResult prelimResult = CheckResult.failed(ViolationType.NOFALL, severity, String.format("Reduced damage from %.1f block fall (expected: %.1f, actual: %.1f, reduction: %.0f%%, violations: %d, trust: %.2f)", fallDistance, expectedDamage, actualDamage, damageReduction * 100.0, tracker.getViolationCount(), trustScore));
            if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.NOFALL, prelimResult)) {
                tracker.decay();
                return CheckResult.passed();
            }
            return prelimResult;
        }
        this.falsePositiveFilter.recordLegitAction(player, ViolationType.NOFALL);
        tracker.decay();
        return CheckResult.passed();
    }

    private double calculateExpectedFallDamage(double fallDistance) {
        double damage = Math.max(0.0, fallDistance - 3.0);
        return damage;
    }

    private double applyDamageReduction(Player player, double damage) {
        Location loc = player.getLocation();
        Block below = loc.clone().subtract(0.0, 1.0, 0.0).getBlock();
        Material belowType = below.getType();
        if (belowType == Material.SLIME_BLOCK) {
            return 0.0;
        }
        if (belowType == Material.HAY_BLOCK) {
            return damage * 0.2;
        }
        if (belowType == Material.HONEY_BLOCK) {
            return damage * 0.2;
        }
        if (belowType == Material.POINTED_DRIPSTONE) {
            return damage * 1.5;
        }
        if (belowType == Material.POWDER_SNOW) {
            return 0.0;
        }
        if (belowType == Material.SWEET_BERRY_BUSH) {
            return damage * 0.5;
        }
        if (belowType.name().contains("BED")) {
            return damage * 0.5;
        }
        if (belowType.name().contains("SCAFFOLDING")) {
            return 0.0;
        }
        return damage;
    }

    private double applyArmorReduction(Player player, double damage) {
        double reduction = 1.0;
        int totalFeatherFalling = 0;
        int totalProtection = 0;
        for (ItemStack armor : player.getInventory().getArmorContents()) {
            if (armor == null || armor.getType() == Material.AIR) continue;
            int ffLevel = armor.getEnchantmentLevel(Enchantment.getByName((String)"PROTECTION_FALL"));
            int protLevel = armor.getEnchantmentLevel(Enchantment.getByName((String)"PROTECTION_ENVIRONMENTAL"));
            totalFeatherFalling += ffLevel;
            totalProtection += protLevel;
        }
        if (totalFeatherFalling > 0) {
            reduction -= Math.min(0.8, (double)totalFeatherFalling * 0.12);
        }
        if (totalProtection > 0) {
            reduction -= Math.min(0.4, (double)totalProtection * 0.04);
        }
        return damage * Math.max(0.0, reduction);
    }

    private boolean isInSafeLandingBlock(Player player) {
        Location loc = player.getLocation();
        Material type = loc.getBlock().getType();
        Material below = loc.clone().subtract(0.0, 1.0, 0.0).getBlock().getType();
        if (type == Material.WATER || type == Material.LAVA) {
            return true;
        }
        if (type == Material.COBWEB) {
            return true;
        }
        if (type == Material.POWDER_SNOW) {
            return true;
        }
        if (type.name().contains("VINE")) {
            return true;
        }
        if (type.name().contains("LADDER")) {
            return true;
        }
        if (type.name().contains("SCAFFOLDING")) {
            return true;
        }
        if (below == Material.SLIME_BLOCK) {
            return true;
        }
        if (below == Material.HONEY_BLOCK) {
            return true;
        }
        return below.name().contains("BED");
    }

    public void cleanup(UUID playerId) {
        this.trackers.remove(playerId);
        this.falsePositiveFilter.cleanup(playerId);
    }

    private static class NoFallTracker {
        private int violations = 0;
        private long lastViolation = 0L;
        private long lastLanding = 0L;
        private double totalFallTracked = 0.0;
        private int suspiciousFalls = 0;
        private int ascentCount = 0;
        private long lastAscent = 0L;

        private NoFallTracker() {
        }

        void recordFall(double distance) {
            this.totalFallTracked += distance;
        }

        void recordViolation() {
            ++this.violations;
            this.lastViolation = System.currentTimeMillis();
        }

        void recordSuspiciousFall(double serverDist, double clientDist) {
            ++this.suspiciousFalls;
            if (this.suspiciousFalls >= 3) {
                ++this.violations;
                this.lastViolation = System.currentTimeMillis();
            }
        }

        void recordAscent() {
            long now = System.currentTimeMillis();
            this.ascentCount = now - this.lastAscent < 1000L ? ++this.ascentCount : 1;
            this.lastAscent = now;
        }

        int getAscentCount() {
            if (System.currentTimeMillis() - this.lastAscent > 1000L) {
                this.ascentCount = 0;
            }
            return this.ascentCount;
        }

        void landed() {
            this.lastLanding = System.currentTimeMillis();
            this.totalFallTracked = 0.0;
            this.ascentCount = 0;
        }

        void decay() {
            if (System.currentTimeMillis() - this.lastViolation > 10000L) {
                this.violations = Math.max(0, this.violations - 1);
                this.suspiciousFalls = Math.max(0, this.suspiciousFalls - 1);
            }
        }

        void reset() {
            this.totalFallTracked = 0.0;
            this.ascentCount = 0;
        }

        int getViolationCount() {
            return this.violations;
        }

        long getTimeSinceLanded() {
            return System.currentTimeMillis() - this.lastLanding;
        }
    }
}

